package com.xdja.im.uikit.ui.adapter;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.support.annotation.NonNull;
import android.text.ClipboardManager;
import android.text.TextUtils;
import android.view.View;
import android.widget.ListView;
import android.widget.Toast;

import com.google.gson.Gson;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.base.frame.presenter.BaseItemAdapterPresenter;
import com.xdja.im.base.frame.view.AdapterBaseView;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.common.utils.SystemHelper;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.account.AccountBean;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.ImageFileInfo;
import com.xdja.im.core.model.message.file.NormalFileInfo;
import com.xdja.im.core.model.message.file.VideoFileInfo;
import com.xdja.im.core.model.message.file.VoiceFileInfo;
import com.xdja.im.core.proxy.IMUiKitProxyImpl;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.lib.audio.manager.TelphoneState;
import com.xdja.im.lib.filemanage.util.NetworkUtils;
import com.xdja.im.lib.filemanage.util.XToast;
import com.xdja.im.lib.share.model.WebPageInfo;
import com.xdja.im.lib.share.util.ShareUtils;
import com.xdja.im.uikit.IntentParam;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.ui.CustomItemDataProvider;
import com.xdja.im.uikit.ui.CustomMessageData;
import com.xdja.im.uikit.ui.ListItemUtils;
import com.xdja.im.uikit.ui.activity.FileDownloadActivity;
import com.xdja.im.uikit.ui.activity.FilePreviewActivity;
import com.xdja.im.uikit.ui.dialog.CustomDialog;
import com.xdja.im.uikit.ui.presenter.ChatListAdapterPresenter;
import com.xdja.im.uikit.ui.presenter.ChatListPresenter;
import com.xdja.im.uikit.utils.SpUtils;
import com.xdja.im.uikit.utils.audio.AudioPlayerManagerProxy;
import com.xdja.im.uikit.utils.file.FileUtils;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.im.uikit.utils.tbs.TBSBrowserActivity;

import java.util.ArrayList;
import java.util.List;

import rx.Subscriber;

/**
 * @Package: com.xdja.im.uikit.presenter.adapter
 * @Author: xdjaxa
 * @Date: 2017-04-10 18:00
 * @Version V1.0
 * @Comment: 聊天详情列表界面适配器
 */
public class ChatListAdapter extends BaseItemAdapterPresenter<ChatListAdapterPresenter, TalkMessageBean>
        implements ChatListAdapterPresenter {

    private ProxyRepository proxyRepository;

    private ChatListPresenter mPresenter;

    private ListView listView;

    /**
     * 消息列表
     */
    private final List<TalkMessageBean> mDataSource;

    private List<Class<? extends AdapterBaseView<ChatListAdapterPresenter, TalkMessageBean>>> vuClasses;

    public ChatListAdapter(ChatListPresenter presenter, List<TalkMessageBean> dataSource,
                           ListView listView, ProxyRepository proxyRepository) {
        this.mPresenter = presenter;
        this.mDataSource = dataSource;
        this.listView = listView;
        this.proxyRepository = proxyRepository;
    }

    @Override
    protected Activity getActivity() {
        return mPresenter.getActivity();
    }

    @Override
    protected ChatListAdapterPresenter getPresenter() {
        return this;
    }

    @Override
    protected TalkMessageBean getDataSource(int position) {
        return mDataSource == null ? null : mDataSource.get(position);
    }

    @Override
    public int getCount() {
        return mDataSource == null ? 0 : mDataSource.size();
    }

    @Override
    public Object getItem(int position) {
        return mDataSource == null ? null : mDataSource.get(position);
    }

    @Override
    public long getItemId(int position) {
        return mDataSource != null ? mDataSource.get(position).getId() : 0;
    }

    @NonNull
    @Override
    protected Class<? extends AdapterBaseView<ChatListAdapterPresenter, TalkMessageBean>> getVuClassByViewType(int itemViewType) {
        Class<? extends AdapterBaseView<ChatListAdapterPresenter, TalkMessageBean>> vuCls = null;
        return ListItemUtils.getInstance().getChatListItemByTypeKey(itemViewType);
    }

    @Override
    public int getViewTypeCount() {
        return ListItemUtils.getInstance().getChatListItemTypeSize();
    }

    @Override
    public int getItemViewType(int position) {

        TalkMessageBean talkMessageBean = mDataSource.get(position);
        if (talkMessageBean == null) {
            LogUtil.e("Item viewType data is null.");
            return Constant.ITEM_TYPE_DEFAULT;
        }

        int chatDetailType;
        switch (talkMessageBean.getMessageType()) {
            case ConstDef.MSG_TYPE_AT_TEXT:    //@
                if (talkMessageBean.isMine()) {
                    chatDetailType = Constant.ITEM_TYPE_SEND_TEXT;
                } else {
                    chatDetailType = Constant.ITEM_TYPE_RECEIVE_TEXT;
                }
                break;
            case ConstDef.MSG_TYPE_TEXT:    //文本
                if (talkMessageBean.getContent().startsWith(ConstDef.IM_CUSTOM_MSG_TIP)) {
                    String content = talkMessageBean.getContent().substring(ConstDef.IM_CUSTOM_MSG_TIP.length());
                    CustomMessageData customMessageData = new Gson().fromJson(content, CustomMessageData.class);
                    if (customMessageData != null && customMessageData.getType() != null) {
                        CustomMessageData.TypeBean typeBean = customMessageData.getType();
                        if (talkMessageBean.isMine()) {
                            chatDetailType = ListItemUtils.getInstance().getRealCustomType(typeBean.getSendType());
                        } else {
                            chatDetailType = ListItemUtils.getInstance().getRealCustomType(typeBean.getRecvType());
                        }
                    } else {
                        if (talkMessageBean.isMine()) {
                            chatDetailType = Constant.ITEM_TYPE_SEND_TEXT;
                        } else {
                            chatDetailType = Constant.ITEM_TYPE_RECEIVE_TEXT;
                        }
                    }
                } else {
                    if (talkMessageBean.isMine()) {
                        chatDetailType = Constant.ITEM_TYPE_SEND_TEXT;
                    } else {
                        chatDetailType = Constant.ITEM_TYPE_RECEIVE_TEXT;
                    }
                }
                break;

            case ConstDef.MSG_TYPE_VOICE:   //音频
                if (talkMessageBean.isMine()) {
                    chatDetailType = Constant.ITEM_TYPE_SEND_VOICE;
                } else {
                    chatDetailType = Constant.ITEM_TYPE_RECEIVE_VOICE;
                }
                break;

            case ConstDef.MSG_TYPE_PHOTO:   //图片
                if (talkMessageBean.isMine()) {
                    chatDetailType = Constant.ITEM_TYPE_SEND_IMAGE;
                } else {
                    chatDetailType = Constant.ITEM_TYPE_RECEIVE_IMAGE;
                }
                break;

            case ConstDef.MSG_TYPE_VIDEO:   //视频
                if (talkMessageBean.isMine()) {
                    chatDetailType = Constant.ITEM_TYPE_SEND_VIDEO;
                } else {
                    chatDetailType = Constant.ITEM_TYPE_RECEIVE_VIDEO;
                }
                break;

            case ConstDef.MSG_TYPE_FILE:    //文件
                if (talkMessageBean.isMine()) {
                    chatDetailType = Constant.ITEM_TYPE_SEND_FILE;
                } else {
                    chatDetailType = Constant.ITEM_TYPE_RECEIVE_FILE;
                }
                break;

            case ConstDef.MSG_TYPE_WEB:    //图文
                if (talkMessageBean.isMine()) {
                    chatDetailType = Constant.ITEM_TYPE_SEND_WEB;
                } else {
                    chatDetailType = Constant.ITEM_TYPE_RECEIVE_WEB;
                }
                break;

            case ConstDef.MSG_TYPE_LOCATION://位置
                if (talkMessageBean.isMine()) {
                    chatDetailType = Constant.ITEM_TYPE_SEND_LOCATION;
                } else {
                    chatDetailType = Constant.ITEM_TYPE_RECEIVE_LOCATION;
                }
                break;

            case ConstDef.MSG_TYPE_PRESENTATION: //通知
                chatDetailType = Constant.ITEM_TYPE_NOTIFY;
                break;

            case ConstDef.MSG_TYPE_CUSTOM:   //自定义
                if (talkMessageBean.isMine()) {
                    chatDetailType = Constant.ITEM_TYPE_SEND_CUSTOM;
                } else {
                    chatDetailType = Constant.ITEM_TYPE_REC_CUSTOM;
                }
                break;
            case ConstDef.MSG_TYPE_ASSISTANT:   //自定义View
                CustomItemDataProvider customItemDataProvider = new Gson().fromJson(talkMessageBean.getContent(), CustomItemDataProvider.class);
                if (customItemDataProvider != null)
                    chatDetailType = ListItemUtils.getInstance().getRealCustomType(customItemDataProvider.getType());
                else chatDetailType = Constant.ITEM_TYPE_DEFAULT;
                break;

            default://默认
                chatDetailType = Constant.ITEM_TYPE_DEFAULT;
                break;
        }
        return chatDetailType;
    }

    @Override
    public void reSendMessage(final int position) {
        final TalkMessageBean messageBean = (TalkMessageBean) getItem(position);
        if (messageBean == null) {
            LogUtil.e("ERROR: reSend message, but bean is null." + position);
            return;
        }
        proxyRepository.reSendMessage(messageBean, new Subscriber<Integer>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("Resend message error, " + e.getMessage());
            }

            @Override
            public void onNext(Integer messageId) {
                LogUtil.d("Resend message successfully, " + messageBean.toString());
                //此处更新状态为发送中
                messageBean.setMessageState(ConstDef.STATE_SENDING);
                mDataSource.set(position, messageBean);
                updateItem(position, messageBean);
            }
        });
    }

    @Override
    public void deleteItem(int position) {
        final TalkMessageBean messageBean = (TalkMessageBean) getItem(position);
        if (messageBean != null) {
            final long id = messageBean.getId();
            List<Long> ids = new ArrayList<>();
            ids.add(id);

            LogUtil.d("Start delete message " + id);
            proxyRepository.deleteMessages(ids, new Subscriber<Integer>() {
                @Override
                public void onCompleted() {

                }

                @Override
                public void onError(Throwable e) {
                    LogUtil.e("Delete message error, " + e.getMessage());
                    CommonTool.showToast(R.string.im_uikit_delete_msg_failed);
                }

                @Override
                public void onNext(Integer integer) {
                    LogUtil.d("Delete message id " + id + ", result:" + integer);
                    postDeleteMessageEvent(messageBean);
                }
            });
        }
    }

    @Override
    public void recallMessage(int position) {
        final TalkMessageBean messageBean = (TalkMessageBean) getItem(position);
        if (messageBean == null) {
            LogUtil.e("ERROR: recall message, but bean is null." + position);
            return;
        }

        if (NetworkUtils.checkNetworkState(getActivity())) {
            if (!SpUtils.getInstance().getBoolean(ConstDef.RECALL_MSG)) {
                showDialog(R.string.im_uikit_first_use_recall, true, messageBean);
            } else {
                recallMsg(messageBean);
            }
        } else {
            new XToast(getActivity()).display(R.string.im_uikit_network_no_server);
        }
    }

    private void recallMsg(TalkMessageBean talkMessageBean) {
        proxyRepository.recallMessages(talkMessageBean.getId(), ConstDef.RECALL, new Subscriber<Integer>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                LogUtil.e("ERROR: recall message failed." + e.getMessage());
            }

            @Override
            public void onNext(Integer integer) {
                LogUtil.d("message is already recalled.");
                //CommonTool.showToast("message is already recalled.");
                if (integer != null) {
                    if (integer == ConstDef.OVER_TIME_RECALL_FAIL) {
                        showDialog(R.string.im_uikit_not_recall_two_minutes, false, null);
                    }
                }

            }
        });
    }

    @Override
    public void copyMessage(int position) {
        final TalkMessageBean messageBean = (TalkMessageBean) getItem(position);
        if (messageBean == null) {
            LogUtil.e("ERROR: copy message, but bean is null." + position);
            return;
        }

        //文本消息
        if (messageBean.getMessageType() == ConstDef.MSG_TYPE_TEXT || messageBean.getMessageType() == ConstDef.MSG_TYPE_AT_TEXT) {
            ClipboardManager cmb = (ClipboardManager) getActivity().getSystemService(Context.CLIPBOARD_SERVICE);
            if (!TextUtils.isEmpty(messageBean.getContent())) {
                cmb.setText(messageBean.getContent());
            }
        }
    }

    /**
     * 消息转发事件
     *
     * @param position
     */
    @Override
    public void forwardMessage(int position) {

        Intent intent = ShareUtils.buildForwardIntent(getActivity(), getTalkMessage(position));
        if (intent == null) {
            Toast.makeText(getActivity(), "请先打开或下载文件", Toast.LENGTH_SHORT).show();
        } else
            getActivity().startActivity(intent);
    }

    @Override
    public void updateItem(int position, TalkMessageBean talkMessageBean) {
        try {

            int firstVisPosition = listView.getFirstVisiblePosition();
            int lastVisPosition = listView.getLastVisiblePosition();

            if (position < firstVisPosition - 1 || position > lastVisPosition) {
                //如果是动图就需要进行更新，是否暂停播放
                if (getDataSource(position).getMessageType() == ConstDef.MSG_TYPE_PHOTO) {
                    ImageFileInfo imageFileInfo = (ImageFileInfo) getDataSource(position).getFileInfo();
                    if (imageFileInfo != null
                            && !ConstDef.GIF_TYPE.equalsIgnoreCase(imageFileInfo.getSuffix())) {
                        return;
                    }
                } else {
                    return;
                }
            }

            View view = listView.getChildAt(position - firstVisPosition);
            if (view != null) {
                AdapterBaseView<ChatListAdapterPresenter, TalkMessageBean> tag =
                        (AdapterBaseView<ChatListAdapterPresenter, TalkMessageBean>) view.getTag();
                tag.bindDataSource(position, getDataSource(position));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * ListView下滑至指定位置
     *
     * @param position
     */
    @Override
    public void scrollTo(int position) {
        if (listView != null) {
            if (position == listView.getLastVisiblePosition()) {
                listView.setSelection(position);
            }
        }
    }

    @Override
    public void sendReadReceipt(final TalkMessageBean talkMessageBean) {
        //消息需要处理状态
        proxyRepository.changeMessageState(talkMessageBean, ConstDef.STATE_READED, new Subscriber<Integer>() {
            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.d("ERROR: Change message state error." + e.getMessage());
            }

            @Override
            public void onNext(Integer integer) {
                if (integer != 0) {
                    LogUtil.e("Change message state failed.");
                    return;
                }
                LogUtil.d("Change message state success.");
//                int length = mDataSource.size();
//                for (int i = length - 1; i >= 0; i--) {
//                    if (mDataSource.get(i).getId() == talkMessageBean.getId()) {
//                        talkMessageBean.setMessageState(ConstDef.STATE_READED);
//                        mDataSource.set(i, talkMessageBean);
//                        updateItem(i, talkMessageBean);
//                        break;
//                    }
//                }
            }
        });

    }

    @Override
    public void postDestroyAnimate(TalkMessageBean bean) {
        IMProxyMessageEvent.RefreshMessageEvent event = new IMProxyMessageEvent.RefreshMessageEvent();
        event.setMsgAccount(bean.getFrom());
        bean.setMessageState(ConstDef.STATE_DESTROY);
        event.setTalkMessageBean(bean);
        BusProvider.getInstance().post(event);
    }

    @Override
    public TalkMessageBean getTalkMessage(int position) {
        if (position < 0 || position >= getCount()) {
            return null;
        }
        return (TalkMessageBean) getItem(position);
    }

    @Override
    public void getTalkMessageEntity(long msgId, Subscriber<TalkMessageBean> subscriber) {
        proxyRepository.getTalkMessage(msgId, subscriber);
    }

    @Override
    public void getAccount(String account, int talkType, Subscriber<AccountBean> subscriber) {
        IMUiKitProxyImpl.getInstance().getAccount(account, talkType, subscriber);
    }

    @Override
    public void clickWebMessage(TalkMessageBean talkMessageBean) {
        if (talkMessageBean == null) {
            return;
        }
        WebPageInfo pInfo = (WebPageInfo) talkMessageBean.getWebPageInfo();
        if (pInfo == null) {
            new XToast(getActivity()).display(getActivity().getString(R.string.im_uikit_file_open_failed));
            return;
        }
        if (pInfo.getWebUri() != null) {
            Intent intent = new Intent();
            intent.setAction(getActivity().getPackageName() + "." + TBSBrowserActivity.ACTION_VIEW);
            Uri uri = Uri.parse(pInfo.getWebUri());
            intent.setData(uri);
            getActivity().startActivity(intent);
        }
    }

    @Override
    public void downLoadFile(final TalkMessageBean talkMessageBean, final FileInfo fileInfo) {

        if (fileInfo == null) {
            LogUtil.e("start load file, but file info is null error.");
            return;
        }

        List<FileInfo> fileInfos = new ArrayList<>();
        fileInfos.add(fileInfo);

        proxyRepository.downloadFile(fileInfos, new Subscriber<Integer>() {
            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
                LogUtil.d("Download image error." + e.getMessage());
            }

            @Override
            public void onNext(Integer integer) {
                LogUtil.d("Start to loadNormalFile file.");
                if (fileInfo.getFileType() == ConstDef.FILE_TYPE_VOICE) {
                    boolean isCurrentPlaying = AudioPlayerManagerProxy.isPlayingWithMessage(fileInfo);
                    if (AudioPlayerManagerProxy.isPlaying()) {
                        AudioPlayerManagerProxy.stopPlay();
                    }

                    if (isCurrentPlaying) {
                        return;
                    }

                    //语音播放
                    AudioPlayerManagerProxy.startToPlay(fileInfo, talkMessageBean.getId(),
                            false);
                }

            }
        });
    }

    @Override
    public void playVoice(TalkMessageBean bean) {
        LogUtil.d("start play voice file.");

        //如果正在通话，则不进行播放
        if (!TelphoneState.getPhotoStateIsIdle(getActivity().getApplicationContext())) {
            CommonTool.showToast(R.string.im_uikit_phone_is_inCall);
            return;
        }

        VoiceFileInfo fileInfo = (VoiceFileInfo) bean.getFileInfo();
        if (fileInfo == null || TextUtils.isEmpty(fileInfo.getFilePath())) {
            LogUtil.e("ERROR: start to play voice, but file is null  or file not exist.");
            return;
        }

        if (bean.isMine() || (!bean.isMine() && (fileInfo.getFileState() == ConstDef.DONE))) {
            if (!FileUtils.isFileExists(fileInfo.getFilePath())) {
                CommonTool.showToast(R.string.im_uikit_file_not_exist);
                return;
            }
        } else {
            //下载语音文件
            downLoadFile(bean, fileInfo);
            return;
        }

        boolean isCurrentPlaying = AudioPlayerManagerProxy.isPlayingWithMessage(fileInfo);
        if (AudioPlayerManagerProxy.isPlaying()) {
            AudioPlayerManagerProxy.stopPlay();
        }

        if (isCurrentPlaying) {
            return;
        }

        //语音播放
        AudioPlayerManagerProxy.startToPlay(fileInfo, bean.getId(),
                bean.getMessageState() < ConstDef.STATE_READED);

        //更新信息状态
        if (!bean.isMine() && bean.getMessageState() < ConstDef.STATE_READED) {
            sendReadReceipt(bean);
        }
    }

    @Override
    public void previewVideo(TalkMessageBean dataSource) {
        LogUtil.d("start play video file.");

        VideoFileInfo fileInfo = (VideoFileInfo) dataSource.getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: video file info is null error.");
            return;
        }

        //视频缩略图文件不存在，则不进行下载
        if (!FileUtils.isFileExists(fileInfo.getFilePath())) {
            LogUtil.e("ERROR: file not exit.");
            return;
        }

        startPreviewActivity(fileInfo.getTalkMessageId());

        if (!dataSource.isMine() && dataSource.getMessageState() < ConstDef.STATE_READED) {
            sendReadReceipt(dataSource);
        }
    }

    @Override
    public void previewImage(TalkMessageBean dataSource) {
        LogUtil.d("Start preview image.");

        ImageFileInfo imageFile = (ImageFileInfo) dataSource.getFileInfo();

        if (imageFile == null) {
            LogUtil.e("ERROR: image file info is null error.");
            return;
        }

        //视频缩略图文件不存在，则不进行下载 进行文件准确判断
        if (!FileUtils.isFileExists(imageFile.getFilePath())) {
            LogUtil.e("ERROR: file not exit.");
            return;
        }

        startPreviewActivity(imageFile.getTalkMessageId());
    }

    @Override
    public void loadNormalFile(TalkMessageBean dataSource) {
        LogUtil.d("start open file.");
        if (dataSource == null) {
            LogUtil.e("ERROR: start open file, but data source is null.");
            return;
        }

        NormalFileInfo normalFileInfo = (NormalFileInfo) dataSource.getFileInfo();

        Intent intent = new Intent(getActivity(), FileDownloadActivity.class);
        intent.putExtra(Constant.TAG_MSGID, dataSource.getId());
        intent.putExtra(Constant.TAG_FILE_TRANS_SIZE, normalFileInfo.getTranslateSize());
        mPresenter.startActivityForResult(intent, Constant.RESULT_CODE_FILE_DOWNLOAD);
        if (!dataSource.isMine())
            sendReadReceipt(dataSource);
    }

    @Override
    public void openNormalFile(TalkMessageBean dataSource) {
        NormalFileInfo fileInfo = (NormalFileInfo) dataSource.getFileInfo();
        if (fileInfo == null) {
            LogUtil.e("ERROR: open file failed, file info is null.");
            return;
        }
        //判断文件完整存在
        if (FileUtils.getFileExistAndComplete(fileInfo.getFilePath(), fileInfo.getFileState())) {
            SystemHelper.openFile(getActivity(), fileInfo.getFilePath());
        } else {
            CommonTool.showToast(R.string.im_uikit_file_not_exist);
        }
    }

    @Override
    public void openSetting(String talkId, int talkType) {
        try {
            boolean result = IMUiKitProxyImpl.getInstance().openChatSettingCallback(talkId, talkType);
            if (!result) {
                IntentParam.startSettingActivity(getActivity(), talkId, talkType);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void openUserInfo(String account) {
        try {
            boolean result = IMUiKitProxyImpl.getInstance().openChatUserInfoCallback(account);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Override
    public void onLongClickAtUser(final String groupId, final String account) {
        IMUiKitProxyImpl.getInstance().getAccount(account, ConstDef.CHAT_TYPE_P2P, new Subscriber<AccountBean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onNext(AccountBean accountBean) {
                mPresenter.onLongClickAtUser(groupId, account, accountBean.getDisplayName());
            }
        });
    }

    /**
     * 启动文件预览界面
     *
     * @param msgId 消息ID
     */
    private void startPreviewActivity(long msgId) {
        Intent intent = new Intent();
        intent.setClass(mPresenter.getActivity(), FilePreviewActivity.class);
        intent.putExtra(Constant.TAG_TALKERID, mPresenter.getTalkFlag());
        intent.putExtra(Constant.TAG_MSGID, msgId);
        mPresenter.startActivity(intent);
    }

    /**
     * 发送删除消息事件
     *
     * @param talkMessageBean 消息实体
     * @return
     */
    private int postDeleteMessageEvent(TalkMessageBean talkMessageBean) {
        try {
            //构建事件对象
            IMProxyMessageEvent.DeleteMessageEvent messageEvent = new IMProxyMessageEvent.DeleteMessageEvent();
            messageEvent.setMsgAccount(talkMessageBean.getFrom());
            messageEvent.setTalkMessageBean(talkMessageBean);

            //发送事件
            BusProvider.getInstance().post(messageEvent);
            //返回正确处理结果
            return ConstDef.CALLBACK_HANDLED;
        } catch (Exception ex) {
            //处理异常信息
            return ConstDef.CALLBACK_NOT_HANDLED;
        }
    }

    /**
     * 语音文件播放开始，页面需要做的更新操作
     *
     * @param filePath  播放的文件的路径
     * @param messageId 播放文件对应的messageId
     */
    public void startAnimation(String filePath, long messageId) {
        int position = getPositionWithId(messageId);
        if (position >= 0) {
            updateItem(position, getDataSource(position));
        }
    }

    /**
     * 语音文件播放完毕或者播放停止对应的页面的跟新操作
     *
     * @param filePath  播放的文件的路径
     * @param messageId 播放文件对应的messageId
     */
    public void stopAnimation(String filePath, long messageId) {
        int position = getPositionWithId(messageId);
        if (position >= 0) {
            updateItem(position, getDataSource(position));
        }
    }

    /**
     * 播放下一个语音文件,如果存在的话
     *
     * @param filePath  播放的文件的路径
     * @param messageId 播放文件对应的messageId
     * @param isNext    是否需要继续播放下一条语音
     */
    public void playNextAudio(String filePath, long messageId, boolean isNext) {
        LogUtil.d("start play next voice file.msgId:" + messageId);

        int position = getPositionWithId(messageId);
        if (position >= 0) {

            //获取当前播放的文件信息
            TalkMessageBean dataSource = getDataSource(position);
            //先停止页面动画
            updateItem(position, dataSource);

            if (dataSource.isMine() || !isNext) {
                return;
            }

            //联播
            TalkMessageBean bean = findNextVoiceItem(position, messageId);
            if (bean != null && !bean.isMine()) {
                playVoice(bean);
            }
        }
    }

    /**
     * 获取下一个可以播放的语音文件
     *
     * @param messageId 当前播放的文件对应的messageID
     * @return 返回找到的数据实例
     */
    private TalkMessageBean findNextVoiceItem(int curPosition, long messageId) {
        TalkMessageBean bean = new TalkMessageBean();
        bean.setId(messageId);

        int size = mDataSource.size();
        if (curPosition >= 0) {
            for (int i = curPosition + 1; i < size; i++) {
                TalkMessageBean talkMessageBean = mDataSource.get(i);
                if (talkMessageBean.isMine()) {
                    continue;
                }

                //查找下一条接收的语音文件消息
                FileInfo fileInfo = talkMessageBean.getFileInfo();
                if (fileInfo == null || fileInfo.getFileType() != ConstDef.FILE_TYPE_VOICE) {
                    continue;
                }

                if (talkMessageBean.getMessageState() < ConstDef.STATE_READED &&
                        fileInfo.getFileState() == ConstDef.DONE) {
                    return talkMessageBean;
                }
                break;
            }
        }
        return null;
    }

    /**
     * 根据messageId找到在显示列表对应的位置
     *
     * @param messageId 需要查找的messageId
     * @return 查询到则返回对应值，-1表示未查询到
     */
    private int getPositionWithId(long messageId) {
        TalkMessageBean bean = new TalkMessageBean();
        bean.setId(messageId);
        return mDataSource.indexOf(bean);
    }

    @Override
    public boolean getVoiceMessageIsPlaying(FileInfo fileInfo) {
        return AudioPlayerManagerProxy.isPlayingWithMessage(fileInfo);
    }

    private void showDialog(int resoureId, final boolean isCanRecall, final TalkMessageBean messageBean) {
        new CustomDialog(getActivity(), new CustomDialog.OnCloseListener() {
            @Override
            public void onClick(Dialog dialog, boolean confirm) {
                if (isCanRecall && messageBean != null) {
                    if (confirm) {
                        SpUtils.getInstance().saveBoolean(ConstDef.RECALL_MSG, true);
                        recallMsg(messageBean);
                    }
                }
            }
        })
                .setIfShowNegative(false)
                .setDialogTitle(R.string.im_uikit_hint)
                .setDialogContent(resoureId)
                .setPositiveButton(R.string.im_uikit_ok_text)
                .show();
    }
}
