package com.xdja.im.common.imagecompresser;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import androidx.annotation.Nullable;
import androidx.annotation.UiThread;
import androidx.annotation.WorkerThread;
import android.util.Log;

import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.io.IOException;

/**
 * @Package: com.xdja.im.common.imagecompresser
 * @Author: xdjaxa
 * @Creation: 2017-07-05 10:58
 * @Version V1.0
 * @Description:
 */
public class ImageCompressor implements Handler.Callback {

    private static final String TAG = "ImageCompressor";

    private static final String DEFAULT_DISK_CACHE_DIR = "image_disk_cache";

    private static final int MSG_COMPRESS_SUCCESS = 0;
    private static final int MSG_COMPRESS_START = 1;
    private static final int MSG_COMPRESS_ERROR = 2;

    private File file;
    private File cacheDir;
    private OnCompressListener onCompressListener;

    private Handler mHandler;

    public ImageCompressor(Builder builder) {
        this.file = builder.file;
        this.cacheDir = builder.cacheDir;
        this.onCompressListener = builder.onCompressListener;
        mHandler = new Handler(Looper.getMainLooper(), this);
    }

    public static Builder with(Context context) {
        return new Builder(context);
    }

    /**
     * Returns a file with a cache audio name in the private cache directory.
     *
     * @param context
     *     A context.
     */
    private File getImageCacheFile(Context context) {
        if (getImageCacheDir(context) != null) {
            return new File(getImageCacheDir(context) + "/" + System.currentTimeMillis() + (int) (Math.random() * 1000) + ".jpg");
        }
        return null;
    }

    /**
     * Returns a directory with a default name in the private cache directory of the application to
     * use to store retrieved audio.
     *
     * @param context
     *     A context.
     *
     * @see #getImageCacheDir(Context, String)
     */
    @Nullable
    private File getImageCacheDir(Context context) {
        return getImageCacheDir(context, DEFAULT_DISK_CACHE_DIR);
    }

    /**
     * Returns a directory with the given name in the private cache directory of the application to
     * use to store retrieved media and thumbnails.
     *
     * @param context
     *     A context.
     * @param cacheName
     *     The name of the subdirectory in which to store the cache.
     *
     * @see #getImageCacheDir(Context)
     */
    @Nullable
    private File getImageCacheDir(Context context, String cacheName) {
        if (cacheDir == null) {
            cacheDir = context.getExternalCacheDir();
        }
        if (cacheDir != null) {
            File result = new File(cacheDir, cacheName);
            if (!result.mkdirs() && (!result.exists() || !result.isDirectory())) {
                // File wasn't able to create a directory, or the result exists but not a directory
                return null;
            }
            return result;
        }
        if (Log.isLoggable(TAG, Log.ERROR)) {
            LogUtil.e(TAG, "default disk cache dir is null");
        }
        return null;
    }

    /**
     * start compress and return the file
     */
    @WorkerThread
    private File get(final Context context) throws IOException {
        return new Engine(file, getImageCacheFile(context)).compress();
    }

    /**
     * start asynchronous compress thread
     */
    @UiThread
    private void start(final Context context) {
        if (file == null && onCompressListener != null) {
            onCompressListener.onError(new NullPointerException("image file cannot be null"));
        }

        new Thread(new Runnable() {
            @Override public void run() {
                try {
                    mHandler.sendMessage(mHandler.obtainMessage(MSG_COMPRESS_START));

                    File result = new Engine(file, getImageCacheFile(context)).compress();
                    mHandler.sendMessage(mHandler.obtainMessage(MSG_COMPRESS_SUCCESS, result));
                } catch (IOException e) {
                    mHandler.sendMessage(mHandler.obtainMessage(MSG_COMPRESS_ERROR, e));
                }
            }
        }).start();
    }

    @Override
    public boolean handleMessage(Message msg) {
        if (onCompressListener == null) return false;

        switch (msg.what) {
            case MSG_COMPRESS_START:
                onCompressListener.onStart();
                break;
            case MSG_COMPRESS_SUCCESS:
                onCompressListener.onSuccess((File) msg.obj);
                break;
            case MSG_COMPRESS_ERROR:
                onCompressListener.onError((Throwable) msg.obj);
                break;
        }
        return false;
    }

    public static class Builder{

        private Context context;

        private File file;

        private File cacheDir;

        private OnCompressListener onCompressListener;

        public Builder(Context context) {
            this.context = context;
        }

        public Builder load(File file) {
            this.file = file;
            return this;
        }

        public File get() throws IOException {
            return build().get(context);
        }

        public Builder setCompressListener(OnCompressListener listener) {
            this.onCompressListener = listener;
            return this;
        }

        public Builder setCacheDir(File dir) {
            this.cacheDir = dir;
            return this;
        }

        public ImageCompressor build() {
            return new ImageCompressor(this);
        }

        public void start() {
            build().start(context);
        }
    }
}
