package com.xdja.im.common.network;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RawRes;

import com.xdja.im.base.di.DiConfig;
import com.xdja.im.base.di.scope.Scoped;
import com.xdja.im.common.cache.interf.UserCache;
import com.xdja.im.common.utils.CachePath;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManagerFactory;

import okhttp3.Cache;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.logging.HttpLoggingInterceptor;

/**
 * @Package: com.xdja.im.common.network
 * @Author: xdjaxa
 * @Creation: 2017-05-31 17:02
 * @Version V1.0
 * @Description:
 */
public class OkHttpsClientMe extends OkHttpClient {

    /**
     * 缓存文件父目录
     */
    private static final String DISK_CACHE_NAME = "http-cache";
    /**
     * 缓存文件大小
     */
    private static final long DISK_CACHE_SIZE = 50 * 1024 * 1024;
    /**
     * 网络连接超时时间
     */
    public static final int TIME_OUT_UNIT = 30 * 1000;
    /**
     * 默认的PIN码
     */
//    private static final String DEFAULT_PASSWORD = "111111";
    private static final String DEFAULT_PASSWORD = "xdja1234";

    public static final String HEAD_TICKET = "ticket";

    private Context context;

    private UserCache userCache;

    private String password;

    /**
     * 是否验证服务器主机地址
     */
    private boolean isVerifyHostName = false;

    @Inject
    public OkHttpsClientMe(@NonNull @Scoped(DiConfig.CONTEXT_SCOPE_APP) Context context,
                           UserCache userCache) {
        this(context, userCache, DEFAULT_PASSWORD);
    }

    public OkHttpsClientMe(Context context, UserCache userCache, String pwd) {
        this.context = context;
        this.userCache = userCache;
        this.password = pwd;
    }

    public OkHttpClient.Builder createBuilder() {

        String dirBuilder = CachePath.getParentCachePath() +
                File.separator +
                DISK_CACHE_NAME;

        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        builder.connectTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .readTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .writeTimeout(TIME_OUT_UNIT, TimeUnit.MILLISECONDS)
                .cache(new Cache(new File(dirBuilder), DISK_CACHE_SIZE))
                .sslSocketFactory(makeSSLSocketFactory())
                .hostnameVerifier(new HostnameVerifier() {
                    @Override
                    public boolean verify(String hostname, SSLSession session) {
                        return !isVerifyHostName;
                    }
                })
                .addInterceptor(
                        new Interceptor() {
                            @Override
                            public Response intercept(Chain chain) throws IOException {

                                Request request = chain.request();
                                Request.Builder builder = request.newBuilder();
                                builder.header(HEAD_TICKET, userCache.get().getTicket());
                                return chain.proceed(builder.build());
                            }
                        }
                )
                .addInterceptor(new HttpLoggingInterceptor(new HttpLoggingInterceptor.Logger() {
                    @Override
                    public void log(String message) {
                        LogUtil.d(message);
                    }
                }).setLevel(HttpLoggingInterceptor.Level.BODY));
        return builder;
    }

    /**
     * 获取SSLSocketFactory
     *
     * @return
     */
    private SSLSocketFactory makeSSLSocketFactory() {
        SSLContext sslContext = getSSLContext(
                readKeyStore(R.raw.trust, context));
        if (sslContext == null) {
            return null;
        }
        return sslContext.getSocketFactory();
    }

    /**
     * 读取本地证书
     *
     * @param res 证书资源ID
     * @return 获取到的证书
     */
    @Nullable
    private KeyStore readKeyStore(@RawRes int res, @NonNull Context context) {
        InputStream inputStream = null;
        KeyStore keyStore = null;
        try {
            keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            inputStream = context.getResources().openRawResource(res);
            keyStore.load(inputStream, password.toCharArray());
        } catch (KeyStoreException
                | CertificateException
                | NoSuchAlgorithmException
                | IOException e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return keyStore;
    }

    /**
     * 获取SSL连接上下文
     *
     * @param keyStore
     * @return
     */
    @Nullable
    private SSLContext getSSLContext(@Nullable KeyStore keyStore) {
        if (keyStore == null) {
            return null;
        }
        SSLContext sslContext = null;
        try {
            sslContext = SSLContext.getInstance("TLS");

            TrustManagerFactory trustManagerFactory =
                    TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(keyStore);

            KeyManagerFactory keyManagerFactory =
                    KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(keyStore, password.toCharArray());

            sslContext.init(keyManagerFactory.getKeyManagers(), trustManagerFactory.getTrustManagers(), new SecureRandom());
        } catch (NoSuchAlgorithmException
                | KeyStoreException
                | KeyManagementException
                | UnrecoverableKeyException e) {
            e.printStackTrace();
        }
        return sslContext;
    }
}
