package com.xdja.im.common.persistent;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;

import com.xdja.im.base.di.DiConfig;
import com.xdja.im.base.di.scope.Scoped;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.inject.Inject;

/**
 * @Package: com.xdja.im.common.persistent
 * @Author: xdjaxa
 * @Creation: 2017-04-11 10:55
 * @Version V1.0
 * @Description:配置文件操作工具类
 */
public class PropertyUtil {

    /**
     * 上下文句柄
     */
    private Context context;
    /**
     * 配置文件对象
     */
    private Properties properties;

    @Inject
    public PropertyUtil(@NonNull
                        @Scoped(value = DiConfig.CONTEXT_SCOPE_APP)
                                Context context) {
        this.context = context;
    }

    /**
     * 从某个文件中加载配置
     *
     * @param fileName 文件名称
     */
    public void load(@NonNull String fileName) {

        if (TextUtils.isEmpty(fileName)) {
            LogUtil.e("ERROR: Load file name is null.");
            return;
        }
        try {
            InputStream is = this.context.getAssets().open(fileName);
            properties = new Properties();
            properties.load(is);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 获取一个配置的值
     *
     * @param name key
     * @return value
     */
    public String get(@NonNull String name) {
        if (properties != null && !TextUtils.isEmpty(name)) {
            return properties.getProperty(name);
        }
        return null;
    }

    /**
     * 获取一个配置的值
     *
     * @param name         key
     * @param defaultValue 默认值
     * @return value
     */
    public String get(@NonNull String name, @Nullable String defaultValue) {
        if (properties != null && !TextUtils.isEmpty(name)) {
            return properties.getProperty(name, defaultValue);
        }
        return null;
    }

    /**
     * 获取所有的配置信息
     *
     * @return
     */
    public Map<String, String> getAll() {
        if (properties != null) {
            Set<String> keys = properties.stringPropertyNames();
            if (keys != null && !keys.isEmpty()) {
                Map<String, String> kv = new HashMap<>();
                for (String key : keys) {
                    String value = properties.getProperty(key);
                    kv.put(key, value);
                }
                return kv;
            }
        }
        return null;
    }
}
