package com.xdja.im.core.model.message.mapper;

import androidx.annotation.NonNull;

import com.xdja.im.common.cache.interf.UserCache;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.message.AtInfo;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.ImageFileInfo;
import com.xdja.im.core.model.message.file.NormalFileInfo;
import com.xdja.im.core.model.message.file.VideoFileInfo;
import com.xdja.im.core.model.message.file.VoiceFileInfo;
import com.xdja.im.lib.share.model.WebPageInfo;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.imsdk.constant.ImSdkFileConstant;
import com.xdja.imsdk.constant.MsgPackType;
import com.xdja.imsdk.model.IMFileInfo;
import com.xdja.imsdk.model.IMMessage;
import com.xdja.imsdk.model.IMSession;
import com.xdja.imsdk.model.body.IMATBody;
import com.xdja.imsdk.model.body.IMFileBody;
import com.xdja.imsdk.model.body.IMImageBody;
import com.xdja.imsdk.model.body.IMMediaBody;
import com.xdja.imsdk.model.body.IMMessageBody;
import com.xdja.imsdk.model.body.IMNormalFileBody;
import com.xdja.imsdk.model.body.IMPicBody;
import com.xdja.imsdk.model.body.IMRecallBody;
import com.xdja.imsdk.model.body.IMSystemBody;
import com.xdja.imsdk.model.body.IMTextBody;
import com.xdja.imsdk.model.body.IMVideoBody;
import com.xdja.imsdk.model.body.IMVoiceBody;
import com.xdja.imsdk.model.body.IMWebBody;
import com.xdja.imsdk.model.file.IMHDThumbFileInfo;
import com.xdja.imsdk.model.file.IMRawFileInfo;

import javax.inject.Inject;

/**
 * @Package: com.xdja.im.common.entity.mapper
 * @Author: xdjaxa
 * @Creation: 2017-04-11 13:56
 * @Version V1.0
 * @Description:数据转换器(只处理数据转化，不参与逻辑处理)
 */
public class DataMapper {

    private UserCache userCache;

    @Inject
    public DataMapper(UserCache userCache) {
        this.userCache = userCache;
    }

    /**
     * 将会话实体转化为会话业务对象
     * IMSession ==> TalkSessionBean
     *
     * @param session 会话实体
     * @return 会话相关业务对象
     */
    public TalkSessionBean mapTalkSessionBean(IMSession session) {
        TalkSessionBean talkSessionBean = new TalkSessionBean();

        talkSessionBean.setTalkerAccount(session.getImPartner());
        talkSessionBean.setLastTime(session.getDisplayTime());
        talkSessionBean.setNotReadCount(session.getRemindCount());
        talkSessionBean.setTalkFlag(session.getSessionTag());
        talkSessionBean.setTalkType(ValueConverter.sessionTypeConvert(session.getSessionType()));

        IMMessage message = session.getLastMessage();
        if (message != null) {
            TalkMessageBean msg = mapTalkMessageBean(message);

            talkSessionBean.setLastMsg(msg);
            talkSessionBean.setLastMsgAccount(msg.getFrom());
            talkSessionBean.setLastMsgType(msg.getMessageType());
            talkSessionBean.setContent(msg.getContent());
        }
        return talkSessionBean;
    }

    /**
     * 消息实体对象转换为消息业务对象
     * IMMessage ==> TalkMessageBean
     *
     * @param imMessage 消息实体对象
     * @return 消息业务对象
     */
    public TalkMessageBean mapTalkMessageBean(IMMessage imMessage) {
        TalkMessageBean talkMessageBean = new TalkMessageBean();
        talkMessageBean.setId(imMessage.getIMMessageId());
        talkMessageBean.setFrom(imMessage.getFrom());
        talkMessageBean.setSenderCardId(imMessage.getCardId());
        talkMessageBean.setTo(imMessage.getTo());
        talkMessageBean.setMessageType(ValueConverter.imMsgTypeConvert(imMessage));
        talkMessageBean.setBomb(imMessage.isBombIMMessage());
        talkMessageBean.setGroupMsg(imMessage.isGroupIMMessage());

        talkMessageBean.setMessageState(ValueConverter.imMsgStateConvert(imMessage.getState()));
        talkMessageBean.setLimitTime(imMessage.getTimeToLive());
        talkMessageBean.setMine(userCache.isMine(imMessage.getFrom()));
        talkMessageBean.setShowTime(imMessage.getIMMessageTime());
        talkMessageBean.setFailCode(imMessage.getFailCode());

        //撤回类型
        if (imMessage.isRecalled()) {
            IMRecallBody recallBody = (IMRecallBody) imMessage.getMessageBody();
            talkMessageBean.setMessageType(ConstDef.MSG_TYPE_PRESENTATION);
            talkMessageBean.setContent(recallBody.getNotice());
            return talkMessageBean;
        }
        //@类型
        if (imMessage.isATMessage()) {
            IMATBody atTextBody = (IMATBody) imMessage.getMessageBody();
            talkMessageBean.setContent(atTextBody.getContent());
            talkMessageBean.setAtAccountList(atTextBody.getAccounts());
            return talkMessageBean;
        }

        //Web类型
        if (imMessage.isWebIMMessage()) {
            talkMessageBean.setMessageType(ConstDef.MSG_TYPE_WEB);
            talkMessageBean.setWebPageInfo(mapWebInfo(imMessage));
            return talkMessageBean;
        }

        //文本类型
        if (imMessage.isTextIMMessage()) {
            if (imMessage.getType() == MsgPackType.CUSTOM_PG_TEXT || imMessage.getType() == MsgPackType.CUSTOM_PP_TEXT) {
                talkMessageBean.setMessageType(ConstDef.MSG_TYPE_CUSTOM);
            }
            IMTextBody textBody = (IMTextBody) imMessage.getMessageBody();
            talkMessageBean.setContent(textBody.getContent());
            return talkMessageBean;
        }

        //文件类型
        if (imMessage.isFileIMMessage()) {
            talkMessageBean.setFileInfo(mapFileInfo(imMessage));
        }

        //自定义文本消息
        if (imMessage.isCustom()) {
            IMTextBody textBody = (IMTextBody) imMessage.getMessageBody();
            talkMessageBean.setContent(textBody.getContent());
            talkMessageBean.setMessageType(ConstDef.MSG_TYPE_CUSTOM);
            return talkMessageBean;
        }

        if (imMessage.getMessageBody() instanceof IMSystemBody ){
            IMSystemBody imSystemBody = (IMSystemBody) imMessage.getMessageBody();
            talkMessageBean.setContent(imSystemBody.getSystemInfo());
            talkMessageBean.setMessageType(ConstDef.MSG_TYPE_ASSISTANT);
        }

        return talkMessageBean;
    }

    /**
     * IMMessage ==> WebPageInfo
     * @return WebPageInfo
     */
    @NonNull
    private WebPageInfo mapWebInfo(IMMessage imMessage) {
        if (imMessage == null || imMessage.getMessageBody() == null) {
            return null;
        }

        if (!imMessage.isWebIMMessage()) {
            return null;
        }

        IMWebBody imWebFileBody = (IMWebBody) imMessage.getMessageBody();
        WebPageInfo webPageInfo = new WebPageInfo();
        webPageInfo.setFileState(mapFileState(imWebFileBody.getState()));
        webPageInfo.setTranslateSize(imWebFileBody.getTranslateSize());
        webPageInfo.setFileName(imWebFileBody.getDisplayName());
        webPageInfo.setFileSize(imWebFileBody.getFileSize());
        webPageInfo.setFilePath(imWebFileBody.getLocalPath());
        webPageInfo.setSuffix(imWebFileBody.getSuffix());
        webPageInfo.setTalkMessageId(imMessage.getIMMessageId());
        webPageInfo.setTitle(imWebFileBody.getTitle());
        webPageInfo.setDescription(imWebFileBody.getDescription());
        webPageInfo.setWebUri(imWebFileBody.getUrl());
        webPageInfo.setSource(imWebFileBody.getSource());
        webPageInfo.setTranslatePercent(getPercent(imWebFileBody.getFileSize(), imWebFileBody.getTranslateSize()));
        return webPageInfo;
    }


    /**
     * 消息实体对象转换为消息业务对象
     * IMMessage ==> TalkMessageBean
     *
     * @param imMessage 消息实体对象
     * @return 消息业务对象
     */
    public TalkMessageBean mapMessage(IMMessage imMessage) {
        if (imMessage == null) {
            return null;
        }

        TalkMessageBean talkMessageBean = new TalkMessageBean();
        talkMessageBean.setId(imMessage.getIMMessageId());
        talkMessageBean.setFrom(imMessage.getFrom());
        talkMessageBean.setSenderCardId(imMessage.getCardId());
        talkMessageBean.setTo(imMessage.getTo());
        talkMessageBean.setMessageType(ValueConverter.imMsgTypeConvert(imMessage));
        talkMessageBean.setBomb(imMessage.isBombIMMessage());
        talkMessageBean.setGroupMsg(imMessage.isGroupIMMessage());

        talkMessageBean.setMessageState(ValueConverter.imMsgStateConvert(imMessage.getState()));
        talkMessageBean.setLimitTime(imMessage.getTimeToLive());
        talkMessageBean.setMine(userCache.isMine(imMessage.getFrom()));
        talkMessageBean.setShowTime(imMessage.getIMMessageTime());
        talkMessageBean.setFailCode(imMessage.getFailCode());

        if (imMessage.isRecalled()) {
            IMRecallBody recallBody = (IMRecallBody) imMessage.getMessageBody();
            talkMessageBean.setMessageType(ConstDef.MSG_TYPE_PRESENTATION);
            talkMessageBean.setContent(recallBody.getNotice());
            return talkMessageBean;
        }

        if (imMessage.isATMessage()) {
            IMATBody atTextBody = (IMATBody) imMessage.getMessageBody();
            talkMessageBean.setContent(atTextBody.getContent());
            talkMessageBean.setAtAccountList(atTextBody.getAccounts());
            return talkMessageBean;
        }

        if (imMessage.isTextIMMessage()) {
            IMTextBody textBody = (IMTextBody) imMessage.getMessageBody();
            talkMessageBean.setContent(textBody.getContent());
            return talkMessageBean;
        }

        if (imMessage.isFileIMMessage()) {
            FileInfo fileInfo = mapFileInfo(imMessage);
            talkMessageBean.setFileInfo(fileInfo);
            return talkMessageBean;
        }

        //对不支持的消息类型转化成文字提醒消息
//        if (talkMessageBean.getMessageType() == ConstDef.MSG_TYPE_DEFAULT) {
//            talkMessageBean.setContent(ActomaController.getAppConfigContext().getString(R.string.unsupport_message_type));
//            talkMessageBean.setMessageType(ConstDef.MSG_TYPE_PRESENTATION);
//            return talkMessageBean;
//        }
        return talkMessageBean;
    }

    /**
     * 消息业务对象转换为消息实体对象
     * TalkMessageBean ==> IMMessage
     *
     * @param talkMessageBean 消息业务对象
     * @return 消息实体对象
     */
    public IMMessage mapIMMessage(TalkMessageBean talkMessageBean) {
        if (talkMessageBean == null) {
            return null;
        }

        IMMessage imMessage = new IMMessage();
        imMessage.setIMMessageId(talkMessageBean.getId());
        imMessage.setFrom(talkMessageBean.getFrom());
        imMessage.setCardId(talkMessageBean.getSenderCardId());
        imMessage.setTo(talkMessageBean.getTo());
        imMessage.setFailCode(talkMessageBean.getFailCode());
        imMessage.setType(ValueConverter.talkMsgTypeConvert(talkMessageBean));
        imMessage.setIMMessageTime(talkMessageBean.getShowTime());
        imMessage.setTimeToLive(talkMessageBean.getLimitTime());
        imMessage.setMessageBody(mapBody(talkMessageBean, imMessage.isFileIMMessage()));

        imMessage.setState(ValueConverter.talkMsgStateConvert(talkMessageBean.getMessageState()));
        return imMessage;
    }

    /**
     * 文件业务对象转换为文件实体对象
     * FileInfo ==> IMFileInfo
     *
     * @param fileInfo
     * @return
     */
    public IMFileInfo mapIMFileInfo(FileInfo fileInfo) {
        IMFileInfo imFileInfo = new IMFileInfo();

        IMMessage message = new IMMessage();
        message.setIMMessageId(fileInfo.getTalkMessageId());
        if (fileInfo instanceof ImageFileInfo) {
            ImageFileInfo imageFileInfo = (ImageFileInfo) fileInfo;
            imageFileInfo.getFileType();
            if (imageFileInfo.getType() == ConstDef.IS_HD) {
                imFileInfo.setFileType(ImSdkFileConstant.FileType.IS_HD);
            } else if (imageFileInfo.getType() == ConstDef.IS_RAW) {
                imFileInfo.setFileType(ImSdkFileConstant.FileType.IS_RAW);
            } else {
                imFileInfo.setFileType(ImSdkFileConstant.FileType.IS_SHOW);
            }
        } else if (fileInfo instanceof VideoFileInfo) {
            VideoFileInfo videoFileInfo = (VideoFileInfo) fileInfo;
            if (videoFileInfo.getType() == ConstDef.IS_RAW) {
                imFileInfo.setFileType(ImSdkFileConstant.FileType.IS_RAW);
            } else {
                imFileInfo.setFileType(ImSdkFileConstant.FileType.IS_SHOW);
            }
        } else {
            imFileInfo.setFileType(ImSdkFileConstant.FileType.IS_SHOW);
        }
        imFileInfo.setMessage(message);
        return imFileInfo;
    }

    /**
     * 消息文件业务对象转换为ImSdk消息内容对象
     * FileInfo ==> IMFileBody
     *
     * @param fileInfo 消息文件业务对象
     * @return ImSdk消息内容对象
     */
    public IMFileBody mapFileBody(FileInfo fileInfo) {

        IMFileBody fileBody = null;
        if (fileInfo instanceof VoiceFileInfo) {
            VoiceFileInfo voiceFileInfo = (VoiceFileInfo) fileInfo;
            fileBody = new IMVoiceBody(ImSdkFileConstant.FILE_VOICE, voiceFileInfo.getAmountOfTime());
        } else if (fileInfo instanceof ImageFileInfo) {
            ImageFileInfo imageFileInfo = (ImageFileInfo) fileInfo;
            if (fileInfo.getFileType() == ConstDef.FILE_TYPE_PHOTO) {
                IMPicBody imageBody = new IMPicBody(ImSdkFileConstant.FILE_PIC);
                FileInfo info = imageFileInfo.getRawFileInfo();
                if (info != null) {
                    imageBody.setRawFileInfo(mapRaw(info));
                }
                fileBody = imageBody;
            } else {
                IMImageBody imageBody = new IMImageBody(ImSdkFileConstant.FILE_IMAGE);
                FileInfo hdFileInfo = imageFileInfo.getHdFileInfo();
                if (hdFileInfo != null) {
                    imageBody.setHdTFileInfo(mapIMHDThumbFileInfo(hdFileInfo));
                }

                FileInfo rawFileInfo = imageFileInfo.getRawFileInfo();
                if (rawFileInfo != null) {
                    imageBody.setRawFileInfo(mapIMRawFileInfo(rawFileInfo));
                }
                fileBody = imageBody;
            }
        } else if (fileInfo instanceof VideoFileInfo) {

            if (fileInfo.getFileType() == ConstDef.FILE_TYPE_VIDEO) {
                VideoFileInfo videoFileInfo = (VideoFileInfo) fileInfo;

                IMMediaBody videoBody = new IMMediaBody(ImSdkFileConstant.FILE_MEDIA);
                videoBody.setDuration(videoFileInfo.getAmountOfTime());
                videoBody.setSize(videoFileInfo.getFileSize());
                FileInfo rawFileInfo = videoFileInfo.getRawFileInfo();
                videoBody.setRawFileInfo(mapRaw(rawFileInfo));
                fileBody = videoBody;
            } else {
                VideoFileInfo videoFileInfo = (VideoFileInfo) fileInfo;
                FileInfo rawFileInfo = videoFileInfo.getRawFileInfo();

                if (rawFileInfo != null) {
                    IMVideoBody videoBody = new IMVideoBody(ImSdkFileConstant.FILE_VIDEO,
                            videoFileInfo.getAmountOfTime(), rawFileInfo.getFileSize());
                    videoBody.setRawFileInfo(mapIMRawFileInfo(rawFileInfo));
                    fileBody = videoBody;
                }
            }
        } else {
            fileBody = new IMFileBody(ImSdkFileConstant.FILE_NORMAL);
        }
        fileBody.setLocalPath(fileInfo.getFilePath());
        fileBody.setFileSize(fileInfo.getFileSize());
        fileBody.setTranslateSize(fileInfo.getTranslateSize());
        fileBody.setDisplayName(fileInfo.getFileName());
        fileBody.setSuffix(fileInfo.getSuffix());
        return fileBody;
    }

    private IMRawFileInfo mapRaw(FileInfo extra) {
        IMRawFileInfo rawFileInfo = new IMRawFileInfo();
        rawFileInfo.setRawDisplayName(extra.getFileName());
        rawFileInfo.setRawLocalPath(extra.getFilePath());
        rawFileInfo.setRawFileSize(extra.getFileSize());
        rawFileInfo.setRawFileTranslateSize(extra.getTranslateSize());
        return rawFileInfo;
    }

    /**
     * 消息实体对象转换为文件对象
     * IMMessage ==> FileInfo
     *
     * @param message
     * @return
     */
    private FileInfo mapFileInfo(IMMessage message) {
        IMMessageBody body = message.getMessageBody();
        FileInfo fileInfo;
        if (body instanceof IMVoiceBody) {
            fileInfo = mapVoiceFileInfo(body);
        } else if (body instanceof IMPicBody) {
            fileInfo = mapPicFileInfo(body);
        } else if (body instanceof IMImageBody) {
            fileInfo = mapImageFileInfo(body);
        } else if (body instanceof IMMediaBody) {
            fileInfo = mapMediaFileInfo(body);
        } else if (body instanceof IMVideoBody) {
            fileInfo = mapVideoFileInfo(body);
        } else if (body instanceof IMNormalFileBody) {
            fileInfo = mapNormalFileInfo(body);
        } else {
            fileInfo = new FileInfo();
        }
        IMFileBody fileBody = (IMFileBody) body;
        fileInfo.setFileState(mapFileState(fileBody.getState()));
        fileInfo.setTranslateSize(fileBody.getTranslateSize());
        fileInfo.setFileName(fileBody.getDisplayName());
        fileInfo.setFileSize(fileBody.getFileSize());
        fileInfo.setFilePath(fileBody.getLocalPath());
        fileInfo.setSuffix(fileBody.getSuffix());
        fileInfo.setTalkMessageId(message.getIMMessageId());
        return fileInfo;
    }


    /**
     * 文件操作回调参数转换得到文件信息业务对象
     * IMFileInfo ==> FileInfo
     *
     * @param imFileInfo 文件操作回调参数
     * @return 文件信息业务对象
     */
    public FileInfo mapFileInfo(IMFileInfo imFileInfo) {
        IMMessage message = imFileInfo.getMessage();
        if (message.isRecalled()) {
            LogUtil.d("message is already recall.");
            return null;
        }
        FileInfo fileInfo = mapFileInfo(message);
        fileInfo.setFileProperty(mapFileProperty(imFileInfo.getFileType()));
        return fileInfo;
    }

    /**
     * FileType ==> int
     *
     * @param fileType
     * @return int {@link com.xdja.im.core.config.ConstDef.FileProperty}
     */
    public int mapFileProperty(ImSdkFileConstant.FileType fileType) {
        int type = ConstDef.IS_SHOW;
        if (fileType == ImSdkFileConstant.FileType.IS_SHOW) {
            type = ConstDef.IS_SHOW;
        } else if (fileType == ImSdkFileConstant.FileType.IS_HD) {
            type = ConstDef.IS_HD;
        } else if (fileType == ImSdkFileConstant.FileType.IS_RAW) {
            type = ConstDef.IS_RAW;
        }
        return type;
    }

    public IMATBody mapAtBody(AtInfo info) {
        IMATBody imatBody = new IMATBody(info.getContent(), info.getAccounts());
        return imatBody;
    }

    /**
     * FileState ==> int
     *
     * @param state state
     * @return int {@link com.xdja.im.core.config.ConstDef.FileState}
     */
    private int mapFileState(ImSdkFileConstant.FileState state) {
        int result = ConstDef.INACTIVE;
        if (state == ImSdkFileConstant.FileState.DOWNLOAD_DONE) {
            result = ConstDef.DONE;
        } else if (state == ImSdkFileConstant.FileState.DOWNLOAD_FAIL) {
            result = ConstDef.FAIL;
        } else if (state == ImSdkFileConstant.FileState.DOWNLOAD_INACTIVE) {
            result = ConstDef.INACTIVE;
        } else if (state == ImSdkFileConstant.FileState.DOWNLOAD_LOADING) {
            result = ConstDef.LOADING;
        } else if (state == ImSdkFileConstant.FileState.DOWNLOAD_PAUSE) {
            result = ConstDef.PAUSE;
        } else if (state == ImSdkFileConstant.FileState.UPLOAD_DONE) {
            result = ConstDef.UPLOAD_DONE;
        } else if (state == ImSdkFileConstant.FileState.UPLOAD_FAIL) {
            result = ConstDef.UPLOAD_FAIL;
        } else if (state == ImSdkFileConstant.FileState.UPLOAD_INACTIVE) {
            result = ConstDef.UPLOAD_INACTIVE;
        } else if (state == ImSdkFileConstant.FileState.UPLOAD_LOADING) {
            result = ConstDef.UPLOAD_LOADING;
        } else if (state == ImSdkFileConstant.FileState.UPLOAD_PAUSE) {
            result = ConstDef.UPLOAD_PAUSE;
        }
        return result;
    }

    /**
     * 文件实体转化为语音消息对象
     *
     * @param fileBody 文件实体类
     * @return
     */
    private VoiceFileInfo mapVoiceFileInfo(IMMessageBody fileBody) {
        IMVoiceBody voiceBody = (IMVoiceBody) fileBody;
        VoiceFileInfo fileInfo = new VoiceFileInfo();
        fileInfo.setFileType(ConstDef.FILE_TYPE_VOICE);
        fileInfo.setAmountOfTime(voiceBody.getDuration());
        return fileInfo;
    }

    /**
     * 文件实体转化为图片消息对象
     *
     * @param fileBody
     * @return
     */
    private ImageFileInfo mapPicFileInfo(IMMessageBody fileBody) {
        IMPicBody imageBody = (IMPicBody) fileBody;
        ImageFileInfo fileInfo = new ImageFileInfo();
        fileInfo.setFileType(ConstDef.FILE_TYPE_PHOTO);

        IMRawFileInfo raw = imageBody.getRawFileInfo();
        //高清缩略图
        //原图
        if (raw != null) {
            fileInfo.setOriginal(true);
            FileInfo rawFileInfo = new FileInfo();
            rawFileInfo.setFileName(raw.getRawDisplayName());
            rawFileInfo.setFilePath(raw.getRawLocalPath());
            rawFileInfo.setFileSize(raw.getRawFileSize());
            rawFileInfo.setTranslateSize(raw.getRawFileTranslateSize());
//            rawFileInfo.setFileState(mapFileState(((IMPicBody) fileBody).getState()));
            rawFileInfo.setFileState((mapFileState(raw.getRawState())));

            fileInfo.setFileName(raw.getRawDisplayName());
            fileInfo.setFilePath(raw.getRawLocalPath());
            fileInfo.setFileSize(raw.getRawFileSize());
            fileInfo.setTranslateSize(raw.getRawFileTranslateSize());
//            fileInfo.setFileState(mapFileState(((IMPicBody) fileBody).getState()));
			fileInfo.setFileState(mapFileState(raw.getRawState()));
            fileInfo.setRawFileInfo(rawFileInfo);
        }
        return fileInfo;
    }

    /**
     * 文件实体转化为图片消息对象
     *
     * @param fileBody
     * @return
     */
    private ImageFileInfo mapImageFileInfo(IMMessageBody fileBody) {
        IMImageBody imageBody = (IMImageBody) fileBody;
        ImageFileInfo fileInfo = new ImageFileInfo();
        fileInfo.setFileType(ConstDef.FILE_TYPE_PHOTO);

        IMHDThumbFileInfo hd = imageBody.getHdTFileInfo();
        IMRawFileInfo raw = imageBody.getRawFileInfo();
        //高清缩略图
        if (hd != null) {
            FileInfo hdFileInfo = new FileInfo();
            hdFileInfo.setFileName(hd.getHdTDisplayName());
            hdFileInfo.setFilePath(hd.getHdTLocalPath());
            hdFileInfo.setFileSize(hd.getHdTFileSize());
            hdFileInfo.setTranslateSize(hd.getHdTFileTranslateSize());
            hdFileInfo.setFileState(mapFileState(hd.getHdState()));
            fileInfo.setHdFileInfo(hdFileInfo);
        }

        //原图
        if (raw != null) {
            fileInfo.setOriginal(true);
            FileInfo rawFileInfo = new FileInfo();
            rawFileInfo.setFileName(raw.getRawDisplayName());
            rawFileInfo.setFilePath(raw.getRawLocalPath());
            rawFileInfo.setFileSize(raw.getRawFileSize());
            rawFileInfo.setTranslateSize(raw.getRawFileTranslateSize());
            rawFileInfo.setFileState(mapFileState(raw.getRawState()));
            fileInfo.setRawFileInfo(rawFileInfo);
        }
        return fileInfo;
    }


    /**
     * 文件实体转化为视频消息对象
     *
     * @param fileBody
     * @return
     */
    private VideoFileInfo mapMediaFileInfo(IMMessageBody fileBody) {
        IMMediaBody videoBody = (IMMediaBody) fileBody;
        VideoFileInfo fileInfo = new VideoFileInfo();
        fileInfo.setFileType(ConstDef.FILE_TYPE_VIDEO);
        fileInfo.setAmountOfTime(videoBody.getDuration());
        IMRawFileInfo raw = videoBody.getRawFileInfo();
        //视频源文件信息
        if (raw != null) {
            FileInfo rawFileInfo = new FileInfo();
            rawFileInfo.setFileName(raw.getRawDisplayName());
            rawFileInfo.setFilePath(raw.getRawLocalPath());
            rawFileInfo.setFileSize(raw.getRawFileSize());
            rawFileInfo.setTranslateSize(raw.getRawFileTranslateSize());
            rawFileInfo.setFileState(mapFileState(raw.getRawState()));
            fileInfo.setRawFileSize(raw.getRawFileSize());
            fileInfo.setRawFileInfo(rawFileInfo);
        }
        return fileInfo;
    }

    /**
     * 文件实体转化为视频消息对象
     *
     * @param fileBody
     * @return
     */
    private VideoFileInfo mapVideoFileInfo(IMMessageBody fileBody) {
        IMVideoBody videoBody = (IMVideoBody) fileBody;
        VideoFileInfo fileInfo = new VideoFileInfo();
        fileInfo.setFileType(ConstDef.FILE_TYPE_VIDEO);
        fileInfo.setAmountOfTime(videoBody.getDuration());
        fileInfo.setRawFileSize(videoBody.getSize());
        IMRawFileInfo raw = videoBody.getRawFileInfo();
        //视频源文件信息
        if (raw != null) {
            FileInfo rawFileInfo = new FileInfo();
            rawFileInfo.setFileName(raw.getRawDisplayName());
            rawFileInfo.setFilePath(raw.getRawLocalPath());
            rawFileInfo.setFileSize(raw.getRawFileSize());
            rawFileInfo.setTranslateSize(raw.getRawFileTranslateSize());
            rawFileInfo.setFileState(mapFileState(raw.getRawState()));
            fileInfo.setRawFileInfo(rawFileInfo);
        }
        return fileInfo;
    }

    /**
     * 文件实体转化为视频消息对象
     *
     * @param fileBody
     * @return
     */
    private NormalFileInfo mapNormalFileInfo(IMMessageBody fileBody) {
        IMNormalFileBody normalFileBody = (IMNormalFileBody) fileBody;
        NormalFileInfo fileInfo = new NormalFileInfo();
        fileInfo.setFileType(ConstDef.FILE_TYPE_NORMAL);
        fileInfo.setNormalFileType(ValueConverter.imNormalFileTypeConvert(normalFileBody));
        return fileInfo;
    }

    /**
     * 文件扩展信息业务对象转换为ImSdk高清缩略图文件信息
     *
     * @param fileInfo 文件信息业务对象
     * @return ImSdk高清缩略图文件信息
     */
    private IMHDThumbFileInfo mapIMHDThumbFileInfo(FileInfo fileInfo) {
        IMHDThumbFileInfo hdThumbFileInfo = new IMHDThumbFileInfo();
        hdThumbFileInfo.setHdTDisplayName(fileInfo.getFileName());
        hdThumbFileInfo.setHdTLocalPath(fileInfo.getFilePath());
        hdThumbFileInfo.setHdTFileSize(fileInfo.getFileSize());
        hdThumbFileInfo.setHdTFileTranslateSize(fileInfo.getTranslateSize());
        return hdThumbFileInfo;
    }

    /**
     * 文件扩展信息业务对象转换为ImSdk原始文件信息
     *
     * @param fileInfo 文件信息业务对象
     * @return ImSdk高清缩略图文件信息
     */
    private IMRawFileInfo mapIMRawFileInfo(FileInfo fileInfo) {
        IMRawFileInfo rawFileInfo = new IMRawFileInfo();
        rawFileInfo.setRawDisplayName(fileInfo.getFileName());
        rawFileInfo.setRawLocalPath(fileInfo.getFilePath());
        rawFileInfo.setRawFileSize(fileInfo.getFileSize());
        rawFileInfo.setRawFileTranslateSize(fileInfo.getTranslateSize());
        return rawFileInfo;
    }

    /**
     * 消息业务对象转换为ImSdk消息内容对象
     *
     * @param talkMessageBean 消息业务对象
     * @param isFile          是否是文件类型
     * @return ImSdk消息内容对象
     */
    private IMMessageBody mapBody(TalkMessageBean talkMessageBean, boolean isFile) {
        if (!isFile) {
            return new IMTextBody(talkMessageBean.getContent());
        }
        if (talkMessageBean.getFileInfo() == null) {
            return new IMFileBody(ImSdkFileConstant.FILE_UNKNOWN);
        }
        return mapFileBody(talkMessageBean.getFileInfo());
    }

    /**
     * 网页消息业务对象转换为ImSdk消息内容对象
     * @param info 消息文件业务对象
     * @return ImSdk消息内容对象
     */
    public IMWebBody mapWebBody(WebPageInfo info) {
        IMWebBody imWebBody = new IMWebBody(ImSdkFileConstant.FILE_WEB);
        imWebBody.setTitle(info.getTitle());
        imWebBody.setSource(info.getSource());
        imWebBody.setDescription(info.getDescription());
        imWebBody.setUrl(info.getWebUri());
        imWebBody.setLocalPath(info.getFilePath());
        imWebBody.setFileSize(info.getFileSize());
        imWebBody.setTranslateSize(info.getTranslateSize());
        imWebBody.setDisplayName(info.getFileName());
        imWebBody.setSuffix(info.getSuffix());
        return imWebBody;
    }

    /**
     *
     * @param size
     * @param tSize
     * @return
     */
    private int getPercent(long size, long tSize) {
        if (size == 0) {
            return 0;
        }
        if (tSize >= size) {
            return 100;
        }
        return (int) (tSize * 100 / size);
    }
}
