package com.xdja.im.lib.album.utils;

import android.content.ContentResolver;
import android.content.Context;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.provider.MediaStore;
import android.text.TextUtils;

import com.xdja.im.common.utils.FileSizeUtils;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.ImageFileInfo;
import com.xdja.im.core.utils.FilePathUtils;
import com.xdja.im.lib.album.ImagePickerConfig;
import com.xdja.im.lib.album.ImagePickerManager;
import com.xdja.im.lib.album.bean.ImageItem;
import com.xdja.im.lib.album.bean.ImageThumbBean;
import com.xdja.im.lib.album.bean.PickerType;
import com.xdja.im.lib.filemanage.util.ToolUtil;
import com.xdja.im.uikit.utils.file.FileUtils;
import com.xdja.im.uikit.utils.gif.GifDecoder;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.imsdk.constant.ImSdkFileConstant;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import rx.Observable;
import rx.functions.Func1;
import rx.functions.Func3;
import rx.schedulers.Schedulers;


/**
 * @Package: com.xdja.im.lib.album.utils
 * @Author: xdjaxa
 * @Creation: 2017-04-24 15:55
 * @Version V1.0
 * @Description:数据处理工具类
 */
public class DataTools {


    /**
     * 加载本地图片文件
     *
     * @param context 上下文
     * @return 本地图片列表
     */
    public static List<ImageItem> loadImageData(Context context) {
        //查询内容
        Uri mImageUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI;
        ContentResolver mContentResolver = context.getContentResolver();
        //查询内容
        String[] projection = {MediaStore.Images.Media._ID,
                MediaStore.Images.Media.DISPLAY_NAME,
                MediaStore.Images.Media.DATA,
                MediaStore.Images.Media.SIZE,
                MediaStore.Images.Media.WIDTH,
                MediaStore.Images.Media.HEIGHT};
        //排序
        String sortOrder = MediaStore.Images.Media.DATE_MODIFIED + " desc";
        //检索
        Cursor cursor = mContentResolver.query(mImageUri, projection, null, null, sortOrder);
        List<ImageItem> localPicInfoList = new ArrayList<>();
        if (cursor != null) {
            while (cursor.moveToNext()) {
                // 获取图片大小
                int width = cursor.getInt(cursor.getColumnIndex(MediaStore.Images.Media.WIDTH));
                int height = cursor.getInt(cursor.getColumnIndex(MediaStore.Images.Media.HEIGHT));
                String path = cursor.getString(cursor.getColumnIndex(MediaStore.Images.Media.DATA));
                String name = cursor.getString(cursor.getColumnIndex(MediaStore.Images.Media.DISPLAY_NAME));
                int size = cursor.getInt(cursor.getColumnIndex(MediaStore.Images.Media.SIZE));

                if (!FileUtils.isFileExists(path)) {
                    continue;
                }
                if (width > 0 && height > 0) {
                    ImageItem info = new ImageItem(name, path, size, width, height,
                            PickerType.TYPE_IMAGE);

                    if (!localPicInfoList.contains(info)) {
                        localPicInfoList.add(info);
                    }
                } else {
                    //如果图库读取大小为0，则实地加载图片（该方法比较耗时）
                    BitmapFactory.Options options = new BitmapFactory.Options();
                    options.inJustDecodeBounds = true;
                    BitmapFactory.decodeFile(path, options);
                    if (options.outWidth > 0 && options.outHeight > 0) {
                        ImageItem info = new ImageItem(name, path, size, options.outWidth,
                                options.outHeight, PickerType.TYPE_IMAGE);
                        if (!localPicInfoList.contains(info)) {
                            localPicInfoList.add(info);
                        }
                    }
                }
            }
            cursor.close();
        }
        return localPicInfoList;
    }

    /**
     * 生成图片缩略图信息
     *
     * @param imageList 图片原信息
     * @return 图片信息列表：至少包含图片缩略图和高清缩略图，如果需要发送原图，则含有原图信息
     */
    public static Observable<List<ImageThumbBean>> createImageThumb(List<ImageItem> imageList) {

        return Observable.from(imageList)
                .concatMap(new Func1<ImageItem, Observable<ImageThumbBean>>() {
                    @Override
                    public Observable<ImageThumbBean> call(ImageItem imageItem) {
                        return createThumbNails(imageItem);
                    }
                }).toList();
    }

    /**
     * 单张图片生成缩略图
     *
     * @param imageItem
     * @return
     */
    public static Observable<ImageThumbBean> createThumbNails(final ImageItem imageItem) {

        if (!FileUtils.isFileExists(imageItem.getPath())) {
            return Observable.error(new IllegalArgumentException("File not exist(" + imageItem.getPath() + ")"));
        }

        String suffix = imageItem.getPath().substring(imageItem.getPath().lastIndexOf(".") + 1).toLowerCase();
        if (ConstDef.GIF_TYPE.equalsIgnoreCase(suffix)) {
            return createGifImageFileInfo(imageItem, false);
        } else {
            //缩略图文件保存父路径
            String imageRootPath = ImagePickerManager.getInstance().getConfig().getCachePath();
            if (TextUtils.isEmpty(imageRootPath)) {
                imageRootPath = FilePathUtils.getImagePath();
            }
            //文件名称
            final String baseFileName = UUID.randomUUID().toString();
            final String thumbName = ImagePickerConfig.THUMBNAIL_FILE_PREFIX + baseFileName;
            final String hdThumbName = ImagePickerConfig.THUMBNAIL_FILE_PREFIX + baseFileName + "_hd";

            return Observable.zip(
                    //高清缩略图
                    createHDThumbnail(imageItem.getPath(), new File(imageRootPath, hdThumbName)),
                    //缩略图
                    createThumbnail(imageItem.getPath(), new File(imageRootPath, thumbName)),
                    //原图
                    createRawFilePath(imageItem.getPath(), new File(imageRootPath, baseFileName), imageItem.isOriginal()),
                    //生成实体类
                    new Func3<String, String, String, ImageThumbBean>() {

                        @Override
                        public ImageThumbBean call(String hdPath, String thumbPath, String rawPath) {
                            ImageThumbBean thumbBean = new ImageThumbBean();
                            thumbBean.setThumbName(thumbName);
                            thumbBean.setThumbPath(thumbPath);
                            thumbBean.setThumbSize(FileSizeUtils.getFileSize(thumbPath));

                            thumbBean.setHdThumbName(hdThumbName);
                            thumbBean.setHdThumbPath(hdPath);
                            thumbBean.setHdThumbSize(FileSizeUtils.getFileSize(hdPath));

                            //原图信息
                            if (imageItem.isOriginal()) {
                                thumbBean.setRawName(imageItem.getName());
                                thumbBean.setRawPath(rawPath);
                                thumbBean.setRawSize(FileSizeUtils.getFileSize(rawPath));
                            }
                            return thumbBean;
                        }
                    }
            );
        }
    }


    /**
     * 高清缩略图处理
     *
     * @param originalPath
     * @param thumbFile
     * @return
     */
    private static Observable<String> createHDThumbnail(final String originalPath,
                                                        final File thumbFile) {

        return Observable.just(originalPath)
                .subscribeOn(Schedulers.newThread())
                .map(new Func1<String, String>() {
                    @Override
                    public String call(String path) {
                        return ThumbnailUtils.createImageThumbnail(path,
                                ThumbnailUtils.KIND_HD, thumbFile);
                    }
                });
    }

    /**
     * 缩略图处理
     *
     * @param originalPath
     * @param thumbFile
     */
    private static Observable<String> createThumbnail(final String originalPath,
                                                      final File thumbFile) {
        return Observable.just(originalPath)
                .subscribeOn(Schedulers.newThread())
                .map(new Func1<String, String>() {
                    @Override
                    public String call(String path) {
                        String suffix = path.substring(path.lastIndexOf(".") + 1).toLowerCase();
                        if (!TextUtils.isEmpty(suffix)) {
                            if (suffix.equals("gif")) {
                                GifDecoder gifDecoder = new GifDecoder();
                                File file = new File(originalPath);
                                try {
                                    byte[] bytes = gifDecoder.getByte(file);
                                    gifDecoder.read(bytes);
                                    gifDecoder.setFrameIndex(0);
                                    Bitmap bitmap = gifDecoder.getNextFrame();
                                    return ThumbnailUtils.saveBitmap(bitmap, 20, thumbFile);
                                } catch (Exception e) {
                                }
                            } else {
                                return ThumbnailUtils.createImageThumbnail(path,
                                        ThumbnailUtils.KIND_TH, thumbFile);
                            }
                        }
                        return null;

                    }
                });
    }

    /**
     * 原图处理
     *
     * @param originalPath
     * @param filePath
     * @param isOriginalPic
     * @return
     */
    private static Observable<String> createRawFilePath(final String originalPath,
                                                        final File filePath,
                                                        final boolean isOriginalPic) {
        return Observable.just(originalPath)
                .subscribeOn(Schedulers.io())
                .map(new Func1<String, String>() {
                    @Override
                    public String call(String originalPath) {
                        //将原图拷贝至缓存目录下
                        //if (isOriginalPic) {
                        //    return ThumbUtils.copyFile(originalPath, filePath);
                        //} else {
                        return originalPath;
                        //}
                    }
                });
    }


    /**
     * 根据本地动态图片信息，生成图片文件相关信息
     *
     * @param imageItem 本地图片信息
     * @return 图片文件信息实体类
     */
    public static Observable<ImageThumbBean> createGifImageFileInfo(final ImageItem imageItem,
                                                                    final boolean isFile) {
        if (imageItem == null) {
            return null;
        }

        final String imageFilePath = imageItem.getPath();

        try {
            File file = new File(imageFilePath);
            if (!file.exists()) {
                return Observable.error(new IllegalArgumentException("File not exist"));
            }
        } catch (Exception e) {
            LogUtil.e("", e);
        }

        //缩略图文件保存父路径
        String imageRootPath = ImagePickerManager.getInstance().getConfig().getCachePath();
        //文件名称
        final String baseFileName = UUID.randomUUID().toString();
        final String thumbName = ImSdkFileConstant.THUMBNAIL_FILE_PREFIX + baseFileName;
        final String hdThumbName = ImagePickerConfig.THUMBNAIL_FILE_PREFIX + baseFileName + "_hd";
        return Observable.zip(
                //高清缩略图
                createRawFilePath(imageFilePath, new File(imageRootPath, baseFileName), imageItem.isOriginal()),
                //缩略图
                createGifFirstFrameFilePath(imageFilePath, new File(imageRootPath, thumbName)),
                //原图
                createRawFilePath(imageFilePath, new File(imageRootPath, baseFileName), imageItem.isOriginal()),
                //生成实体类
                new Func3<String, String, String, ImageThumbBean>() {

                    @Override
                    public ImageThumbBean call(String hdThumbFilePath,
                                               String thumbFilePath,
                                               String rawFilePath) {
                        if (TextUtils.isEmpty(hdThumbFilePath) ||
                                TextUtils.isEmpty(rawFilePath)) {
                            return null;
                        }
                        if (TextUtils.isEmpty(thumbFilePath)) {
                            thumbFilePath = rawFilePath;
                        }
                        //fix bug 21243 by jyg 20171017
                        if (isFile) {
//                            PictureFileInfo imageFileInfo = new PictureFileInfo();
//                            //缩略图信息
//                            imageFileInfo.setFileName(thumbName);
//                            imageFileInfo.setFilePath(thumbFilePath);
//                            imageFileInfo.setFileSize(ToolUtil.getFileSize(thumbFilePath));
//                            imageFileInfo.setTranslateSize(0);
//                            imageFileInfo.setSuffix(ConstDef.GIF_TYPE);
//
//                            FileExtraInfo fileExtraInfo = new FileExtraInfo();
//                            //高清缩略图信息
//                            fileExtraInfo.setRawFileName(baseFileName);
//                            fileExtraInfo.setRawFileUrl(imageFilePath);
//                            fileExtraInfo.setRawFileSize(ToolUtil.getFileSize(imageFilePath));
//                            imageFileInfo.setOriginal(true);
//                            //数据填充
//                            imageFileInfo.setExtraInfo(fileExtraInfo);
//                            return imageFileInfo;
                            ImageFileInfo imageFileInfo = new ImageFileInfo();
                            //缩略图信息
                            imageFileInfo.setFileName(thumbName);
                            imageFileInfo.setFilePath(thumbFilePath);
                            imageFileInfo.setFileSize(ToolUtil.getFileSize(thumbFilePath));
                            imageFileInfo.setTranslateSize(0);
                            imageFileInfo.setSuffix(ConstDef.GIF_TYPE);

                            FileInfo fileExtraInfo = new FileInfo();
                            //高清缩略图信息
                            fileExtraInfo.setFileName(baseFileName);
                            fileExtraInfo.setFilePath(imageFilePath);
                            fileExtraInfo.setFileSize(ToolUtil.getFileSize(hdThumbFilePath));
                            fileExtraInfo.setTranslatePercent(0);
                            imageFileInfo.setOriginal(false);
                            //数据填充
                            imageFileInfo.setHdFileInfo(fileExtraInfo);


                            ImageThumbBean thumbBean = new ImageThumbBean();
                            thumbBean.setThumbName(thumbName);
                            thumbBean.setThumbPath(thumbFilePath);
                            thumbBean.setThumbSize(FileSizeUtils.getFileSize(thumbFilePath));

                            thumbBean.setHdThumbName(hdThumbName);
                            thumbBean.setHdThumbPath(hdThumbFilePath);
                            thumbBean.setHdThumbSize(FileSizeUtils.getFileSize(hdThumbFilePath));

                            //原图信息
                            thumbBean.setRawName(imageItem.getName());
                            thumbBean.setRawPath(rawFilePath);
                            thumbBean.setRawSize(FileSizeUtils.getFileSize(rawFilePath));

                            return thumbBean;
                        } else {
                            ImageFileInfo imageFileInfo = new ImageFileInfo();
                            //缩略图信息
                            imageFileInfo.setFileName(thumbName);
                            imageFileInfo.setFilePath(thumbFilePath);
                            imageFileInfo.setFileSize(ToolUtil.getFileSize(thumbFilePath));
                            imageFileInfo.setTranslateSize(0);
                            imageFileInfo.setSuffix(ConstDef.GIF_TYPE);

                            FileInfo fileExtraInfo = new FileInfo();
                            //高清缩略图信息
                            fileExtraInfo.setFileName(baseFileName);
                            fileExtraInfo.setFilePath(imageFilePath);
                            fileExtraInfo.setFileSize(ToolUtil.getFileSize(hdThumbFilePath));
                            fileExtraInfo.setTranslatePercent(0);
                            imageFileInfo.setOriginal(false);
                            //数据填充
                            imageFileInfo.setHdFileInfo(fileExtraInfo);


                            ImageThumbBean thumbBean = new ImageThumbBean();
                            thumbBean.setThumbName(thumbName);
                            thumbBean.setThumbPath(thumbFilePath);
                            thumbBean.setThumbSize(FileSizeUtils.getFileSize(thumbFilePath));

                            thumbBean.setHdThumbName(hdThumbName);
                            thumbBean.setHdThumbPath(hdThumbFilePath);
                            thumbBean.setHdThumbSize(FileSizeUtils.getFileSize(hdThumbFilePath));

                            //原图信息
                            thumbBean.setRawName(imageItem.getName());
                            thumbBean.setRawPath(rawFilePath);
                            thumbBean.setRawSize(FileSizeUtils.getFileSize(rawFilePath));

                            return thumbBean;
                        }
                    }
                }
        );
    }

    /**
     * gif原图处理
     *
     * @param originalPath
     * @param thumbFile
     * @return
     */
    public static Observable<String> createGifFirstFrameFilePath(final String originalPath,
                                                                 final File thumbFile) {

        return Observable.just(originalPath)
                .subscribeOn(Schedulers.io())
                .map(new Func1<String, String>() {
                    @Override
                    public String call(String originalPath) {
                        GifDecoder gifDecoder = new GifDecoder();
                        File file = new File(originalPath);
                        try {
                            byte[] bytes = gifDecoder.getByte(file);
                            gifDecoder.read(bytes);
                            gifDecoder.setFrameIndex(0);
                            Bitmap bitmap = gifDecoder.getNextFrame();
                            return com.xdja.imsdk.util.ThumbnailUtils.saveBitmap(bitmap, 20, thumbFile);
                        } catch (Exception e) {
                        }
                        return null;
                    }
                });
    }

}
