package com.xdja.im.lib.camera.ui.activity;

import android.Manifest;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.hardware.Camera;
import android.os.Build;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.common.utils.PermissionUtils;
import com.xdja.im.lib.camera.CameraManager;
import com.xdja.im.lib.camera.bean.PhotoItem;
import com.xdja.im.lib.camera.bean.PhotoThumbBean;
import com.xdja.im.lib.camera.ui.presenter.TakePhotoPresenter;
import com.xdja.im.lib.camera.ui.view.ViewTakePhoto;
import com.xdja.im.lib.camera.ui.view.vu.TakePhotoVu;
import com.xdja.im.lib.camera.utils.CameraHelper;
import com.xdja.im.lib.camera.utils.DataTools;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.base.ImBaseActivity;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.utils.log.LogUtil;

import rx.Observable;
import rx.Subscriber;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * @Package: com.xdja.im.lib.camera.ui
 * @Author: xdjaxa
 * @Creation: 2017-04-26 10:13
 * @Version V1.0
 * @Description:相机拍照和预览界面
 */
public class TakePhotoActivity extends ImBaseActivity<TakePhotoPresenter, TakePhotoVu>
        implements TakePhotoPresenter {

    @NonNull
    @Override
    protected Class<? extends TakePhotoVu> getVuClass() {
        return ViewTakePhoto.class;
    }

    @NonNull
    @Override
    protected TakePhotoPresenter getPresenter() {
        return this;
    }

    @Override
    protected void onBindView(@Nullable Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);
        //请求相机权限
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            openCamera();
        } else {
            requestPermission();
        }
    }

    /**
     * 尝试打开相机
     */
    @SuppressWarnings("SuspiciousNameCombination")
    private void openCamera() {
        Camera camera = null;
        try {
            camera = Camera.open(0);
            CameraHelper.takePhoto(this, Constant.RESULT_CODE_CAMERA);
        } catch (Exception e) {
            LogUtil.e("ERROR: Open camera error.");
            CommonTool.showToast(R.string.im_uikit_camera_deny_for_occupy);
            finish();
        } finally {
            if (camera != null) {
                camera.release();
                camera = null;
            }
        }
    }

    /**
     * 权限申请
     */
    private void requestPermission() {
        if (PermissionUtils.hasSelfPermissions(this,
                Manifest.permission.CAMERA,
                Manifest.permission.READ_EXTERNAL_STORAGE,
                Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
            CameraHelper.takePhoto(this, Constant.RESULT_CODE_CAMERA);
        } else {
            //请求权限
            PermissionUtils.requestPermissions(this,
                    new String[]{Manifest.permission.CAMERA,
                            Manifest.permission.READ_EXTERNAL_STORAGE,
                            Manifest.permission.WRITE_EXTERNAL_STORAGE},
                    Constant.REQUEST_PERMISSION_CAMERA);
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == Constant.REQUEST_PERMISSION_CAMERA) {
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                CameraHelper.takePhoto(this, Constant.RESULT_CODE_CAMERA);
            } else {
                showToast(R.string.im_uikit_camera_deny);
                finish();
            }
        }
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (resultCode == RESULT_OK && requestCode == Constant.RESULT_CODE_CAMERA) {
            //发送广播通知图片增加了
            CameraHelper.galleryAddPic(this, CameraHelper.getTakePhotoFile());
            getVu().loadImage(CameraHelper.getTakePhotoFile());
        } else {
            finish();
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (CameraManager.getInstance().getImageLoader() != null) {
            CameraManager.getInstance().getImageLoader().clearMemoryCache(this);
        }
    }

    @Override
    public void sendPhoto(PhotoItem photoItem, final boolean needOrigin) {
        Observable.just(photoItem)
                .subscribeOn(Schedulers.computation())
                .flatMap(new Func1<PhotoItem, Observable<PhotoThumbBean>>() {
                    @Override
                    public Observable<PhotoThumbBean> call(PhotoItem photoItem) {
                        photoItem.setbOriginal(needOrigin);
                        return DataTools.createImageThumb(photoItem);
                    }
                }).subscribe(new Subscriber<PhotoThumbBean>() {

            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {

            }

            @Override
            public void onNext(PhotoThumbBean photoThumbBean) {
                setResult(photoThumbBean);
            }
        });
    }

    /**
     * 返回结果
     *
     * @param thumbBean 缩略图信息
     */
    private void setResult(PhotoThumbBean thumbBean) {
        Intent intent = new Intent();
        //发送图片数据
        intent.putExtra(Constant.TAG_PHOTO, thumbBean);
        //返回结果
        setResult(Constant.RESULT_CODE_CAMERA, intent);
        finish();
    }
}
