package com.xdja.im.lib.share;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.view.View;

import com.squareup.otto.Subscribe;
import com.xdja.im.base.eventbus.BusProvider;
import com.xdja.im.common.utils.PermissionUtils;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.event.IMProxyMessageEvent;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.TalkSessionBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.ImageFileInfo;
import com.xdja.im.core.model.message.file.VideoFileInfo;
import com.xdja.im.core.proxy.IMUiKitProxyImpl;
import com.xdja.im.core.repository.interf.ProxyRepository;
import com.xdja.im.lib.album.bean.ImageItem;
import com.xdja.im.lib.album.bean.ImageThumbBean;
import com.xdja.im.lib.album.utils.DataTools;
import com.xdja.im.lib.file.bean.FileItem;
import com.xdja.im.lib.filemanage.util.HistoryFileUtils;
import com.xdja.im.lib.filemanage.util.IMFileUtils;
import com.xdja.im.lib.filemanage.util.ToolUtil;
import com.xdja.im.lib.share.adapter.ShareSessionListAdapter;
import com.xdja.im.lib.share.model.IntentBean;
import com.xdja.im.lib.share.model.ShareBean;
import com.xdja.im.lib.share.model.ShareSelectType;
import com.xdja.im.lib.share.model.WebPageInfo;
import com.xdja.im.lib.share.presenter.ShareSessionListPresenter;
import com.xdja.im.lib.share.util.ShareUtils;
import com.xdja.im.lib.share.view.ShareViewSessionList;
import com.xdja.im.lib.share.view.vu.ShareViewSessionListVu;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.base.ShareImBaseActivity;
import com.xdja.im.uikit.constans.Constant;
import com.xdja.im.uikit.utils.DataConvertUtils;
import com.xdja.im.uikit.utils.ObjectUtil;
import com.xdja.im.uikit.utils.log.LogUtil;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.inject.Inject;

import dagger.Lazy;
import rx.Observable;
import rx.Subscriber;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.Func1;
import rx.schedulers.Schedulers;

/**
 * Created by wenqi on 2018/3/1.
 * 分享
 */

public class ShareSessionListActivity extends ShareImBaseActivity<ShareSessionListPresenter, ShareViewSessionListVu> implements ShareSessionListPresenter {

    private final static String TAG = ShareSessionListActivity.class.getSimpleName();
    /**
     * IMSDK回调
     */
    @Inject
    Lazy<ProxyRepository> proxyRepository;


    private ShareSessionListAdapter adapter;

    private Intent intent;

    private BusProvider busProvider;


    /**
     * 会话数据集合
     */
    private List<TalkSessionBean> mSessionList = new ArrayList<>();

    /**
     * 会话数据类型(默认加载全部数据)
     */
    private int mTalkType = ConstDef.CHAT_TYPE_DEFAULT;

    /**
     * 转发类型
     *
     * @param savedInstanceState
     */
    private String mType;

    @SuppressLint("NewApi")
    @Override
    protected void preBindView(Bundle savedInstanceState) {
        super.preBindView(savedInstanceState);
        intent = getIntent();
        mType = IMFileUtils.filterFileType(intent);
    }

    @Override
    protected void onBindView(@Nullable Bundle savedInstanceState) {
        super.onBindView(savedInstanceState);
        if (mComponent == null) {
            return;
        }
        try {
            mComponent.inject(this);
            busProvider = BusProvider.getInstance();
            busProvider.register(this);
            adapter = new ShareSessionListAdapter(mSessionList, this);
            getVu().setAdapter(adapter);
            getVu().setToolBarVisibility(View.VISIBLE);
            getVu().setTitle("选择");
            getVu().setTitleColor(R.color.im_uiKit_share_title_color);
            getVu().setToolBarAddVisibility(View.INVISIBLE);
            //加载会话数据
            startLoadSessionData();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        this.intent = intent;
    }

    @Override
    protected void onResume() {
        super.onResume();
        requestPermissions(this);
        getVu().clearShareSearch();
    }

    public void requestPermissions(Activity context) {
        if (!PermissionUtils.hasSelfPermissions(context,
                Manifest.permission.READ_EXTERNAL_STORAGE,
                Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
            //请求权限
            PermissionUtils.requestPermissions(context,
                    new String[]{Manifest.permission.READ_EXTERNAL_STORAGE,
                            Manifest.permission.WRITE_EXTERNAL_STORAGE},
                    Constant.REQUEST_PERMISSION_STORAGE);
        }
    }

    @NonNull
    @Override
    protected Class<? extends ShareViewSessionListVu> getVuClass() {
        return ShareViewSessionList.class;
    }

    @NonNull
    @Override
    protected ShareSessionListPresenter getPresenter() {
        return this;
    }

    @Override
    public Activity getActivity() {
        return this;
    }

    /**
     * 搜索
     */
    @Override
    public void onSearchChanged(String searchKey) {
        IMUiKitProxyImpl.getInstance().callSessionSearch(getActivity(), searchKey, ShareSelectType.SHARE_SEARCH_CONTACT, intent.getAction());
    }

    /**
     * 创建新会话
     */
    @Override
    public void onCreateSessionClick() {
        IMUiKitProxyImpl.getInstance().callSessionContact(getActivity(), ShareSelectType.SHARE_NEW_CONTACT, intent.getAction());
    }

    /**
     * 选择群组
     */
    @Override
    public void onSelectGroup() {
        IMUiKitProxyImpl.getInstance().callSessionContact(getActivity(), ShareSelectType.SHARE_MORE_CONTACT, intent.getAction());
    }

    /**
     * 列表项被点击
     *
     * @param item
     */
    @Override
    public void onListItemClick(TalkSessionBean item) {
        if (item != null) {
            getVu().showDialog(intent, item);
        }
    }

    /**
     * 转发对话框确认发送
     *
     * @param list
     * @param intent
     * @param liuYan
     */
    @Override
    public void onDialogRequestForwardSend(List<ShareBean> list, final Intent intent, final String liuYan) {
        Observable.from(list)
                .filter(new Func1<ShareBean, Boolean>() {
                    @Override
                    public Boolean call(ShareBean shareBean) {
                        if (!TextUtils.isEmpty(liuYan.trim())) {
                            sendTextMessage(liuYan, shareBean.getAccount(), shareBean.getType());
                        }
                        return true;
                    }
                })
                .map(new Func1<ShareBean, Boolean>() {
                    @Override
                    public Boolean call(ShareBean shareBean) {
                        return sendForwardMessage(shareBean, intent);
                    }
                })
                .subscribe(new Subscriber<Boolean>() {
                    @Override
                    public void onCompleted() {
                        if (!intent.getAction().equals("android.intent.action.FORWARD"))
                            getVu().showSimpleDialog();
                        else
                            getVu().showFwordSimpleDialog();

                        IMProxyMessageEvent.RefreshMessageListEvent event =
                                new IMProxyMessageEvent.RefreshMessageListEvent();
                        event.setNeedClearInput(true);
                        busProvider.post(event);
                    }

                    @Override
                    public void onError(Throwable e) {
                        e.printStackTrace();
                    }

                    @Override
                    public void onNext(Boolean b) {
                        LogUtil.d("转发完成");
                    }
                });

    }

    private boolean sendForwardMessage(ShareBean shareBean, Intent intent) {
        ArrayList<FileInfo> fileList = (ArrayList<FileInfo>) intent.getSerializableExtra(ConstDef.TAG_SELECTFILE);
        if (ObjectUtil.collectionIsEmpty(fileList)) {
            int msgType = intent.getIntExtra(ConstDef.FORWARD_MSG_TYPE, 0);
            switch (msgType) {
                case ConstDef.MSG_TYPE_TEXT:
                case ConstDef.MSG_TYPE_AT_TEXT:
                    String content = intent.getExtras().getString(Intent.EXTRA_TEXT);
                    if (!TextUtils.isEmpty(content))
                        sendTextMessage(content, shareBean.getAccount(), shareBean.getType());
                    break;
                case ConstDef.MSG_TYPE_ASSISTANT:
                    String assistant_content = intent.getExtras().getString(Intent.EXTRA_TEXT);
                    //sendAssistantMessage(assistant_content, shareBean.getAccount(), shareBean.getType());
                    break;
                case ConstDef.MSG_TYPE_PHOTO:
                    ImageFileInfo imageFileInfo = (ImageFileInfo) intent.getExtras().getSerializable(ConstDef.FORWARD_FILE);
                    List<FileInfo> fileInfos = new ArrayList<>();
                    if (!imageFileInfo.isOriginal()) {
                        imageFileInfo.setFileType(ConstDef.FILE_TYPE_PIC);
                    }
                    fileInfos.add(imageFileInfo);
                    sendFileMessage(fileInfos, shareBean.getAccount(), shareBean.getType());
                    break;
                case ConstDef.MSG_TYPE_VIDEO:
                case ConstDef.MSG_TYPE_FILE:
                    FileInfo fileInfo2 = (FileInfo) intent.getExtras().getSerializable(ConstDef.FORWARD_FILE);
                    List<FileInfo> fileInfos2 = new ArrayList<>();
                    fileInfo2.setTranslateSize(0l);
                    fileInfo2.setTranslatePercent(0);
                    fileInfos2.add(fileInfo2);
                    sendFileMessage(fileInfos2, shareBean.getAccount(), shareBean.getType());
                    break;
                case ConstDef.MSG_TYPE_WEB:
                    WebPageInfo webInfo = (WebPageInfo) intent.getSerializableExtra(ConstDef.FORWARD_WEB);
                    if (webInfo != null) {
                        sendWebMessage(webInfo, shareBean.getAccount(), shareBean.getType());
                    }
                    break;
            }
        } else {
            List<FileInfo> results = new ArrayList<>();
            for (FileInfo fInfo : fileList) {
                if (fInfo instanceof ImageFileInfo) {
                    if (((ImageFileInfo) fInfo).getHdFileInfo() == null) {//图片文件转发没有高清缩略图
                        boolean isExist = HistoryFileUtils.isFileExist(((ImageFileInfo) fInfo).getRawFileInfo().getFilePath());
                        if (!isExist) {
                            FileInfo fileInfo = new FileInfo();
                            fileInfo.setFileName(fInfo.getFileName());
                            fileInfo.setFileSize(fInfo.getFileSize());
                            fileInfo.setFilePath(fInfo.getFilePath());
                            fileInfo.setTranslateSize(fInfo.getTranslateSize());
                            ((ImageFileInfo) fInfo).setRawFileInfo(fileInfo);
                        }

                        results.add(fInfo);
                    } else {
                        //从聊天文件中转发图片，可能存在高清缩略图不存在
                        boolean isExist = HistoryFileUtils.isFileExist(((ImageFileInfo) fInfo).getHdFileInfo().getFilePath());
                        if (!isExist) {
                            FileInfo fileInfo = new FileInfo();
                            fileInfo.setFileName(fInfo.getFileName());
                            fileInfo.setFileSize(fInfo.getFileSize());
                            fileInfo.setFilePath(fInfo.getFilePath());
                            fileInfo.setTranslateSize(fInfo.getTranslateSize());
                            ((ImageFileInfo) fInfo).setHdFileInfo(fileInfo);

                        }
                        //如果聊天文件中待转发的图片，原图不存在
                        if (((ImageFileInfo) fInfo).isOriginal()) {
                            boolean isRawExist = HistoryFileUtils.isFileExist(((ImageFileInfo) fInfo).getRawFileInfo().getFilePath());
                            if (!isRawExist) {
                                ((ImageFileInfo) fInfo).setOriginal(false);
                                fInfo.setFileType(ConstDef.FILE_TYPE_PIC);
                            }
                        } else {
                            ((ImageFileInfo) fInfo).setOriginal(false);
                            fInfo.setFileType(ConstDef.FILE_TYPE_PIC);
                        }
                        results.add(fInfo);
                    }

                } else if (fInfo instanceof VideoFileInfo) {
                    results.add(fInfo);
                } else {
                    results.add(resetFileInfo(fInfo));
                }
            }
            sendFileMessage(results, shareBean.getAccount(), shareBean.getType());
        }
        return true;
    }


    private FileInfo resetFileInfo(FileInfo oldFileInfo) {
        FileInfo fileInfo = new FileInfo();
        fileInfo.setFilePath(oldFileInfo.getFilePath());
        fileInfo.setFileName(oldFileInfo.getFileName());
        fileInfo.setFileSize(oldFileInfo.getFileSize());
        fileInfo.setSuffix(oldFileInfo.getSuffix());
        fileInfo.setFileType(oldFileInfo.getFileType());
        return fileInfo;
    }


    /**
     * 分享对话框确认发送
     *
     * @param shareBeanList
     * @param intentBean
     * @param dialogContent
     */
    @Override
    public void onDialogRequestSend(final List<ShareBean> shareBeanList, final IntentBean intentBean, final String dialogContent) {

        Observable.from(shareBeanList)
                .filter(new Func1<ShareBean, Boolean>() {
                    @Override
                    public Boolean call(ShareBean shareBean) {
                        if (!TextUtils.isEmpty(intentBean.getContent()) && mType != ConstDef.WEB_SHARE_TYPE) {
                            sendTextMessage(intentBean.getContent(), shareBean.getAccount(), shareBean.getType());
                        }
                        return true;
                    }
                })
                .filter(new Func1<ShareBean, Boolean>() {
                    @Override
                    public Boolean call(ShareBean shareBean) {
                        if (!TextUtils.isEmpty(intentBean.getContent()) && mType == ConstDef.WEB_SHARE_TYPE) {
                            //对第三方浏览分享数据进行处理
                            WebPageInfo webPageInfo = ShareUtils.getShareWebInfo(getContext(), intent, intentBean);
                            sendWebMessage(webPageInfo, shareBean.getAccount(), shareBean.getType());
                        }
                        return true;
                    }
                })
                .filter(new Func1<ShareBean, Boolean>() {
                    @Override
                    public Boolean call(ShareBean shareBean) {
                        if (!TextUtils.isEmpty(dialogContent.trim())) {
                            sendTextMessage(dialogContent, shareBean.getAccount(), shareBean.getType());
                        }
                        return true;
                    }
                })
                .toList()
                .map(new Func1<List<ShareBean>, ShareBean>() {
                    @Override
                    public ShareBean call(List<ShareBean> shareBeans) {
                        if (intentBean.getUriArrayList() != null && intentBean.getUriArrayList().size() > 0)
                            sendIntentFile(shareBeans, intentBean);
                        return null;
                    }
                })
                .subscribe(new Subscriber<ShareBean>() {
                    @Override
                    public void onCompleted() {
                        if (!intentBean.getAction().equals("android.intent.action.FORWARD"))
                            getVu().showSimpleDialog();
                        else getVu().showFwordSimpleDialog();

                        IMProxyMessageEvent.RefreshMessageListEvent event =
                                new IMProxyMessageEvent.RefreshMessageListEvent();
                        event.setNeedClearInput(true);
                        busProvider.post(event);
                    }

                    @Override
                    public void onError(Throwable e) {
                        e.printStackTrace();
                    }

                    @Override
                    public void onNext(ShareBean shareBean) {
                        LogUtil.d("分享成功");
                    }
                });
    }

    private void sendIntentFile(final List<ShareBean> shareBeans, final IntentBean intentBean) {

        Observable.from(intentBean.getUriArrayList())
                .subscribeOn(Schedulers.io())
                .flatMap(new Func1<String, Observable<?>>() {
                    @Override
                    public Observable<?> call(String s) {
                        try {
                            if (!TextUtils.isEmpty(intentBean.getType()) && intentBean.getType().startsWith(Constant.IMAGE_SHARE_TYPE)) {
                                String path = IMFileUtils.getImagePath(getContext(), Uri.parse(Uri.decode(s)).toString());
                                if (!TextUtils.isEmpty(path)) {
                                    FileItem fileItem = IMFileUtils.getFileItem(path);
                                    ImageItem imageItem = new ImageItem(fileItem.getFileName(), path, fileItem.getFileSize());
                                    imageItem.setOriginalState(false);
                                    //ImageFileInfo imageFileInfo = new ImageFileInfo();
                                    //imageFileInfo.setFilePath(path);
                                    //return ModelGenerator.createContentImageFile(path, ImUiKit.getInstance().getComponent().accountCache().get().getAccount());
                                    return DataTools.createThumbNails(imageItem);
                                }
                            } else {
                                FileItem info = IMFileUtils.queryLocalFiles(getContext(), Uri.parse(Uri.decode(s)));
                                FileInfo fileInfo = new FileInfo();
                                fileInfo.setFilePath(info.getFilePath());
                                fileInfo.setFileName(info.getFileName());
                                fileInfo.setFileSize(info.getFileSize());
                                String name = info.getFileName();
                                String suffix = ToolUtil.getLastString(name, ".");
                                if (!TextUtils.isEmpty(suffix)) {
                                    fileInfo.setSuffix(suffix);
                                }
                                fileInfo.setFileType(info.getFileType());
                                boolean isVideo = IMFileUtils.isVideoFileType(name);
                                boolean isImage = IMFileUtils.isImageFileType(name);

                                if (isImage) {
                                    fileInfo = DataConvertUtils.mapImagePicFileInfo(fileInfo);

                                }
                                if (isVideo) {
                                    fileInfo = DataConvertUtils.mapVideoFileInfo(fileInfo);
                                    if (fileInfo != null) {
                                        fileInfo.setFileType(ConstDef.FILE_TYPE_VIDEO);
                                    }
                                }
                                return Observable.just(fileInfo);
                            }
                        } catch (Exception e) {
                            return Observable.error(e);
                        }

                        return Observable.just(null);
                    }
                })
                .toList()
                .subscribeOn(AndroidSchedulers.mainThread())
                .subscribe(new Subscriber<List<Object>>() {
                    @Override
                    public void onCompleted() {

                    }

                    @Override
                    public void onError(Throwable e) {
                        e.printStackTrace();
                    }

                    @Override
                    public void onNext(List<Object> fileInfos) {
                        if (fileInfos != null) {
                            List<FileInfo> files = new ArrayList<>();
                            for (Object file : fileInfos) {
                                if (file != null) {
                                    if (file instanceof FileInfo) {
                                        files.add((FileInfo) file);
                                    } else {
                                        files.add(DataConvertUtils.mapImageFileInfo((ImageThumbBean) file));
                                    }
                                }
                            }
                            if (files != null) {
                                for (ShareBean shareBean : shareBeans) {
                                    sendFileMessage(files, shareBean.getAccount(), shareBean.getType());
                                }
                            }
                        }
                    }
                });
    }


    /**
     * 对话框取消发送
     */
    @Override
    public void onDialogCancleSend() {
        //showToast("取消发送");
    }

    /**
     * 对话框分享完成确认
     */
    @Override
    public void onTipsDialogTrueClick() {
        IMUiKitProxyImpl.getInstance().shareOverCallback(getActivity());
        finish();
    }

    /**
     * 对话框分享完成取消
     */
    @Override
    public void onTipsDialogClick() {
        finish();
    }

    /**
     * 发送文本消息
     *
     * @param message 文本消息内容
     * @return {@code true} 消息可发送，并且发送的动作操作成功 <br />
     * {@code false} 消息不可发送，或者发送的动作操作失败
     */
    @Override
    public boolean sendTextMessage(String message, String sendTo, int type) {

        if (proxyRepository == null || proxyRepository.get() == null) {
            return false;
        }
        proxyRepository.get().sendTextMessage(message, sendTo, type, false, type == ConstDef.CHAT_TYPE_P2G, new Subscriber<TalkMessageBean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onNext(TalkMessageBean talkMessageBean) {
                //showToast("发送成功");
            }
        });

        return false;
    }

    /**
     * 发送文件消息
     *
     * @param fileList 文件列表
     * @return {@code true} 消息可发送，并且发送的动作操作成功 <br />
     * {@code false} 消息不可发送，或者发送的动作操作失败
     */
    @Override
    public void sendFileMessage(List<FileInfo> fileList, String sendTo, int type) {

        if (proxyRepository == null || proxyRepository.get() == null) {
            return;
        }
        proxyRepository.get().sendFileMessage(sendTo, type, false, type == ConstDef.CHAT_TYPE_P2G, fileList, new Subscriber<TalkMessageBean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onNext(TalkMessageBean talkMessageBean) {
                //showToast("文件发送成功");
            }
        });
    }


    /**
     * 加载会话列表数据
     */
    private void startLoadSessionData() {
        //加载动画
        getVu().setProgressBarVisibility(View.VISIBLE);
        if (proxyRepository == null || proxyRepository.get() == null) {
            return;
        }
        proxyRepository.get().getSessionListBeans("", 0, mTalkType, new Subscriber<List<TalkSessionBean>>() {
            @Override
            public void onCompleted() {
            }

            @Override
            public void onError(Throwable e) {
                getVu().setProgressBarVisibility(View.GONE);
                LogUtil.e("ERROR: Load session data error. " + e.getMessage());
            }

            @Override
            public void onNext(List<TalkSessionBean> sessionBeanList) {
                LogUtil.d("Load session data finished.");
                if (sessionBeanList != null && sessionBeanList.size() > 0) {
                    mSessionList.clear();
                    //剔除重复的会话
                    removeExists(sessionBeanList);

                    for (int i = 0; i < sessionBeanList.size(); i++) {
                        if (sessionBeanList.get(i).getTalkType() != ConstDef.CHAT_TYPE_SYSTEM_INFO
                                || sessionBeanList.get(i).getTalkerAccount().equals(ConstDef.FILE_TRANSFER_ASS_ACCOUNT)) {
                            mSessionList.add(sessionBeanList.get(i));
                        }
                    }
//                    mSessionList.addAll(sessionBeanList);
                    adapter.notifyDataSetChanged();
                    getVu().setShareSessionEmptyVisibility(View.GONE);
                } else {
                    getVu().setShareSessionEmptyVisibility(View.VISIBLE);
                }
                getVu().setProgressBarVisibility(View.GONE);
            }
        });
    }

    @Override
    protected void onDestroy() {
        if (busProvider != null) {
            busProvider.unregister(this);
        }
        super.onDestroy();
    }

    /**
     * 移除重复消息
     *
     * @param beans
     */
    private void removeExists(List<TalkSessionBean> beans) {
        Iterator<TalkSessionBean> iterator = beans.iterator();
        for (; iterator.hasNext(); ) {
            TalkSessionBean bean = iterator.next();

            int index = beans.indexOf(bean);
            if (index > -1 && bean != beans.get(index)) {
                iterator.remove();
            }
        }
    }


    @Subscribe
    public void onShareContactRefreshEvent(IMProxyMessageEvent.ShareContactRefreshEvent event) {
        int type = event.getType();
        List<ShareBean> list = event.getList();
        switch (type) {
            case ShareSelectType.SHARE_NEW_CONTACT:
            case ShareSelectType.SHARE_MORE_CONTACT:

                break;
            case ShareSelectType.SHARE_SEARCH_CONTACT:

                break;
        }
        getVu().showMultiContactDialog(intent, list);

    }

    /**
     * 发送Web消息
     *
     * @param sendTo
     * @param type
     */
    @Override
    public boolean sendWebMessage(WebPageInfo webPageInfo, String sendTo, int type) {

        if (proxyRepository == null || proxyRepository.get() == null) {
            return false;
        }
        proxyRepository.get().sendWebMessage(sendTo, false, type == ConstDef.CHAT_TYPE_P2G, webPageInfo, new Subscriber<TalkMessageBean>() {
            @Override
            public void onCompleted() {

            }

            @Override
            public void onError(Throwable e) {
                e.printStackTrace();
            }

            @Override
            public void onNext(TalkMessageBean bean) {
                showToast("文件发送成功");
            }
        });
        return false;
    }

}
