package com.xdja.im.uikit.ui.view;

import android.annotation.SuppressLint;
import android.graphics.Bitmap;
import android.os.Handler;
import android.os.Message;
import androidx.annotation.NonNull;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.bumptech.glide.Glide;
import com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.xdja.im.common.utils.CommonTool;
import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.file.FileInfo;
import com.xdja.im.core.model.message.file.VideoFileInfo;
import com.xdja.im.core.model.screen.ScreenInfo;
import com.xdja.im.core.utils.FilePathUtils;
import com.xdja.im.lib.filemanage.util.DateUtils;
import com.xdja.im.lib.filemanage.util.HistoryFileUtils;
import com.xdja.im.lib.filemanage.util.XToast;
import com.xdja.im.lib.video.palyer.IVideoPlayCallback;
import com.xdja.im.lib.video.palyer.VideoPlayState;
import com.xdja.im.lib.video.utils.DisplayUtils;
import com.xdja.im.lib.video.widget.VideoPlayView;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.action.options.ImOption;
import com.xdja.im.uikit.ui.dialog.BaseRecyclerListDialog;
import com.xdja.im.uikit.utils.file.FileUtils;
import com.xdja.im.uikit.utils.image.BitmapUtils;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.im.uikit.utils.sys.ScreenUtil;

import java.io.File;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;

/**
 * @Package: com.xdja.im.uikit.ui.view
 * @Author: xdjaxa
 * @Creation: 2017-05-15 11:36
 * @Version V1.0
 * @Description:视频预览控件
 */
public class ViewPreviewVideo extends BaseFilePreview implements IVideoPlayCallback {

    private RelativeLayout mLayout;
    /**
     * 图片加载控件
     */
    private ImageView mPhotoView;
    /**
     * 播放按钮
     */
    private Button mPlayBtn;
    /**
     * 视频文件加载进度
     */
    private ProgressBar mLoadingPbar;
    /**
     * 视频播放控件
     */
    private VideoPlayView mVideoPlayView;
    /**
     * 屏幕信息
     */
    private ScreenInfo mScreenInfo;
    /**
     * 视频关闭按钮
     */
    private TextView mCloseTv;

    /**
     * 选项框
     */
    private BaseRecyclerListDialog mDialog;

    /**
     * 创建handler
     * @return
     */
    private Handler handler = new myHandler(this);

    @Override
    protected int getLayoutRes() {
        return R.layout.im_uikit_item_preview_video;
    }

    @SuppressLint("NewApi")
    @Override
    protected void injectView() {
        super.injectView();
        mLayout = (RelativeLayout) findViewById(R.id.layout_video_container);
        mPhotoView = (ImageView) findViewById(R.id.view_photo);
        mPlayBtn = (Button) findViewById(R.id.btn_play);
        mLoadingPbar = (ProgressBar) findViewById(R.id.pbar_video_loading);
        mVideoPlayView = (VideoPlayView) findViewById(R.id.view_video_player);
        mCloseTv = (TextView) findViewById(R.id.btn_close_player);
        mVideoPlayView.setVideoPlayCallback(this);
        mVideoPlayView.setVisibility(View.VISIBLE);
        mDialog = new BaseRecyclerListDialog(getContext()) {
            @Override
            public List<ImOption> getOptions() {
                return buildImOption();
            }
        };


        mLayout.setOnLongClickListener(new View.OnLongClickListener() {
            @Override
            public boolean onLongClick(View v) {
                VideoFileInfo videoFileInfo = (VideoFileInfo) dataSource.getFileInfo();
                if (videoFileInfo == null) {
                    return false;
                }
                //如果视频文件未下载，则不支持保存
                FileInfo rawFileInfo = videoFileInfo.getRawFileInfo();
                //此处保存，进行文件完整性判断
                if (rawFileInfo == null || !FileUtils.getFileExistAndComplete(rawFileInfo.getFilePath(), rawFileInfo.getFileState())) {
                    return false;
                }
                mDialog.show();
                return false;
            }
        });

        mCloseTv.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                getActivity().finish();
            }
        });

        mPlayBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (!HistoryFileUtils.isFileExist(dataSource) && DateUtils.isOverdue(dataSource.getShowTime())) {
                    new XToast(getActivity()).display(R.string.im_uikit_history_select_all_overdue);
                    return;
                }
                VideoFileInfo videoFileInfo = (VideoFileInfo) dataSource.getFileInfo();
                FileInfo rawFile = videoFileInfo.getRawFileInfo();

                if (HistoryFileUtils.isFileExist(dataSource)) {
                    startPlay(rawFile.getFilePath());
                } else {
                    //视频文件不存在：如果是自己发送的，或者文件状态为失败
                    if (videoFileInfo.getFileState() == ConstDef.STATE_SEND_FAILED) {
                        loadError();
                        CommonTool.showToast(R.string.im_uikit_video_not_exist);
                        return;
                    }

                    videoFileInfo.setType(ConstDef.IS_RAW);
                    videoFileInfo.setFileState(ConstDef.LOADING);
                    dataSource.setFileInfo(videoFileInfo);
                    loadVideo(videoFileInfo);
                }
            }
        });

        mDialog.setOnItemClickListener(new BaseRecyclerListDialog.onItemClickListener() {
            @Override
            public void onItemClick(View view, ImOption option, int itemPosition) {
                switch (option.getOptionType()) {
                    case 0:  //保存视频
                        saveVideo();
                        break;
                    case 1: //删除视频
                        deleteVideo();
                        break;
                }
            }
        });

        //焦点监听
        getView().getViewTreeObserver().addOnWindowFocusChangeListener(
                new ViewTreeObserver.OnWindowFocusChangeListener() {
                    @Override
                    public void onWindowFocusChanged(boolean hasFocus) {
                        if (!hasFocus) {
                            if (mVideoPlayView != null && mVideoPlayView.isPlaying()) {
                                mVideoPlayView.pause();
                            }
                        } else {
                            if (mVideoPlayView != null && mVideoPlayView.getVideoPlayState() ==
                                    VideoPlayState.PAUSE) {
                                mVideoPlayView.resume();
                            }
                        }
                    }
                });

        if (mVideoPlayView.getmHelper() != null) {
            mVideoPlayView.getmHelper().setmPlayCallback(this);
        }

        mScreenInfo = ScreenUtil.getScreenInfo(getActivity());
    }

    private static class myHandler extends Handler {
        private WeakReference<ViewPreviewVideo> context;
        private myHandler(ViewPreviewVideo viewPreviewVideo) {
            context = new WeakReference<>(viewPreviewVideo);
        }
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            if (context.get() != null) {
                context.get().mPlayBtn.setVisibility(View.VISIBLE);
                context.get().mCloseTv.setVisibility(View.VISIBLE);
            }
        }
    };

    @Override
    public void bindDataSource(int position, @NonNull final TalkMessageBean dataSource) {
        super.bindDataSource(position, dataSource);
        VideoFileInfo videoFileInfo = (VideoFileInfo) dataSource.getFileInfo();
        if (videoFileInfo == null) {
            loadError();
            return;
        }

        if (!loadImage(videoFileInfo.getFilePath())) {
            LogUtil.w("ERROR: Start preview video, but thumb is failed.");
            return;
        }

        if (!dataSource.isMine() && dataSource.getFileInfo().getFileState() == ConstDef.LOADING) {
            mLoadingPbar.setVisibility(View.VISIBLE);
            return;
        }

        mVideoPlayView.setDuration(videoFileInfo.getAmountOfTime());
        if (handler != null) {
            handler.sendEmptyMessageDelayed(0, 500);
        }

        FileInfo rawFileInfo = videoFileInfo.getRawFileInfo();
        if (rawFileInfo == null) {
            LogUtil.e("ERROR: Video raw file info is null.");
            return;

        }

        resetVideoPlayViewParam(dataSource.getFileInfo().getFilePath());
    }

    @Override
    public void onViewDestroy() {
        super.onViewDestroy();
        stopPlay();
        if (handler != null) {
            handler.removeCallbacksAndMessages(null);
        }
    }

    @Override
    public void onViewPause() {
        super.onViewPause();
        stopPlay();
        mPhotoView.setImageBitmap(null);
    }

    private List<ImOption> buildImOption() {
        List<ImOption> data = new ArrayList<>();
        data.add(new ImOption(0, getStringRes(R.string.im_uikit_video_option_save)));
        data.add(new ImOption(1, getStringRes(R.string.im_uikit_video_option_delete)));
        return data;
    }

    /**
     * 加载失败显示图片
     */
    private void loadError() {
        mPhotoView.setImageResource(R.mipmap.im_uikit_ic_video_failed);
        mPhotoView.setScaleType(ImageView.ScaleType.CENTER);

        ViewGroup.LayoutParams params = mPhotoView.getLayoutParams();
        params.width = ViewGroup.LayoutParams.WRAP_CONTENT;
        params.width = ViewGroup.LayoutParams.WRAP_CONTENT;

        mPlayBtn.setVisibility(View.GONE);
        mVideoPlayView.setVisibility(View.GONE);
        mLoadingPbar.setVisibility(View.GONE);
    }

    /**
     * 加载本地图片
     *
     * @param filePath
     */
    private boolean loadImage(String filePath) {
        if (FileUtils.isFileExists(filePath)) {
            Glide.with(getActivity())
                    .load(filePath)
                    .diskCacheStrategy(DiskCacheStrategy.NONE)
                    .error(R.mipmap.im_uikit_pic_default)
                    .into(mPhotoView);
            return true;
        } else {
            loadError();
            return false;
        }
    }

    private void loadVideo(VideoFileInfo rawFileInfo) {
        mPlayBtn.setVisibility(View.GONE);
        mLoadingPbar.setVisibility(View.VISIBLE);
        getPresenter().download(rawFileInfo);
    }

    /**
     * 开始播放视频
     *
     * @param filePath 视频文件路径
     */
    private void startPlay(String filePath) {
        mLoadingPbar.setVisibility(View.GONE);
        mPlayBtn.setVisibility(View.GONE);
       // mPhotoView.setVisibility(View.GONE);
        mVideoPlayView.play(filePath);

    }

    /**
     * 停止播放
     */
    private void stopPlay() {
        mPhotoView.setVisibility(View.VISIBLE);
        mPlayBtn.setVisibility(View.VISIBLE);

    }


    /**
     * 保存视频文件
     */
    private void saveVideo() {
        VideoFileInfo videoFileInfo = (VideoFileInfo) dataSource.getFileInfo();
        if (videoFileInfo == null) {
            return;
        }

        FileInfo rawFileInfo = videoFileInfo.getRawFileInfo();
        if (rawFileInfo == null) {
            return;
        }
        File srcFile = new File(rawFileInfo.getFilePath());
        if (srcFile.exists() && srcFile.length() == rawFileInfo.getFileSize()) {
            String fileName = "videoexport" + System.currentTimeMillis() + ".mp4";
            getPresenter().saveFile(srcFile, FilePathUtils.getVideoRecPath(), fileName);
        } else {
            LogUtil.e("ERROR: Save file failed.");
            CommonTool.showToast(R.string.im_uikit_hint_save_failed);
        }
        if (mDialog != null) {
            mDialog.dismiss();
        }
    }

    /**
     * 删除视频文件
     */
    private void deleteVideo() {

        getPresenter().deleteMsg(dataSource);

        if (mDialog != null) {
            mDialog.dismiss();
        }
    }

    @Override
    public void prepareListener() {

    }

    @Override
    public void progressListener(int pro) {

    }

    @Override
    public void completeListener() {
        mPhotoView.setVisibility(View.VISIBLE);
        mPlayBtn.setVisibility(View.VISIBLE);
    }

    @Override
    public void pauseListener(int currentPos) {

    }

    @Override
    public void resumeListener(int currentPos) {

    }

    @Override
    public void surfacePrepareCallback() {
        if (mPhotoView.getVisibility() == View.VISIBLE) {
            mPhotoView.setVisibility(View.GONE);
        }

    }

    @Override
    public void surfaceDestroyListener(Bitmap currentPosBitmap) {

    }

    @Override
    public void onPageSelected(int lastPos, int curPos) {
        mPhotoView.setImageBitmap(null);
        mPhotoView.setVisibility(View.VISIBLE);
        if (dataSource.getFileInfo().getFileState() != ConstDef.LOADING) {
            mPlayBtn.setVisibility(View.VISIBLE);
        }

        mVideoPlayView.stop();
    }

    @Override
    public void onFileDownloadFinish() {
        if (dataSource == null || dataSource.getFileInfo() == null) {
            LogUtil.e("Illegal video file info.");
            return;
        }

        VideoFileInfo videoFileInfo = (VideoFileInfo) dataSource.getFileInfo();
        FileInfo rawFileInfo = videoFileInfo.getRawFileInfo();
        //判断文件存在，判断准确
        if (rawFileInfo != null && FileUtils.getFileExistAndComplete(rawFileInfo.getFilePath(), rawFileInfo.getFileState())) {
            videoFileInfo.setFileState(ConstDef.DONE);
            dataSource.setFileInfo(videoFileInfo);
            startPlay(rawFileInfo.getFilePath());
        }
    }

    public void resetVideoPlayViewParam(String path) {
        View v = mVideoPlayView.findViewById(R.id.view_surface);
        RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) v.
                getLayoutParams();
        int displayWidth = DisplayUtils.getWidthPixels(getActivity());
        float scale = BitmapUtils.getPictureSacle(path);
        if (scale < 1) {
            mPhotoView.setScaleType(ImageView.ScaleType.FIT_CENTER);
            layoutParams.height = (int) (scale * displayWidth);
            layoutParams.width = displayWidth;
            layoutParams.addRule(RelativeLayout.CENTER_IN_PARENT);
            v.setLayoutParams(layoutParams);
            mVideoPlayView.setGravity(Gravity.CENTER_VERTICAL);
        }
    }
}
