package com.xdja.im.uikit.ui.view;

import android.graphics.drawable.AnimationDrawable;
import android.os.Handler;
import androidx.annotation.NonNull;
import android.view.View;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.xdja.im.core.config.ConstDef;
import com.xdja.im.core.model.message.TalkMessageBean;
import com.xdja.im.core.model.message.file.VoiceFileInfo;
import com.xdja.im.core.model.screen.ScreenInfo;
import com.xdja.im.uikit.R;
import com.xdja.im.uikit.utils.log.LogUtil;
import com.xdja.im.uikit.utils.sys.ScreenUtil;

/**
 * @Package: com.xdja.im.uikit.ui
 * @Author: xdjaxa
 * @Creation: 2017-04-20 14:42
 * @Version V1.0
 * @Description:
 */
public class ViewRecvVoiceItem extends BaseReceiveItem {

    /**
     * 语音信息气泡长度DP值
     */
    private static final int MIN_DP = 80;
    private static final int FIRST_STEP_DP = 140;
    private static final int SECOND_STEP_DP = 180;
    private static final int MAX_DP = 200;

    /**
     * 语音时长提示
     */
    private TextView mVoiceLength;
    /**
     * 语音图标
     */
    private ImageView mVoiceAnimImg;
    /**
     * 读取标识
     */
    private ImageView mNewFlagImg;

    /**
     * 屏幕信息
     */
    private ScreenInfo mScreenInfo;

    private TextView mDestroyView;

    private RelativeLayout mVoiceLayout;

    @Override
    protected int getLayoutRes() {
        return R.layout.im_uikit_item_chat_receive_voice;
    }

    @Override
    protected void injectView() {
        super.injectView();
        mVoiceLength = (TextView) findViewById(R.id.tv_hint_time);
        mVoiceAnimImg = (ImageView) findViewById(R.id.img_anim_voice);
        mNewFlagImg = (ImageView) findViewById(R.id.img_new_flag);
        mDestroyView = (TextView) findViewById(R.id.voice_receive_destroy);
        mVoiceLayout = (RelativeLayout) findViewById(R.id.layout_voice);

        mContentLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                getPresenter().playVoice(dataSource);
            }
        });
    }

    @Override
    public void bindDataSource(int position, @NonNull final TalkMessageBean dataSource) {
        super.bindDataSource(position, dataSource);

        mScreenInfo = ScreenUtil.getScreenInfo(getActivity());

        initView();
    }

    private void initView() {
        if (dataSource == null) {

            LogUtil.e("ERROR: voice dataSource is null.");
            return;
        }

        if (dataSource.getMessageState() == ConstDef.STATE_DESTROY) {
            setMessageDestroy(true);
            mVoiceLayout.setVisibility(View.GONE);
            mDestroyView.setVisibility(View.VISIBLE);
            setDestroyView(mDestroyView, true, getStringRes(R.string.im_uikit_voice_destory_text));
            setClickListener(null);
            setNewMessageFlagVisible(false);
        } else {
            mVoiceLayout.setVisibility(View.VISIBLE);
            mDestroyView.setVisibility(View.GONE);
            setMessageDestroy(false);
            setDestroyView(mDestroyView, false, "");

            VoiceFileInfo fileInfo = (VoiceFileInfo) dataSource.getFileInfo();
            if (fileInfo == null) {
                LogUtil.e("ERROR: Voice file is null.");
                return;
            }

            //语音显示内容
            int recordingTime = fileInfo.getAmountOfTime();
            setBackground(recordingTime);
            setVoiceLengthText(recordingTime);

            //未读标识
            if (dataSource.getMessageState() < ConstDef.STATE_READED) {
                setNewMessageFlagVisible(true);
            } else {
                //清除已读标识
                setNewMessageFlagVisible(false);
            }

            //开始执行销毁动画
            if (dataSource.getMessageState() == ConstDef.STATE_DESTROYING) {
                startBombAnim();
                new Handler().postDelayed(new Runnable() {

                    @Override
                    public void run() {
                        getPresenter().postDestroyAnimate(dataSource);
                    }
                }, 700);
            }
        }
        if (getPresenter().getVoiceMessageIsPlaying(dataSource.getFileInfo())) {
            startVoiceAnim();
            setNewMessageFlagVisible(false);
        } else {
            stopVoiceAnim();
        }
    }

    /**
     * 语音背景框，计算语音显示的长度
     *
     * @param recordingTime 录音文件时长
     */
    private void setBackground(int recordingTime) {
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) mContentLayout.getLayoutParams();
        float bubbleLength;//语音气泡长度
        if (recordingTime <= 10) {
            bubbleLength = (MIN_DP + (FIRST_STEP_DP - MIN_DP) / 10 *
                    recordingTime) * mScreenInfo.getDensity();
        } else if (recordingTime <= 20) {
            bubbleLength = (FIRST_STEP_DP + (SECOND_STEP_DP - FIRST_STEP_DP) / 10 *
                    (recordingTime - 10)) * mScreenInfo.getDensity();
        } else if (recordingTime <= 30) {
            bubbleLength = (SECOND_STEP_DP + (MAX_DP - SECOND_STEP_DP) / 10 *
                    (recordingTime - 20)) * mScreenInfo.getDensity();
        } else {
            bubbleLength = MAX_DP * mScreenInfo.getDensity();
        }
        params.width = (int) bubbleLength;
        mContentLayout.setLayoutParams(params);
    }

    /**
     * 设置是否是新消息
     *
     * @param isVisible
     */
    private void setNewMessageFlagVisible(boolean isVisible) {
        if (mNewFlagImg != null) {
            if (isVisible) {
                mNewFlagImg.setVisibility(View.VISIBLE);
            } else {
                mNewFlagImg.setVisibility(View.GONE);
            }
        }
    }

    /**
     * 设置语音时长
     *
     * @param length 语音时长，单位s
     */
    private void setVoiceLengthText(int length) {
        mVoiceLength.setText((length == 0 ? 1 : length) + "\"");
    }

    /**
     * 开始语音播放动画
     */
    private void startVoiceAnim() {
        if (mVoiceAnimImg != null) {
            AnimationDrawable voiceAnim = (AnimationDrawable) mVoiceAnimImg.getBackground();
            if (voiceAnim != null) {
                if (voiceAnim.isRunning()) {
                    voiceAnim.stop();
                }
                voiceAnim.start();
            }
        }
    }

    /**
     * 停止语音播放动画
     */
    private void stopVoiceAnim() {
        if (mVoiceAnimImg != null) {
            AnimationDrawable voiceAnim = (AnimationDrawable) mVoiceAnimImg.getBackground();
            if (voiceAnim != null) {
                voiceAnim.stop();
                voiceAnim.selectDrawable(0);
            }
        }
    }

    /**
     * 设置点击事件
     *
     * @param listener
     */
    private void setClickListener(View.OnClickListener listener) {
        if (mContentLayout != null) {
            mContentLayout.setOnClickListener(listener);
        }
    }
}
