package com.xdja.jce.crypto.soft.engine.keypair;

import com.xdja.jce.base.cipher.asymmetric.ECUtils;
import com.xdja.jce.base.cipher.asymmetric.KeyPairGeneratorSpi;
import com.xdja.jce.base.cipher.asymmetric.util.ECUtil;
import com.xdja.jce.base.common.CryptoServicesRegistrar;
import com.xdja.jce.base.common.EC5Util;
import com.xdja.jce.base.config.ProviderConfiguration;
import com.xdja.jce.base.ec.BCECPrivateKey;
import com.xdja.jce.base.ec.BCECPublicKey;
import com.xdja.jce.base.ec.ECCurve;
import com.xdja.jce.base.ec.ECPoint;
import com.xdja.jce.base.generators.ECKeyPairGenerator;
import com.xdja.jce.base.key.AsymmetricCipherKeyPair;
import com.xdja.jce.base.params.ECDomainParameters;
import com.xdja.jce.base.params.ECKeyGenerationParameters;
import com.xdja.jce.base.params.ECPrivateKeyParameters;
import com.xdja.jce.base.params.ECPublicKeyParameters;
import com.xdja.jce.base.provider.XdjaProvider;
import com.xdja.jce.base.spec.ECNamedCurveGenParameterSpec;
import com.xdja.jce.base.spec.ECNamedCurveSpec;
import com.xdja.jce.base.spec.ECParameterSpec;
import com.xdja.jce.base.x9.ECNamedCurveTable;
import com.xdja.jce.base.x9.X9ECParameters;
import com.xdja.jce.coding.asn1.ASN1ObjectIdentifier;
import com.xdja.jce.core.util.Integers;

import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidParameterException;
import java.security.KeyPair;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.ECGenParameterSpec;
import java.util.Hashtable;
import java.util.Map;

/**
 * @Author:ggp
 * @Date:2021/7/19 14:52
 * @Description:
 */
public class ECKeyPairGeneratorEngine extends KeyPairGeneratorSpi {
    ECKeyGenerationParameters param;
    ECKeyPairGenerator engine = new ECKeyPairGenerator();
    Object ecParams = null;
    int strength = 239;
    SecureRandom random = CryptoServicesRegistrar.getSecureRandom();
    boolean initialised = false;
    String algorithm;
    ProviderConfiguration configuration;

    static private Hashtable ecParameters;

    static {
        ecParameters = new Hashtable();

        ecParameters.put(Integers.valueOf(192), new ECGenParameterSpec("prime192v1")); // a.k.a P-192
        ecParameters.put(Integers.valueOf(239), new ECGenParameterSpec("prime239v1"));
        ecParameters.put(Integers.valueOf(256), new ECGenParameterSpec("prime256v1")); // a.k.a P-256

        ecParameters.put(Integers.valueOf(224), new ECGenParameterSpec("P-224"));
        ecParameters.put(Integers.valueOf(384), new ECGenParameterSpec("P-384"));
        ecParameters.put(Integers.valueOf(521), new ECGenParameterSpec("P-521"));
    }

    public ECKeyPairGeneratorEngine() {
        super("EC");
        this.algorithm = "EC";
        this.configuration = XdjaProvider.CONFIGURATION;
    }

    public ECKeyPairGeneratorEngine(
            String algorithm,
            ProviderConfiguration configuration) {
        super(algorithm);
        this.algorithm = algorithm;
        this.configuration = configuration;
    }

    public void initialize(
            int strength,
            SecureRandom random) {
        this.strength = strength;
        this.random = random;

        ECGenParameterSpec ecParams = (ECGenParameterSpec) ecParameters.get(Integers.valueOf(strength));
        if (ecParams == null) {
            throw new InvalidParameterException("unknown key size.");
        }

        try {
            initialize(ecParams, random);
        } catch (InvalidAlgorithmParameterException e) {
            throw new InvalidParameterException("key size not configurable.");
        }
    }

    public void initialize(
            AlgorithmParameterSpec params,
            SecureRandom random)
            throws InvalidAlgorithmParameterException {
        if (params == null) {
            ECParameterSpec implicitCA = configuration.getEcImplicitlyCa();
            if (implicitCA == null) {
                throw new InvalidAlgorithmParameterException("null parameter passed but no implicitCA set");
            }

            this.ecParams = null;
            this.param = createKeyGenParamsBC(implicitCA, random);
        } else if (params instanceof ECParameterSpec) {
            this.ecParams = params;
            this.param = createKeyGenParamsBC((ECParameterSpec) params, random);
        } else if (params instanceof java.security.spec.ECParameterSpec) {
            this.ecParams = params;
            this.param = createKeyGenParamsJCE((java.security.spec.ECParameterSpec) params, random);
        } else if (params instanceof ECGenParameterSpec) {
            initializeNamedCurve(((ECGenParameterSpec) params).getName(), random);
        } else if (params instanceof ECNamedCurveGenParameterSpec) {
            initializeNamedCurve(((ECNamedCurveGenParameterSpec) params).getName(), random);
        } else {
            String name = ECUtil.getNameFrom(params);

            if (name != null) {
                initializeNamedCurve(name, random);
            } else {
                throw new InvalidAlgorithmParameterException("invalid parameterSpec: " + params);
            }
        }

        engine.init(param);
        initialised = true;
    }

    public KeyPair generateKeyPair() {
        if (!initialised) {
            initialize(strength, new SecureRandom());
        }

        AsymmetricCipherKeyPair pair = engine.generateKeyPair();
        ECPublicKeyParameters pub = (ECPublicKeyParameters) pair.getPublic();
        ECPrivateKeyParameters priv = (ECPrivateKeyParameters) pair.getPrivate();

        if (ecParams instanceof ECParameterSpec) {
            ECParameterSpec p = (ECParameterSpec) ecParams;

            BCECPublicKey pubKey = new BCECPublicKey(algorithm, pub, p, configuration);
            return new KeyPair(pubKey,
                    new BCECPrivateKey(algorithm, priv, pubKey, p, configuration));
        } else if (ecParams == null) {
            return new KeyPair(new BCECPublicKey(algorithm, pub, configuration),
                    new BCECPrivateKey(algorithm, priv, configuration));
        } else {
            java.security.spec.ECParameterSpec p = (java.security.spec.ECParameterSpec) ecParams;

            BCECPublicKey pubKey = new BCECPublicKey(algorithm, pub, p, configuration);

            return new KeyPair(pubKey, new BCECPrivateKey(algorithm, priv, pubKey, p, configuration));
        }
    }

    protected ECKeyGenerationParameters createKeyGenParamsBC(ECParameterSpec p, SecureRandom r) {
        return new ECKeyGenerationParameters(new ECDomainParameters(p.getCurve(), p.getG(), p.getN(), p.getH()), r);
    }

    protected ECKeyGenerationParameters createKeyGenParamsJCE(java.security.spec.ECParameterSpec p, SecureRandom r) {
        if (p instanceof ECNamedCurveSpec) {
            X9ECParameters x9P = ECUtils.getDomainParametersFromName(((ECNamedCurveSpec) p).getName());

            if (x9P != null) {
                ECDomainParameters dp = new ECDomainParameters(x9P.getCurve(), x9P.getG(), x9P.getN(), x9P.getH());

                return new ECKeyGenerationParameters(dp, r);
            }
        }

        ECCurve curve = EC5Util.convertCurve(p.getCurve());
        ECPoint g = EC5Util.convertPoint(curve, p.getGenerator());
        BigInteger n = p.getOrder();
        BigInteger h = BigInteger.valueOf(p.getCofactor());
        ECDomainParameters dp = new ECDomainParameters(curve, g, n, h);
        return new ECKeyGenerationParameters(dp, r);
    }

    protected ECNamedCurveSpec createNamedCurveSpec(String curveName)
            throws InvalidAlgorithmParameterException {
        // NOTE: Don't bother with custom curves here as the curve will be converted to JCE type shortly

        X9ECParameters p = ECUtils.getDomainParametersFromName(curveName);
        if (p == null) {
            try {
                // Check whether it's actually an OID string (SunJSSE ServerHandshaker setupEphemeralECDHKeys bug)
                p = ECNamedCurveTable.getByOID(new ASN1ObjectIdentifier(curveName));
                if (p == null) {
                    Map extraCurves = configuration.getAdditionalECParameters();

                    p = (X9ECParameters) extraCurves.get(new ASN1ObjectIdentifier(curveName));

                    if (p == null) {
                        throw new InvalidAlgorithmParameterException("unknown curve OID: " + curveName);
                    }
                }
            } catch (IllegalArgumentException ex) {
                throw new InvalidAlgorithmParameterException("unknown curve name: " + curveName);
            }
        }

        // Work-around for JDK bug -- it won't look up named curves properly if seed is present
        byte[] seed = null; //p.getSeed();

        return new ECNamedCurveSpec(curveName, p.getCurve(), p.getG(), p.getN(), p.getH(), seed);
    }

    protected void initializeNamedCurve(String curveName, SecureRandom random)
            throws InvalidAlgorithmParameterException {
        ECNamedCurveSpec namedCurve = createNamedCurveSpec(curveName);
        this.ecParams = namedCurve;
        this.param = createKeyGenParamsJCE(namedCurve, random);
    }
}
