package com.xdja.jce.crypto.soft.engine.keypair;



import com.xdja.jce.base.cipher.asymmetric.KeyPairGeneratorSpi;
import com.xdja.jce.base.common.CryptoServicesRegistrar;
import com.xdja.jce.base.generators.RSAKeyPairGenerator;
import com.xdja.jce.base.key.AsymmetricCipherKeyPair;
import com.xdja.jce.base.params.RSAKeyGenerationParameters;
import com.xdja.jce.base.params.RSAKeyParameters;
import com.xdja.jce.base.params.RSAPrivateCrtKeyParameters;
import com.xdja.jce.base.rsa.BCRSAPrivateCrtKey;
import com.xdja.jce.base.rsa.BCRSAPublicKey;
import com.xdja.jce.base.util.PrimeCertaintyCalculator;
import com.xdja.jce.coding.asn1.DERNull;
import com.xdja.jce.coding.asn1.pkcs.PKCSObjectIdentifiers;
import com.xdja.jce.coding.asn1.x509.AlgorithmIdentifier;

import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyPair;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.RSAKeyGenParameterSpec;

public class RSAKeyPairGeneratorEngine
    extends KeyPairGeneratorSpi
{
    private static final AlgorithmIdentifier PKCS_ALGID = new AlgorithmIdentifier(PKCSObjectIdentifiers.rsaEncryption, DERNull.INSTANCE);
    private static final AlgorithmIdentifier PSS_ALGID = new AlgorithmIdentifier(PKCSObjectIdentifiers.id_RSASSA_PSS);

    final static BigInteger defaultPublicExponent = BigInteger.valueOf(0x10001);

    RSAKeyGenerationParameters param;
    RSAKeyPairGenerator engine;
    AlgorithmIdentifier algId;


    public RSAKeyPairGeneratorEngine()
    {
        this("RSA", PKCS_ALGID);
    }

    public RSAKeyPairGeneratorEngine(
        String algorithmName,
        AlgorithmIdentifier algId)
    {
        super(algorithmName);

        this.algId = algId;
        engine = new RSAKeyPairGenerator();
        param = new RSAKeyGenerationParameters(defaultPublicExponent,
            CryptoServicesRegistrar.getSecureRandom(), 2048, PrimeCertaintyCalculator.getDefaultCertainty(2048));
        engine.init(param);
    }

    public void initialize(
        int strength,
        SecureRandom random)
    {
        param = new RSAKeyGenerationParameters(defaultPublicExponent,
            random, strength, PrimeCertaintyCalculator.getDefaultCertainty(strength));

        engine.init(param);
    }

    public void initialize(
        AlgorithmParameterSpec params,
        SecureRandom random)
        throws InvalidAlgorithmParameterException
    {
        if (!(params instanceof RSAKeyGenParameterSpec))
        {
            throw new InvalidAlgorithmParameterException("parameter object not a RSAKeyGenParameterSpec");
        }
        RSAKeyGenParameterSpec rsaParams = (RSAKeyGenParameterSpec)params;

        param = new RSAKeyGenerationParameters(
            rsaParams.getPublicExponent(),
            random, rsaParams.getKeysize(), PrimeCertaintyCalculator.getDefaultCertainty(2048));

        engine.init(param);
    }

    public KeyPair generateKeyPair()
    {
        AsymmetricCipherKeyPair pair = engine.generateKeyPair();
        RSAKeyParameters pub = (RSAKeyParameters)pair.getPublic();
        RSAPrivateCrtKeyParameters priv = (RSAPrivateCrtKeyParameters)pair.getPrivate();

        return new KeyPair(new BCRSAPublicKey(algId, pub),
            new BCRSAPrivateCrtKey(algId, priv));
    }

    public static class PSS
        extends RSAKeyPairGeneratorEngine
    {
        public PSS()
        {
            super("RSASSA-PSS", PSS_ALGID);
        }
    }
}
