package com.xdja.jwtcomponentslibrary.edittext;

import android.content.ClipData;
import android.content.ClipboardManager;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Point;
import android.graphics.drawable.ColorDrawable;
import android.os.Build;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.v4.view.PagerAdapter;
import android.support.v4.view.ViewPager;
import android.support.v7.widget.DefaultItemAnimator;
import android.support.v7.widget.GridLayoutManager;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.OrientationHelper;
import android.support.v7.widget.RecyclerView;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.DisplayMetrics;
import android.util.SparseArray;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.PopupWindow;
import android.widget.TextView;

import com.xdja.jwtcomponentslibrary.R;
import com.xdja.jwtcomponentslibrary.utils.ConstUtils;
import com.xdja.jwtcomponentslibrary.utils.IdCardUtils;
import com.xdja.jwtcomponentslibrary.utils.RegexUtils;
import com.xdja.jwtcomponentslibrary.utils.ToastUtils;
import com.xdja.jwtcomponentslibrary.utils.VibrationUtils;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author wanjing
 * @date 2017/8/29
 * <p>
 * 警务通输入法控件
 */

public class JWTEditTextInputSoft implements View.OnClickListener, TextWatcher,
        View.OnTouchListener, ViewPager.OnPageChangeListener {

    private Context mContext;

    private JWTEditTextCallBack mJWTEditTextCallBack;

    /**
     * 阈值，大于该阈值则显示历史记录(默认值2)
     */
    private int mThreshold;

    /**
     * 类别，用于校验输入是否合法
     */
    private int mSort;

    /**
     * 默认文本
     */
    private String mDefaultText;

    private final List<String> zmUpperData = new ArrayList<>();
    private final List<String> zmLowerData = new ArrayList<>();
    private final List<String> numData = new ArrayList<>();

    private PopupWindow popupWindow;

    private View popupView;
    private TextView tsclTv, sfTv, zmTv, szTv;
    private RecyclerView listView;
    private JWTEditTextHistoryAdapter historyAdapter;
    private EditText clearEdit;
    private ImageView delIv;
    private ViewPager viewPager;

    private List<View> views = new ArrayList<>();
    private SparseArray<List<String>> data = new SparseArray<>();
    private Integer[] spanCountArray;
    private Integer[] numRowsArray;

    /**
     * 数据库操作工具类
     */
    private JWTEditTextDBUtils dbUtils;

    /**
     * 根据关键字查询到的历史记录
     */
    private List<String> queryData;

    private final String UPPERCASE = "大写";
    private final String LOWERCASE = "小写";
    private final String DELETE = "删除";
    private final String COPY = "复制";
    /**
     * 车牌号输入模式，禁用字母O，I
     */
    private boolean plateMode;

    public JWTEditTextInputSoft(Context context, boolean plateMode) {
        this.mContext = context;
        this.plateMode = plateMode;
        initData();
        initPopupView();
        initPopupWindow();
        initGridViews();
        initPagerAdapter();
    }

    private void initPagerAdapter() {
        PagerAdapter pagerAdapter = new PagerAdapter() {
            @Override
            public boolean isViewFromObject(View arg0, Object arg1) {
                return arg0 == arg1;
            }

            @Override
            public int getCount() {
                return views == null ? 0 : views.size();
            }

            @Override
            public void destroyItem(ViewGroup container, int position, Object view) {
                container.removeView((View) view);
            }

            @Override
            public Object instantiateItem(ViewGroup collection, int position) {
                View view = views.get(position);
                collection.addView(view, 0);
                return view;
            }
        };
        viewPager.setOffscreenPageLimit(1);
        viewPager.setAdapter(pagerAdapter);
    }

    private GridLayoutManager getGridLayoutManager(final int index) {
        GridLayoutManager gridLayoutManager = new GridLayoutManager(mContext, spanCountArray[index]);
        if (index == 2) {
            gridLayoutManager.setSpanSizeLookup(new GridLayoutManager.SpanSizeLookup() {
                @Override
                public int getSpanSize(int position) {
                    String text = data.valueAt(index).get(position);
                    if (RegexUtils.isMatch("\\d", text)) {
                        //数字
                        return 4;
                    } else if (RegexUtils.isMatch("[\u4e00-\u9fa5]{2}", text)) {
                        //特殊操作（大小写切换、删除、复制）
                        return 10;
                    } else {
                        //字母
                        return 5;
                    }
                }
            });
        } else if (index == 3) {
            gridLayoutManager.setSpanSizeLookup(new GridLayoutManager.SpanSizeLookup() {
                @Override
                public int getSpanSize(int position) {
                    if (position > 8) {
                        return 3;
                    } else {
                        return 4;
                    }
                }
            });
        }
        return gridLayoutManager;
    }

    private void initGridViews() {
        for (int index = 0; index < data.size(); index++) {
            View view = LayoutInflater.from(mContext)
                    .inflate(R.layout.jwt_edittext_gridview, null);
            RecyclerView gridView = view.findViewById(R.id.soft_grid_view);
            gridView.setLayoutManager(getGridLayoutManager(index));
            final List<String> buttons = data.valueAt(index);
            final JWTEditTextDataAdapter adapter = new JWTEditTextDataAdapter(
                    mContext, buttons, numRowsArray[index]);
            adapter.setPlateMode(plateMode);
            gridView.setAdapter(adapter);
            adapter.setOnItemClickListener(new AdapterView.OnItemClickListener() {
                @Override
                public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                    vibrate();
                    String button = buttons.get(position);
                    if (UPPERCASE.equals(button)) {
                        buttons.clear();
                        buttons.addAll(numData);
                        buttons.addAll(zmUpperData);
                        adapter.notifyDataSetChanged();
                    } else if (LOWERCASE.equals(button)) {
                        buttons.clear();
                        buttons.addAll(numData);
                        buttons.addAll(zmLowerData);
                        adapter.notifyDataSetChanged();
                    } else if (COPY.equals(button)) {
                        copyAction();
                    } else if (DELETE.equals(button)) {
                        delAction();
                    } else {
                        setResult(button);
                    }
                }
            });
            views.add(view);
        }
    }

    private void initPopupView() {
        if (popupView == null) {
            popupView = LayoutInflater.from(mContext).inflate(R.layout.jwt_edittext_inputsoft, null);
            Button confirmBtn = popupView.findViewById(R.id.confirm_btn);
            tsclTv = popupView.findViewById(R.id.tscl_btn);
            sfTv = popupView.findViewById(R.id.sf_btn);
            zmTv = popupView.findViewById(R.id.zm_btn);
            szTv = popupView.findViewById(R.id.sz_btn);
            delIv = popupView.findViewById(R.id.del_iv);
            listView = popupView.findViewById(R.id.history_list);
            clearEdit = popupView.findViewById(R.id.result_edit);
            viewPager = popupView.findViewById(R.id.edit_viewpager);
            confirmBtn.setOnClickListener(this);
            tsclTv.setOnClickListener(this);
            sfTv.setOnClickListener(this);
            zmTv.setOnClickListener(this);
            szTv.setOnClickListener(this);
            delIv.setOnClickListener(this);
            clearEdit.setOnTouchListener(this);
            clearEdit.addTextChangedListener(this);
            viewPager.addOnPageChangeListener(this);

            // Android版本是5.0以上，此方法生效。隐藏输入法且有光标
            clearEdit.setShowSoftInputOnFocus(false);
        }
    }

    private void initPopupWindow() {
        if (popupWindow == null) {
            WindowManager manager = (WindowManager) mContext.getSystemService(Context.WINDOW_SERVICE);
            DisplayMetrics outMetrics = new DisplayMetrics();
            manager.getDefaultDisplay().getRealMetrics(outMetrics);
            Point point = new Point();
            manager.getDefaultDisplay().getSize(point);
//            popupWindow = new PopupWindow(popupView, ViewGroup.LayoutParams.MATCH_PARENT, point.y + getStatusBarHeightCompat(mContext));
            popupWindow = new PopupWindow(popupView, ViewGroup.LayoutParams.MATCH_PARENT, outMetrics.heightPixels - getNavigationBarHeight(mContext));
            popupWindow.setFocusable(true);
            popupWindow.setOutsideTouchable(true);
            popupWindow.setClippingEnabled(false);
            popupWindow.setBackgroundDrawable(new ColorDrawable(0xbf000000));
            popupWindow.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE);
        }
    }

    private void initData() {
        String[] tsclDataArray = mContext.getResources().getStringArray(R.array.tscl);
        String[] sfDataArray = mContext.getResources().getStringArray(R.array.sf);
        String[] zmYDataArray = mContext.getResources().getStringArray(R.array.zm_y);
        String[] zmXDataArray = mContext.getResources().getStringArray(R.array.zm_x);
        String[] sfzDataArray = mContext.getResources().getStringArray(R.array.sfz);

        zmUpperData.addAll(Arrays.asList(zmYDataArray));
        zmLowerData.addAll(Arrays.asList(zmXDataArray));
        List<String> sfzData = Arrays.asList(sfzDataArray);
        numData.addAll(sfzData.subList(0, sfzDataArray.length - 4));
        numData.add("0");

        data.append(0, Arrays.asList(tsclDataArray));
        data.append(1, Arrays.asList(sfDataArray));
        List<String> zmData = new ArrayList<>();
        zmData.addAll(numData);
        zmData.addAll(zmUpperData);
        data.append(2, zmData);
        data.append(3, sfzData);

        spanCountArray = new Integer[]{3, 7, 40, 12};
        numRowsArray = new Integer[]{4, 5, 5, 4};
    }

    public void setCallBack(JWTEditTextCallBack callBack) {
        this.mJWTEditTextCallBack = callBack;
    }

    public void setSort(int sort) {
        this.mSort = sort;
    }

    public void setThreshold(int threshold) {
        this.mThreshold = threshold;
    }

    public void showInputSoft(int inputType, String defaultText, String hint) {
        this.mDefaultText = defaultText;

        clearEdit.setText(defaultText);
        clearEdit.setSelection(defaultText.length());
        clearEdit.setHint(hint);

        viewPager.setCurrentItem(inputType);

        if (popupWindow != null && popupView != null) {
            popupWindow.showAtLocation(popupView.getRootView(), Gravity.TOP, 0, 0);
        }
    }

    @Override
    public void onClick(View view) {
        vibrate();

        int i = view.getId();
        if (i == R.id.del_iv) {
            clearAction();

        } else if (i == R.id.confirm_btn) {
            confirmAction(clearEdit.getText().toString());

        } else if (i == R.id.tscl_btn) {
            viewPager.setCurrentItem(0);

        } else if (i == R.id.sf_btn) {
            viewPager.setCurrentItem(1);

        } else if (i == R.id.zm_btn) {
            viewPager.setCurrentItem(2);

        } else if (i == R.id.sz_btn) {
            viewPager.setCurrentItem(3);

        }
    }

    /**
     * 释放资源，关闭数据库，dismiss popupWindow。
     */
    private void dismiss() {
        if (dbUtils != null) {
            dbUtils.closeDatabase();
            dbUtils = null;
        }
        popupWindow.dismiss();
    }

    /**
     * 确定操作
     *
     * @param content 输入内容
     */
    private void confirmAction(String content) {
        // 非空校验
        if (!TextUtils.isEmpty(content)) {
            // 校验输入是否合法 0-车牌号 1-身份证号
            if (mSort == 0) {
                if (!RegexUtils.isMatch(ConstUtils.REGEX_NUMBER_PLATE, content)) {
                    showToast("格式有误，请重新输入");
                    return;
                }
            }
            if (mSort == 1) {
                if (!RegexUtils.isMatch(ConstUtils.REGEX_NUMBER_IDCARD, content)) {
                    showToast("格式有误，请重新输入");
                    return;
                }
                if (!IdCardUtils.isIdentityCode(content)) {
                    showToast("身份证校验码无效，不是合法的身份证号码");
                    return;
                }
            }

            if (dbUtils == null) {
                dbUtils = new JWTEditTextDBUtils(mContext);
            }
            dbUtils.saveHistory(content);
        }

        if (mJWTEditTextCallBack != null) {
            mJWTEditTextCallBack.onConfirm(content);
        }
        dismiss();
    }

    /**
     * 删除操作
     */
    private void delAction() {
        String text = clearEdit.getText().toString();
        if (TextUtils.isEmpty(text)) {
            return;
        }
        int selection = clearEdit.getSelectionStart();
        if (selection <= 0) {
            return;
        }
        clearEdit.setText(String.format("%s%s",
                text.substring(0, selection - 1), text.substring(selection)));
        clearEdit.setSelection(selection - 1);
    }

    /**
     * 复制到粘贴板
     */
    private void copyAction() {
        String text = clearEdit.getText().toString();
        if (TextUtils.isEmpty(text)) {
            return;
        }
        ClipboardManager cmb = (ClipboardManager) mContext.getSystemService(Context.CLIPBOARD_SERVICE);
        if (cmb != null) {
            cmb.setPrimaryClip(ClipData.newPlainText(null, text));
            showToast("已经复制到粘贴板！");
        }
    }

    /**
     * 清空内容
     */
    private void clearAction() {
        clearEdit.setText("");
    }

    @Override
    public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {

    }

    @Override
    public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {

    }

    @Override
    public void afterTextChanged(Editable editable) {
        if (TextUtils.isEmpty(editable.toString())) {
            delIv.setVisibility(View.GONE);
            listView.setVisibility(View.GONE);
            return;
        }
        delIv.setVisibility(View.VISIBLE);
        if (!TextUtils.isEmpty(mDefaultText)) {
            listView.setVisibility(View.GONE);
            mDefaultText = "";
            return;
        }
        if (editable.length() > mThreshold) {
            if (dbUtils == null) {
                dbUtils = new JWTEditTextDBUtils(mContext);
            }
            List<String> result = dbUtils.query(editable.toString());
            if (result == null || result.size() == 0) {
                listView.setVisibility(View.GONE);
            } else {
                queryData = result;
                listView.setVisibility(View.VISIBLE);
                if (historyAdapter == null) {
                    listView.setLayoutManager(new LinearLayoutManager(mContext,
                            OrientationHelper.HORIZONTAL, false));
                    listView.setHasFixedSize(true);
                    listView.setItemAnimator(new DefaultItemAnimator());
                    historyAdapter = new JWTEditTextHistoryAdapter(queryData, new AdapterView.OnItemClickListener() {
                        @Override
                        public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                            if (queryData != null && queryData.size() > position) {
                                confirmAction(queryData.get(position));
                            }
                        }
                    });
                    listView.setAdapter(historyAdapter);
                } else {
                    historyAdapter.updateData(queryData);
                }
            }
        }
    }

    @Override
    public boolean onTouch(View view, MotionEvent motionEvent) {
        //隐藏系统输入法，显示光标
        EditText et = (EditText) view;
        // back up the input type
        Class<EditText> cls = EditText.class;
        try {
            Method setShowSoftInputOnFocus = cls.getMethod("setShowSoftInputOnFocus", boolean.class);
            setShowSoftInputOnFocus.setAccessible(false);
            setShowSoftInputOnFocus.invoke(et, false);
        } catch (NoSuchMethodException | IllegalArgumentException
                | IllegalAccessException | InvocationTargetException e) {
            e.printStackTrace();
        }
        // call native handler
        et.onTouchEvent(motionEvent);
        return true;
    }

    @Override
    public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {

    }

    @Override
    public void onPageSelected(int position) {
        tsclTv.setSelected(position == 0);
        sfTv.setSelected(position == 1);
        zmTv.setSelected(position == 2);
        szTv.setSelected(position == 3);
    }

    @Override
    public void onPageScrollStateChanged(int state) {

    }

    /**
     * 展示吐司
     *
     * @param message 展示内容
     */
    private void showToast(String message) {
        ToastUtils.showShortToast(mContext, message);
    }

    /**
     * 震动，默认震动20ms
     */
    private void vibrate() {
        vibrate(20);
    }

    /**
     * 震动
     *
     * @param milliseconds 震动时长
     */
    private void vibrate(long milliseconds) {
        VibrationUtils.vibrate(mContext, milliseconds);
    }

    /**
     * 输入文本
     *
     * @param input 输入内容
     */
    private void setResult(String input) {
        int selection = clearEdit.getSelectionStart();
        String text = clearEdit.getText().toString();
        clearEdit.setText(String.format("%s%s%s", text.substring(0, selection), input, text.substring(selection)));
        clearEdit.setSelection(selection + input.length());
    }

    private int getStatusBarHeightCompat(Context context) {
        int result = 0;
        int resId = context.getResources().getIdentifier("status_bar_height", "dimen", "android");
        if (resId > 0) {
            result = context.getResources().getDimensionPixelOffset(resId);
        }
        if (result <= 0) {
            result = context.getResources().getDimensionPixelOffset(R.dimen.dimen_25dp);
        }
        return result;
    }

    private int getNavigationBarHeight(Context context) {
        if (!checkNavigationBarShow(context)){
            return 0;
        }
        Resources resources = context.getResources();
        int resourceId = resources.getIdentifier("navigation_bar_height", "dimen", "android");
        int height = resources.getDimensionPixelSize(resourceId);
        return height;
    }

    /**
     * 判断虚拟导航栏是否显示
     *
     * @param context 上下文对象
     * @return true(显示虚拟导航栏)，false(不显示或不支持虚拟导航栏)
     */
    private boolean checkNavigationBarShow(@NonNull Context context) {
        boolean hasNavigationBar = false;
        Resources rs = context.getResources();
        int id = rs.getIdentifier("config_showNavigationBar", "bool", "android");
        if (id > 0) {
            hasNavigationBar = rs.getBoolean(id);
        }
        try {
            Class systemPropertiesClass = Class.forName("android.os.SystemProperties");
            Method m = systemPropertiesClass.getMethod("get", String.class);
            String navBarOverride = (String) m.invoke(systemPropertiesClass, "qemu.hw.mainkeys");
            //判断是否隐藏了底部虚拟导航
            int navigationBarIsMin = 0;
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
                navigationBarIsMin = Settings.System.getInt(context.getContentResolver(),
                        "navigationbar_is_min", 0);
            } else {
                navigationBarIsMin = Settings.Global.getInt(context.getContentResolver(),
                        "navigationbar_is_min", 0);
            }
            if ("1".equals(navBarOverride) || 1 == navigationBarIsMin) {
                hasNavigationBar = false;
            } else if ("0".equals(navBarOverride)) {
                hasNavigationBar = true;
            }
        } catch (Exception e) {
        }
        return hasNavigationBar;
    }
}
