package com.xdja.jwtcomponentslibrary.edittext;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by wanjing on 2017/8/30.
 * <p>
 * desc：警务通输入法数据库工具类。
 */

public class JWTEditTextDBUtils {

    private Context mContext;

    private JWTEditTextDBHelper dbHelper;

    private SQLiteDatabase sqLiteDatabase;

    /**
     * 对象锁
     */
    private final static Object object = new Object();

    public JWTEditTextDBUtils(Context context) {
        this.mContext = context;
        this.dbHelper = new JWTEditTextDBHelper(context);

        this.openDatabase();
    }

    private void openDatabase() {
        synchronized (object) {
            if (sqLiteDatabase == null || !sqLiteDatabase.isOpen()) {
                sqLiteDatabase = dbHelper.getWritableDatabase();
            }
        }
    }

    protected void closeDatabase() {
        synchronized (object) {
            if (sqLiteDatabase != null) {
                sqLiteDatabase.close();
                sqLiteDatabase = null;
            }
        }
    }

    /**
     * 添加新的记录
     *
     * @param content 输入文本
     * @return 0，成功；-1，失败
     */
    protected synchronized int insert(String content) {
        int result = 0;
        synchronized (object) {
            try {
                String sql = "insert into t_history(c_content, c_time) values(?, ?)";
                sqLiteDatabase.execSQL(sql, new Object[]{content, System.currentTimeMillis()});
            } catch (Exception e) {
                e.printStackTrace();
                result = -1;
            }
        }
        return result;
    }

    /**
     * 更新历史记录
     *
     * @param content 输入文本
     * @return 0，成功；-1，失败
     */
    protected synchronized int update(String content) {
        int result = 0;
        synchronized (object) {
            try {
                String sql = "update t_history set c_time = ? where c_content = ?";
                sqLiteDatabase.execSQL(sql, new Object[]{System.currentTimeMillis(), content});
            } catch (Exception e) {
                e.printStackTrace();
                result = -1;
            }
        }
        return result;
    }

    /**
     * 根据关键字查询历史记录
     *
     * @param keyword 关键字
     * @return 查询结果
     */
    protected synchronized List<String> query(String keyword) {
        List<String> result = new ArrayList<>();
        Cursor cursor = null;
        try {
            String sql = "select distinct c_content from t_history where c_content like ? order by c_time desc";
            cursor = sqLiteDatabase.rawQuery(sql, new String[]{"%" + keyword + "%"});
            while (cursor != null && cursor.moveToNext()) {
                result.add(cursor.getString(cursor.getColumnIndex("c_content")));
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return result;
    }

    /**
     * 保存输入记录
     *
     * @param content 输入文本
     * @return 0，成功；-1，失败
     */
    protected synchronized int saveHistory(String content) {
        int result;
        Cursor cursor = null;
        try {
            //判断是插入还是更新。
            String querySql = "select 1 from t_history where c_content like ?";
            cursor = sqLiteDatabase.rawQuery(querySql, new String[]{"%" + content + "%"});
            if (cursor != null && cursor.moveToNext()) {
                result = update(content);
            } else {
                result = insert(content);
            }
        } catch (Exception e) {
            e.printStackTrace();
            result = -1;
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return result;
    }
}
