package com.xdja.jwtcomponentslibrary.spinner;

import android.content.Context;
import android.content.res.TypedArray;
import android.support.annotation.ColorInt;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v7.widget.AppCompatSpinner;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.TextView;

import com.xdja.jwtcomponentslibrary.R;

import java.util.ArrayList;
import java.util.List;

/**
 * @author wanjing
 * @date 2018/2/27
 * <p>
 * 警务通spinner组件, 支持搜索。
 */

public class JWTSpinnerV1<E extends BaseSpinnerItem> extends AppCompatSpinner implements AdapterView.OnItemSelectedListener {

    private Context mContext;

    private List<E> mDataItemBeen = new ArrayList<>();

    private JWTSpinnerCallBack mCallBack;

    private String defaultTextOrValue;
    private int threshold;
    private @ColorInt
    int textColor;
    private float textSize;
    private int textAlignment;
    private int paddingLeft;
    private int paddingRight;
    private int dialogWidth;
    private int dialogHeight;
    private int dialogMaxHeight;
    private boolean dialog;
    private boolean adjustResize;
    private boolean hint;

    private JWTSpinnerV1DialogView alertDialog;
    private ArrayAdapter<E> arrayAdapter;

    public JWTSpinnerV1(Context context) {
        this(context, null);
    }

    public JWTSpinnerV1(Context context, AttributeSet attrs) {
        super(context, attrs);

        init(context, attrs);
    }

    public JWTSpinnerV1(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        init(context, attrs);
    }

    private void init(Context context, AttributeSet attrs) {
        this.mContext = context;

        TypedArray ta = mContext.obtainStyledAttributes(attrs, R.styleable.JWTSpinnerV1);
        if (ta != null) {
            defaultTextOrValue = ta.getString(R.styleable.JWTSpinnerV1_default_text_value);
            threshold = ta.getInteger(R.styleable.JWTSpinnerV1_search_threshold, 5);
            textColor = ta.getColor(R.styleable.JWTSpinnerV1_spinner_textColor,
                    ContextCompat.getColor(context, R.color.spinnerV1_default_text_color));
            textSize = ta.getDimensionPixelSize(R.styleable.JWTSpinnerV1_spinner_textSize,
                    getResources().getDimensionPixelSize(R.dimen.spinnerV1_default_text_size));
            textAlignment = ta.getInteger(R.styleable.JWTSpinnerV1_spinner_textAlignment,
                    View.TEXT_ALIGNMENT_VIEW_START);
            paddingLeft = ta.getDimensionPixelSize(R.styleable.JWTSpinnerV1_spinner_paddingLeft, 0);
            paddingRight = ta.getDimensionPixelSize(R.styleable.JWTSpinnerV1_spinner_paddingRight, 0);
            dialogWidth = ta.getLayoutDimension(R.styleable.JWTSpinnerV1_spinner_dialog_width, dp2px(300));
            dialogHeight = ta.getLayoutDimension(R.styleable.JWTSpinnerV1_spinner_dialog_height, -2);
            dialogMaxHeight = ta.getLayoutDimension(R.styleable.JWTSpinnerV1_spinner_dialog_height_max, dp2px(224));
            dialog = ta.getBoolean(R.styleable.JWTSpinnerV1_spinner_dialog, true);
            adjustResize = ta.getBoolean(R.styleable.JWTSpinnerV1_adjust_resize, false);
            hint = ta.getBoolean(R.styleable.JWTSpinnerV1_spinner_hint, false);
            ta.recycle();
        }
        setOnItemSelectedListener(this);
        mDataItemBeen=new ArrayList<>();
        arrayAdapter = new ArrayAdapter<E>(mContext, R.layout.jwt_spinner_v1_simple_list_item_1, mDataItemBeen) {
            @NonNull
            @Override
            public View getView(int position, @Nullable View convertView, @NonNull ViewGroup parent) {
                View view = super.getView(position, convertView, parent);
                TextView textView = (TextView) view;
                textView.setTextColor(textColor);
                textView.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSize);
                textView.setTextAlignment(textAlignment);
                textView.setPadding(paddingLeft, 0, paddingRight, 0);
                return view;
            }

            @Override
            public int getCount() {
                if (hint) {
                    return super.getCount() - 1;
                }
                return super.getCount();
            }
        };
        arrayAdapter.setDropDownViewResource(android.R.layout.simple_list_item_1);
        setAdapter(arrayAdapter);
    }

    /**
     * 设置数据
     *
     * @param dataItemBeen 数据
     */
    public void setData(List<E> dataItemBeen) {
        setData(dataItemBeen, defaultTextOrValue);
    }

    /**
     * 设置数据和默认文本
     *
     * @param dataItemBeen       数据
     * @param defaultTextOrValue 默认文本/默认值
     */
    public void setData(List<E> dataItemBeen, String defaultTextOrValue) {

        this.mDataItemBeen.clear();
        if (dataItemBeen != null && dataItemBeen.size() > 0) {
            mDataItemBeen.addAll(dataItemBeen);
        }
        arrayAdapter.notifyDataSetChanged();
        if (hint) {
            setSelection(arrayAdapter.getCount());
            return;
        }
        setDefaultValue(defaultTextOrValue);
    }

    /**
     * 设置默认文本
     *
     * @param defaultTextOrValue 默认文本/默认值
     */
    public void setDefaultValue(@NonNull String defaultTextOrValue) {
        int selectedPosition = -1;
        if (defaultTextOrValue != null) {
            for (E dataItemBean : mDataItemBeen) {
                if (defaultTextOrValue.equals(dataItemBean.getText()) || defaultTextOrValue.equals(dataItemBean.getValue())) {
                    selectedPosition = mDataItemBeen.indexOf(dataItemBean);
                    break;
                }
            }
        }
        setSelection(selectedPosition, true);
    }

    public void setCallBack(JWTSpinnerCallBack callBack) {
        this.mCallBack = callBack;
    }

    public String getSelectedText() {
        if (mDataItemBeen != null && mDataItemBeen.size() > getSelectedItemPosition()) {
            return mDataItemBeen.get(getSelectedItemPosition()).getText();
        }
        return "";
    }

    public String getSelectedValue() {
        if (mDataItemBeen != null && mDataItemBeen.size() > getSelectedItemPosition()) {
            return mDataItemBeen.get(getSelectedItemPosition()).getValue();
        }
        return "";
    }

    @Override
    public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
        if (mDataItemBeen == null || mDataItemBeen.size() <= position) {
            return;
        }
        if (mCallBack != null) {
            mCallBack.onItemSelected(getSelectedText(), getSelectedValue());
        }
    }

    @Override
    public void onNothingSelected(AdapterView<?> parent) {
        if (mCallBack != null) {
            mCallBack.onNothingSelected();
        }
    }

    @Override
    public boolean performClick() {
        if (mDataItemBeen != null && !mDataItemBeen.isEmpty()) {
            if (dialog || mDataItemBeen.size() > threshold) {
                // 弹框展示数据。
                showDialogView();
                return true;
            }
        }

        // Else let the AppCompatSpinner handle the click
        return super.performClick();
    }

    /**
     * 展示搜索DialogView.
     */
    private void showDialogView() {
        if (alertDialog == null) {
            alertDialog = new JWTSpinnerV1DialogView(mContext, mDataItemBeen, threshold, hint);
            alertDialog.setSearchItemClickCallBack(new JWTSpinnerV1DialogView.OnItemSelectedListener<E>() {

                @Override
                public void onItemSelected(E item, int position) {
                    setSelection(position, true);
                }
            });
        }
        alertDialog.notifyAll(mDataItemBeen);
        alertDialog.show();
        if (!adjustResize) {
            Window window = alertDialog.getWindow();
            if (window != null) {
                int width = dialogWidth;
                int height;
                if (dialogHeight > 0) {
                    height = dialogHeight;
                } else {
                    //wrap_content 最大高度224dp 每个item高度48dp
                    int itemHeight = dp2px(48);
                    if (mDataItemBeen.size() * itemHeight > dialogMaxHeight) {
                        height = dialogMaxHeight;
                    } else {
                        height = dialogHeight;
                    }
                }
                window.setLayout(width, height);
            }
        }

    }

    private int dp2px(int dp) {
        return (int) (mContext.getResources().getDisplayMetrics().density * dp + 0.5f);
    }
}
