package com.xdja.jxlsclient.util;

import com.rabbitmq.client.*;
import com.rabbitmq.client.impl.recovery.AutorecoveringConnection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.ConnectException;
import java.util.ArrayList;
import java.util.List;

/**
 * @ClassName：RabbitMqUtil
 * @Description：rabbitMq工具类
 * @author: wrf
 * @date: 2019/8/16/016 11:22
 * @version: V1.0
 */
public class RabbitMqInit {
    private static final Logger logger = LoggerFactory.getLogger(RabbitMqInit.class);

    private static AutorecoveringConnection connection = null;
    private static Channel channel = null;

    private RabbitMqInit(){}

    private static class RabbitMqInitHolder {
        public static final RabbitMqInit instance = new RabbitMqInit();
    }

    public static RabbitMqInit getInstance() {
        return RabbitMqInitHolder.instance;
    }

    /**
     * @Description: 初始化连接RabbitMq
     * @author wrf
     * @Date 2019-08-19 10:28
     * @param host 连接RabbitMq的ip地址
     * @param port 连接RabbitMq的端口
     * @param mqUserName 连接RabbitMq用户名
     * @param mqPassword 连接RabbitMq密码
     * @return
     */
    public synchronized void init(String host, int port, String mqUserName, String mqPassword) throws Exception {
        logger.info("RabbitMq初始化>>>host:【{}】, port:【{}】, mqUserName:【{}】, mqPassword:【{}】", host, port, mqUserName, mqPassword);
        init(host + ":" + port, mqUserName, mqPassword);
        logger.info("RabbitMq初始化<<<");
    }

    /**
     * @Description: RabbitMq连接初始化
     * @author wrf
     * @Date 2019-08-19 10:23
     * @param connectString 连接RabbitMq初始化字符串，例如，192.168.18.204:8080,192.168.24.51:9090
     * @param mqUserName 连接RabbitMq用户名
     * @param mqPassword 连接RabbitMq密码
     * @return
    */
    public synchronized void init(String connectString, final String mqUserName, final String mqPassword) throws Exception {
        logger.info("RabbitMq初始化>>>connectString:【{}】, mqUserName:【{}】, mqPassword:【{}】", connectString, mqUserName, mqPassword);
        //1.判断连接
        if (null != connection) {
            logger.info("rabbitMq已连接");
            if (null != channel) {
                channel.close();
            }

            connection.close();
        }

        //2、获取连接地址
        String[] aryIpPort = connectString.split(",");
        if (null == aryIpPort || aryIpPort.length < 1) {
            logger.error("rabbitMq连接地址配置有误");
            throw new RuntimeException("rabbitMq连接地址配置有误,connectString标准格式为ip:port多个ip和port用英文逗号分隔，例如：192.168.18.204:8080,192.168.88.204:8082");
        }

        final List<Address> listAddress = new ArrayList<Address>();

        for (String ipPort : aryIpPort) {
            String[] aryIp = ipPort.split(":");
            if (null == aryIp || aryIp.length != 2) {
                continue;
            }
            List<Address> aryAddress = buildMqAddress(aryIp[0], Integer.parseInt(aryIp[1]));
            if (null != aryAddress && aryAddress.size() > 0) {
                listAddress.addAll(aryAddress);
            }
        }

        //3、获取连接
        if (null == listAddress || listAddress.size() < 1) {
            logger.error("连接mq地址不正确");
        }

        if (null != connection && connection.isOpen()) {
            Thread.sleep(2 * 1000);
        }
        //4、初始化连接RabbitMq
        try {
            connection = newConnection(listAddress.toArray(new Address[]{}), mqUserName, mqPassword);
        }catch (Exception e) {
            logger.error("初始化连接RabbitMq失败，启动异步线程定时连接RabbitMq");
        }
        //5、初始化连接RabbitMq失败，启动异步线程连接mq
        if (null == connection) {
            new Thread(new Runnable() {
                @Override
                public void run() {
                    connection = getMqConnection(listAddress.toArray(new Address[]{}), mqUserName, mqPassword);
                }
            }).start();
        }
    }

    /**
     * @Description: 获取连接RabbitMq地址
     * @author wrf
     * @Date 2019-08-16 11:41
     * @param
     * @return
    */
    private List<Address> buildMqAddress(String host, int port) {
        boolean checkIp = CommonUtil.checkIp(host);
        if (! checkIp) {
            return null;
        }
        boolean checkPort = CommonUtil.checkPort(port);
        if (! checkPort) {
           return null;
        }
        List<Address> listAddress = new ArrayList<Address>();
        Address address = new Address(host, port);
        listAddress.add(address);

        return listAddress;
    }

    /**
     * @Description: 获取RabbitMq连接
     * @author wrf
     * @Date 2019-08-23 16:23
     * @param addresses 地址
     * @param mqUserName 用户名
     * @param mqPassword 密码
     * @return AutorecoveringConnection
    */
    private AutorecoveringConnection newConnection(Address[] addresses, String mqUserName, String mqPassword) throws Exception {
        ConnectionFactory factory = new ConnectionFactory();
        factory.setUsername(mqUserName);
        factory.setPassword(mqPassword);
        factory.setAutomaticRecoveryEnabled(true);
        factory.setNetworkRecoveryInterval(5000);
        connection = (AutorecoveringConnection) factory.newConnection(addresses);
        return connection;
    }

    /**
     * @Description: 连接mq
     * @author wrf
     * @date 2018-08-06 15:48
     * @param addresses 地址
     * @param mqUserName 用户名
     * @param mqPassword 密码
     * @return AutorecoveringConnection
     */
    private AutorecoveringConnection getMqConnection(Address[] addresses, String mqUserName, String mqPassword) {
        AutorecoveringConnection connection = null;
        SleepTimeCalculateUtil.SleepTimeCalculateBean sleepTimeCalculateBean = SleepTimeCalculateUtil.getSleepTimeCalculateBean();
        while (true){
            try {
                //默认10s 重连
                connection = newConnection(addresses, mqUserName, mqPassword);
                logger.info("RabbitMqConsumer rabbitMQ connection finish.");
                break;
            }catch (ConnectException e) {
                logger.info("RabbitMqConsumer rabbitMQ connectException, try again: 【{}】", e);
            } catch (IOException e) {
                logger.info("RabbitMqConsumer rabbitMQ IOException, try again: 【{}】", e);
            }catch (Exception e) {
                logger.info("RabbitMqConsumer rabbitMQ Exception, try again: 【{}】", e);
            }
            try {
                SleepTimeCalculateUtil.calculateSleepTime(sleepTimeCalculateBean);

                logger.error("RabbitMqConsumer connect rabbitMq error, sleep {}s connect.", sleepTimeCalculateBean.getSleepTime());
                Thread.sleep(sleepTimeCalculateBean.getSleepTime() * 1000);
            }catch (Exception e) {}
        }

        //双重校验
        if (null == connection) {
            getMqConnection(addresses, mqUserName, mqPassword);
        }

        connection.addShutdownListener(new ShutdownListener() {
            @Override
            public void shutdownCompleted(ShutdownSignalException cause) {
                String hardError = "";
                String applInit = "";

                if (cause.isHardError()) {
                    hardError = "connection";
                } else {
                    hardError = "channel";
                }
                if (cause.isInitiatedByApplication()) {
                    applInit = "application";
                } else {
                    applInit = "broker";
                }
                logger.error("Connectivity to MQ has failed.  It was caused by "
                        + applInit + " at the " + hardError
                        + " level.  Reason received " + cause.getReason());
            }
        });

        return connection;
    }

    private void initQueue(String queueName) throws IOException {
        // 检测队列是否存在
        try {
            channel.queueDeclarePassive(queueName);
        } catch (IOException e) {
            // 声明队列，并绑定交换器
            declareQueue(queueName);
        }
    }

    /**
     * 创建连接通道
     * @throws IOException
     */
    private void createChannel() throws IOException {
        channel = connection.createChannel();
        channel.confirmSelect();
        //增加监听器，channel因异常错误关闭后，channel不会自动为空，但此channel已经不可再用，需要监听后，置为空
        channel.addShutdownListener(new ShutdownListener() {
            @Override
            public void shutdownCompleted(ShutdownSignalException cause) {
                // Beware that proper synchronization is needed here
                logger.debug("Handling channel shutdown...", cause);
                if (cause.isInitiatedByApplication()) {
                    logger.debug("Shutdown is initiated by application. Ignoring it.");
                } else {
                    channel = null;
                }
            }
        });
    }

    /**
     * 获取连接通道
     * @return Channel
     */
    public Channel getChannel() {
        if(channel == null || ! channel.isOpen()){
            try {
                createChannel();
            } catch (IOException e) {
                logger.error("getChannel error: ", e);
            }
        }
        return channel;
    }

    /**
     * 创建交换器
     * @param exchange 交换器名称
     * @param routtingType 路由规则
     * @throws IOException 可能会抛io异常
     */
    public void declareExchange(String exchange,String routtingType) throws IOException {
        if (channel == null  || !channel.isOpen()) {
            createChannel();
        }
        channel.exchangeDeclare(exchange, routtingType, true);
    }

    /**
     * 删除交换器
     * @param exchange 交换器名称
     * @throws IOException
     */
    public void removeExchange(String exchange) throws IOException {
        if (channel == null) {
            createChannel();
        }
        channel.exchangeDelete(exchange);
    }

    /**
     * 创建队列
     * @param queueName 队列名称
     * @throws IOException
     * 默认声明为持久化队列，非自动删除，不排他队列
     */
    public void declareQueue(String queueName) throws IOException {
        declareQueue(queueName, true);
    }

    /**
     * 创建队列
     * @param queueName 队列名称
     * @param durable 是否声明为持久化队列
     * @throws IOException
     */
    public void declareQueue(String queueName, boolean durable) throws IOException {
        if (channel == null) {
            createChannel();
        }
        channel.queueDeclare(queueName, durable, false, false, null);
    }

    /**
     * 删除队列
     * @param queueName 队列名称
     * @throws IOException
     */
    public void removeQueue(String queueName) throws IOException {
        if (channel == null) {
            createChannel();
        }
        channel.queueDelete(queueName);
    }

    /**
     * 绑定路由，队列之间的关系
     * @param exchange 路由名称
     * @param queueName 队列名称
     * @param routing 路由关键词
     * @throws IOException
     */
    public void bindRouting(String exchange, String queueName, String routing) throws IOException {
        if (channel == null) {
            createChannel();
        }
        channel.queueBind(queueName, exchange, routing);
    }

}
