package com.xdja.jxlsclient.util;

import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Channel;
import com.xdja.jxlsclient.bean.common.Consts;
import com.xdja.jxlsclient.handler.ReportLogRedisCacheHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Date;
import java.util.Map;
import java.util.UUID;

/**
 * @ClassName：SendMsgToRabbitMqUtil
 * @Description：发送消息到rabbitMq
 * @author: wrf
 * @date: 2019/8/16/016 12:52
 * @version: V1.0
 */
public class SendMsgToRabbitMqUtil {
    private static final Logger logger = LoggerFactory.getLogger(SendMsgToRabbitMqUtil.class);

    /**
     *  交换器名称
     */
    public static final String EXCHANGE_NAME = "jxls";
    /**
     *  队列名称
     */
    public static final String QUEUE_NAME = "to.jxls";

    /**
     * 发送mq消息到队列中
     *
     * @param routingKey 路由名称
     * @param message    消息体
     */
    public static void sendMessage(String routingKey, String message) throws IOException {
        try {
            AMQP.BasicProperties properties = new AMQP.BasicProperties.Builder().contentType("application/json").contentEncoding("UTF-8").deliveryMode(2).messageId(UUID.randomUUID().toString()).timestamp(new Date()).build();
            Channel channel = RabbitMqInit.getInstance().getChannel();
            checkQueueIsReady(routingKey);
            channel.basicPublish(EXCHANGE_NAME, routingKey, properties, message.getBytes());
        }catch (Exception e) {
            if (Consts.switchRedis) {
                ReportLogRedisCacheHandler.cacheMessage(routingKey, message);
            }else {
                throw e;
            }
        }
    }

    /**
     * 发送mq消息到队列中，并添加附加属性
     *
     * @param routingKey 路由名称
     * @param headers    附加值
     * @param message    消息体
     * @throws IOException
     */
    public static void sendMessage(String routingKey, Map<String, Object> headers, String message) throws IOException {
        try {
            Channel channel = RabbitMqInit.getInstance().getChannel();
            checkQueueIsReady(routingKey);
            AMQP.BasicProperties basicProperties = new AMQP.BasicProperties().builder().headers(headers).build();
            channel.basicPublish(EXCHANGE_NAME, routingKey, basicProperties, message.getBytes());
        }catch (IOException e) {
            if (Consts.switchRedis) {
                ReportLogRedisCacheHandler.cacheMessage(routingKey, message);
            }else {
                throw e;
            }
        }
    }

    /**
     * 检查队列关系是否已经被声明
     *
     * @return
     */
    private static void checkQueueIsReady(String routingKey) throws IOException {
        Channel channel = RabbitMqInit.getInstance().getChannel();
        //检测交换器是否存在
        try {
            channel.exchangeDeclarePassive(EXCHANGE_NAME);
        } catch (Exception e) {
            RabbitMqInit.getInstance().declareExchange(EXCHANGE_NAME,"topic");
        }

        // 检测队列是否存在
        try {
            channel.queueDeclarePassive(QUEUE_NAME);
        } catch (IOException e) {
            // 声明队列，并绑定交换器
            RabbitMqInit.getInstance().declareQueue(QUEUE_NAME);
            RabbitMqInit.getInstance().bindRouting(EXCHANGE_NAME,QUEUE_NAME,routingKey);
            return;
        }
        RabbitMqInit.getInstance().bindRouting(EXCHANGE_NAME, QUEUE_NAME,routingKey);
        logger.info("RabbitMq绑定exchange_name:【{}】, queue_name:【{}】, routingKey:{}", EXCHANGE_NAME,QUEUE_NAME, routingKey);
    }
}
