package com.xdja.lbs;

import android.annotation.SuppressLint;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.IBinder;
import androidx.annotation.RequiresPermission;

import android.provider.Settings;
import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.xdja.lbs.callback.XdjaLocationListener;
import com.xdja.lbs.location.UploadLocation;
import com.xdja.lbs.location.XLocation;
import com.xdja.lbs.provider.AMapProvider;
import com.xdja.lbs.provider.BaiduProvider;
import com.xdja.lbs.provider.BaseProvider;
import com.xdja.lbs.provider.CellProvider;
import com.xdja.lbs.provider.GpsProvider;
import com.xdja.lbs.tools.AssetsProperties;
import com.xdja.lbs.tools.PhoneTools;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static android.provider.Settings.Secure.ANDROID_ID;
import static com.xdja.lbs.config.Constants.ACTION_CHANGE_INTERVAL;
import static com.xdja.lbs.config.Constants.ACTION_LOCATION_CHANGED;
import static com.xdja.lbs.config.Constants.ACTION_START_SERVICE;
import static com.xdja.lbs.config.Constants.ACTION_STOP_SERVICE;
import static com.xdja.lbs.config.Constants.EXTRA_LOCATION;
import static com.xdja.lbs.config.Constants.EXTRA_UPLOAD_INTERVAL;

/**
 * start启动方式，该模式适用于以apk或sdk形式提供时使用，均通过intent方式传递参数
 * 位置信息以广播形式提供，缺点是使用不够灵活
 *
 * @author Guojie
 */
public class LbsService extends Service implements XdjaLocationListener {
    private final String TAG = getClass().getSimpleName();
    private ScheduledExecutorService executorService, dispatchService;
    /**
     * 上传位置信息executor
     */
    private UploadLocationExecutor locationExecutor;
    /**
     * 调度定位方式executor
     */
    private DispatchProviderExecutor dispatchExecutor;
    private BaseProvider gpsProvider, aMapProvider, bdProvider, bsProvider;
    private int interval;
    private XLocation xLocation;
    private boolean isExecutorRunning, isDispatchRunning;
    private Gson gson;
    private AssetsProperties assetsProperties;
    private LocationManager locationManager;

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override
    public void onCreate() {
        super.onCreate();
        gson = new GsonBuilder().create();
        locationManager = (LocationManager) getSystemService(Context.LOCATION_SERVICE);
        assetsProperties = new AssetsProperties(this, "config.properties");
        locationExecutor = new UploadLocationExecutor();
        dispatchExecutor = new DispatchProviderExecutor();
        gpsProvider = new GpsProvider(this,0);
        aMapProvider = new AMapProvider(this,1);
        bdProvider = new BaiduProvider(this,2);
        bsProvider = new CellProvider(this,3);
        //todo provider放入集合
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        if (intent.getAction() != null) {
            Bundle bundle = intent.getExtras();
            switch (intent.getAction()) {
                case ACTION_START_SERVICE:
                    if (bundle != null) {
                        interval = bundle.getInt(EXTRA_UPLOAD_INTERVAL, 2);
                    }
                    if (isExecutorRunning) executorService.shutdownNow();
                    if (isDispatchRunning) dispatchService.shutdownNow();
                    enableProvider(gpsProvider, true);
                    executorService = Executors.newScheduledThreadPool(5);
                    executorService.scheduleAtFixedRate(locationExecutor, 0, interval <= 2 ? 2 : interval, TimeUnit.SECONDS);
                    isExecutorRunning = true;
                    dispatchService = Executors.newSingleThreadScheduledExecutor();
                    executorService.scheduleAtFixedRate(dispatchExecutor, 2, 2, TimeUnit.SECONDS);
                    isDispatchRunning = true;
                    break;
                case ACTION_CHANGE_INTERVAL:
                    int intervalChange = intent.getIntExtra(EXTRA_UPLOAD_INTERVAL, 2);
                    if (isExecutorRunning) {
                        executorService.shutdownNow();
                        executorService = Executors.newScheduledThreadPool(5);
                        executorService.scheduleAtFixedRate(locationExecutor, 0, intervalChange <= 2 ? 2 : intervalChange, TimeUnit.SECONDS);
                        isExecutorRunning = true;
                    }
                    break;
                case ACTION_STOP_SERVICE:
                    stopService();
                    break;
                default:
                    break;
            }
        }
        return super.onStartCommand(intent, flags, startId);
    }

    /**
     * 开启及关闭定位来源
     *
     * @param baseProvider 需要启用或关闭的provider
     * @param enable       开启或关闭
     */
    private void enableProvider(BaseProvider baseProvider, boolean enable) {
        if (enable) {
            baseProvider.setLocationListener(this);
            boolean ret = baseProvider.startLocation();
            Log.i(TAG, "enableProvider 开启定位方式：" + baseProvider.getClass().getSimpleName() + " 开启结果：" + ret);
        } else {
            baseProvider.removeLocationListener();
            baseProvider.stopLocation();
            Log.i(TAG, "enableProvider 关闭定位方式：" + baseProvider.getClass().getSimpleName());
        }
    }


    @Override
    public void onLocationChanged(XLocation xLocation) {
        Log.d(TAG, "onLocationChanged,location provider:" + xLocation.getLocationSource());
        this.xLocation = xLocation;
        Intent intent = new Intent();
        intent.setAction(ACTION_LOCATION_CHANGED);
        intent.putExtra(EXTRA_LOCATION, gson.toJson(xLocation));
        sendBroadcast(intent);
    }

    @Override
    public void onLocationError(String errMessage) {
        Log.d(TAG, "onLocationError:" + errMessage);
    }

    private void stopService() {
        gpsProvider.onDestroy();
        bsProvider.onDestroy();
        aMapProvider.onDestroy();
        bdProvider.onDestroy();
        if (isExecutorRunning) {
            executorService.shutdownNow();
        }
        if (isDispatchRunning) {
            dispatchService.shutdownNow();
        }
        isExecutorRunning = false;
        isDispatchRunning = false;
        stopSelf();
    }

    private class UploadLocationExecutor implements Runnable {
        @SuppressLint("MissingPermission")
        @Override
        public void run() {
            if (xLocation != null) {
                try {
                    UploadLocation uploadLocation = new UploadLocation();
                    uploadLocation.setImei(Settings.System.getString(getContentResolver(),ANDROID_ID));
                    uploadLocation.setLon(xLocation.getLongitude());
                    uploadLocation.setLat(xLocation.getLatitude());
                    uploadLocation.setCountry(xLocation.getCountry());
                    uploadLocation.setProvince(xLocation.getProvince());
                    uploadLocation.setCity(xLocation.getCity());
                    uploadLocation.setDistrict(xLocation.getDistrict());
                    uploadLocation.setStreet(xLocation.getStreet());
                    uploadLocation.setDescription(xLocation.getAddress());
                    uploadLocation.setSource(xLocation.getLocationSource());
                    uploadLocation.setTime(xLocation.getTime());
                    String message = gson.toJson(uploadLocation);
                    Log.d(TAG, "sendStr:" + message);
                    InetAddress inetAddress = InetAddress.getByName(assetsProperties.getString("uploadHost", ""));
                    byte[] buf = message.getBytes();
                    DatagramSocket datagramSocket = new DatagramSocket();
                    DatagramPacket datagramPacket = new DatagramPacket(buf, buf.length, inetAddress, assetsProperties.getInt("uploadPort", 1080));
                    datagramSocket.send(datagramPacket);
//                    byte[] receBuf = new byte[1024];
//                    DatagramPacket recePacket = new DatagramPacket(receBuf, receBuf.length);
//                    datagramSocket.receive(recePacket);
                    datagramSocket.close();
//                    String receStr = new String(recePacket.getData(), 0, recePacket.getLength());
//                    Log.d(TAG, "receStr:" + receStr);
                } catch (SocketException e) {
                    e.printStackTrace();
                } catch (UnknownHostException e) {
                    e.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private class DispatchProviderExecutor implements Runnable {

        @Override
        public void run() {
            if (!gpsProvider.isOpen() && locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
                enableProvider(gpsProvider, true);
            }
            dispatchProvider(gpsProvider);
        }
    }

    /**
     * 递归判断
     *
     * @param checkProvider
     * @return
     */
    private BaseProvider dispatchProvider(BaseProvider checkProvider) {
        BaseProvider nextProvider = null;
        if (checkProvider instanceof GpsProvider) {
            nextProvider = bsProvider;
        } else if (checkProvider instanceof CellProvider) {
            nextProvider = bdProvider;
        } else if (checkProvider instanceof BaiduProvider) {
            nextProvider = aMapProvider;
        } else {
            return null;
        }
        if (checkProvider.isTimeOut()) {
            if (!nextProvider.isOpen()) enableProvider(nextProvider, true);
        } else {
            //关闭低优先级定位方式
            //todo provider放入集合，遍历集合关闭低优先级定位方式
            if (checkProvider instanceof GpsProvider) {
                if (bsProvider.isOpen()) enableProvider(bsProvider, false);
                if (bdProvider.isOpen()) enableProvider(bdProvider, false);
                if (aMapProvider.isOpen()) enableProvider(aMapProvider, false);
            } else if (checkProvider instanceof CellProvider) {
                if (bdProvider.isOpen()) enableProvider(bdProvider, false);
                if (aMapProvider.isOpen()) enableProvider(aMapProvider, false);
            } else if (checkProvider instanceof BaiduProvider) {
                if (aMapProvider.isOpen()) enableProvider(aMapProvider, false);
            }
            return null;
        }
        return dispatchProvider(nextProvider);
    }

}
