package com.xdja.lbs.provider;


import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Environment;

import androidx.core.app.ActivityCompat;

import android.telephony.CellIdentityCdma;
import android.telephony.CellIdentityGsm;
import android.telephony.CellIdentityLte;
import android.telephony.CellIdentityWcdma;
import android.telephony.CellInfo;
import android.telephony.CellInfoCdma;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellInfoWcdma;
import android.telephony.CellLocation;
import android.telephony.PhoneStateListener;
import android.telephony.SignalStrength;
import android.telephony.TelephonyManager;
import android.telephony.cdma.CdmaCellLocation;
import android.telephony.gsm.GsmCellLocation;
import android.text.TextUtils;
import android.util.Log;

import com.blankj.utilcode.util.FileIOUtils;
import com.blankj.utilcode.util.TimeUtils;
import com.google.gson.Gson;
import com.xdja.lbs.AppExecutors;
import com.xdja.lbs.location.BSLocation;
import com.xdja.lbs.location.BSLocationT;
import com.xdja.lbs.tools.AssetsProperties;
import com.xdja.lbs.tools.HttpRequest;
import com.xdja.lbs.transform.BSTransformerT;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static android.Manifest.permission.ACCESS_COARSE_LOCATION;
import static android.Manifest.permission.ACCESS_FINE_LOCATION;
import static android.telephony.PhoneStateListener.LISTEN_CELL_LOCATION;
import static android.telephony.PhoneStateListener.LISTEN_NONE;
import static android.telephony.TelephonyManager.NETWORK_TYPE_CDMA;
import static android.telephony.TelephonyManager.NETWORK_TYPE_LTE;
import static java.lang.Integer.MAX_VALUE;

/**
 * 提供及管理基站定位实现
 *
 * @author Guojie
 */
public class CellProvider extends BaseProvider implements IProvider {
    private TelephonyManager telephonyManager;
    /**
     * 移动设备国家代码,中国460
     */
    private String mcc;
    /**
     * 移动设备网络代码，中国移动 = 00，中国联通 = 01, 中国电信 = 11
     */
    private String mnc;
    /**
     * 网络类型，2G，3G,LTE等
     */
    private int networkType;
    /**
     * 电话类型，GSM，CDMA，NONE，SIP等
     */
    private int phoneType;
    private Gson gson;
    private AssetsProperties assetsProperties;
    private ScheduledExecutorService executorService;
    private UpdateLocationExecutor locationExecutor;
    private boolean isExecutor;
    private String serverIP;
    private int serverPort;

    public CellProvider(Context context, int priority) {
        super(context, priority);
        gson = new Gson();
        mTransformer = new BSTransformerT();
        assetsProperties = new AssetsProperties(mContext, "config.properties");
        timeoutMills = 20 * 1000;
    }

    public CellProvider(Context context, String serverIP, int serverPort) {
        super(context);
        this.serverIP = serverIP;
        this.serverPort = serverPort;
    }

    public void setServerIP(String serverIP) {
        this.serverIP = serverIP;
    }

    public void setServerPort(int serverPort) {
        this.serverPort = serverPort;
    }

    @Override
    public void initProvide() {
        telephonyManager = (TelephonyManager) mContext.getSystemService(Context.TELEPHONY_SERVICE);
        String operator = telephonyManager.getNetworkOperator();
        if (!TextUtils.isEmpty(operator)) {
            mcc = operator.substring(0, 3);
            mnc = operator.substring(3);
        }
        networkType = telephonyManager.getNetworkType();
        phoneType = telephonyManager.getPhoneType();
        executorService = Executors.newScheduledThreadPool(5);
        locationExecutor = new UpdateLocationExecutor();
        isExecutor = false;
    }

    @Override
    public boolean startLocation() {
        openTimeMills = System.currentTimeMillis();
        isOpen = true;
        if (telephonyManager == null) {
            initProvide();
        }
        if (ActivityCompat.checkSelfPermission(mContext, ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            Log.e(TAG, "ACCESS_COARSE_LOCATION permission not granted error");
            if (mXdjaLocationListener != null) {
                mXdjaLocationListener.onLocationError("开启基站定位失败：未授予相关权限");
            }
            return false;
        }
        telephonyManager.listen(phoneStateListener, LISTEN_CELL_LOCATION);
        if (!isExecutor) {
            executorService.scheduleAtFixedRate(locationExecutor, 0, 10, TimeUnit.SECONDS);
            isExecutor = true;
        }

        Log.d(TAG, getClass().getSimpleName() + " start");
        return true;
    }

    @Override
    public void stopLocation() {
        if (telephonyManager != null) {
            telephonyManager.listen(phoneStateListener, LISTEN_NONE);
        }
        telephonyManager = null;
        executorService.shutdownNow();
        isExecutor = false;
        super.stopLocation();
    }

    /**
     * 当开启基站定位、基站位置变化、基站信号强度变化、附近基站变化时请求网络获取最新的位置信息
     * <uses-permission android:name="android.permission.ACCESS_COARSE_LOCATION"/>
     * <uses-permission android:name="android.permission.ACCESS_FINE_LOCATION"/>
     */
    private void getAllCellInfo() {
        if (ActivityCompat.checkSelfPermission(mContext, ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
                && ActivityCompat.checkSelfPermission(mContext, ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            Log.e(TAG, "ACCESS_FINE_LOCATION && ACCESS_COARSE_LOCATION permission not granted error");
            return;
        }
        //获取所有基站信息
        if (telephonyManager != null) {
            List<CellInfo> infoLists = telephonyManager.getAllCellInfo();
            if (infoLists != null && infoLists.size() > 0) {

//                requestFromCellocation(getBSParametre(infoLists));
//                requestLocation(getNetworktype(), getBSParametre(infoLists));
                requestLocationFromNet(getBSParametre(infoLists));

            }
        }
    }

    @Override
    public void onDestroy() {
        if (telephonyManager != null) {
            telephonyManager.listen(phoneStateListener, LISTEN_NONE);
        }
        telephonyManager = null;
        isExecutor = false;
        executorService = null;
        locationExecutor = null;

    }

    private PhoneStateListener phoneStateListener = new PhoneStateListener() {
        @Override
        public void onCellLocationChanged(CellLocation location) {
            super.onCellLocationChanged(location);
            Log.d(TAG, "onCellLocationChanged");
            if (location instanceof GsmCellLocation) {
                GsmCellLocation gsmCellLocation = (GsmCellLocation) location;
                Log.d(TAG, "gsmCellLocation=" + gsmCellLocation.getCid());
            } else if (location instanceof CdmaCellLocation) {
                CdmaCellLocation cdmaCellLocation = (CdmaCellLocation) location;
                Log.d(TAG, "cdmaCellLocation=" + cdmaCellLocation.getBaseStationLatitude());
            }
            getAllCellInfo();
        }

        @Override
        public void onSignalStrengthsChanged(SignalStrength signalStrength) {
            super.onSignalStrengthsChanged(signalStrength);
            Log.d(TAG, "onSignalStrengthsChanged=" + signalStrength.getGsmSignalStrength());
            getAllCellInfo();
        }


        @Override
        public void onCellInfoChanged(List<CellInfo> cellInfo) {
            super.onCellInfoChanged(cellInfo);
            Log.d(TAG, "onCellInfoChanged");
            getAllCellInfo();
        }
    };


    /**
     * 获取基站信息请求参数，需要过滤掉值为{@link Integer#MAX_VALUE}的基站信息
     *
     * @param infoLists 临近基站信息
     * @return 基站定位请求参数
     */
    private String getBSParametre(List<CellInfo> infoLists) {
        StringBuilder sbFilter = new StringBuilder();
        StringBuilder sbAll = new StringBuilder();
        for (int index = 0; index < infoLists.size(); index++) {
            CellInfo cellInfo = infoLists.get(index);
            if (cellInfo instanceof CellInfoCdma) {
                CellInfoCdma cellInfoCdma = (CellInfoCdma) cellInfo;
                CellIdentityCdma cellIdentityCdma = cellInfoCdma.getCellIdentity();
                if (isAvailableCdmaCell(cellIdentityCdma.getSystemId(), cellIdentityCdma.getNetworkId(), cellIdentityCdma.getBasestationId())) {
                    if (sbFilter.length() > 0) sbFilter.append(";");
                    sbFilter.append(cellIdentityCdma.getSystemId() + ",");
                    sbFilter.append(cellIdentityCdma.getNetworkId() + ",");
                    sbFilter.append(cellIdentityCdma.getBasestationId() + ",");
                    sbFilter.append(cellInfoCdma.getCellSignalStrength().getDbm());
                }
                if (sbAll.length() > 1) sbAll.append(";");
                sbAll.append(cellIdentityCdma.getSystemId() + ",");
                sbAll.append(cellIdentityCdma.getNetworkId() + ",");
                sbAll.append(cellIdentityCdma.getBasestationId() + ",");
                sbAll.append(cellInfoCdma.getCellSignalStrength().getDbm());

            } else if (cellInfo instanceof CellInfoLte) {
                CellInfoLte cellInfoLte = (CellInfoLte) cellInfo;
                CellIdentityLte cellIdentityLte = cellInfoLte.getCellIdentity();
                if (isAvailableCell("lte", cellIdentityLte.getMcc(), cellIdentityLte.getMnc(), cellIdentityLte.getTac(), cellIdentityLte.getCi())) {
                    if (sbFilter.length() > 0) sbFilter.append(";");
                    sbFilter.append(cellIdentityLte.getMcc() + ",");
                    sbFilter.append(cellIdentityLte.getMnc() + ",");
                    sbFilter.append(cellIdentityLte.getTac() + ",");
                    sbFilter.append(cellIdentityLte.getCi() + ",");
                    sbFilter.append(cellInfoLte.getCellSignalStrength().getDbm());
                }
                if (sbAll.length() > 0) sbAll.append(";");
                sbAll.append(cellIdentityLte.getMcc() + ",");
                sbAll.append(cellIdentityLte.getMnc() + ",");
                sbAll.append(cellIdentityLte.getTac() + ",");
                sbAll.append(cellIdentityLte.getCi() + ",");
                sbAll.append(cellInfoLte.getCellSignalStrength().getDbm());

            } else if (cellInfo instanceof CellInfoGsm) {
                CellInfoGsm cellInfoGsm = (CellInfoGsm) cellInfo;
                CellIdentityGsm cellIdentityGsm = cellInfoGsm.getCellIdentity();
                if (isAvailableCell("gsm", cellIdentityGsm.getMcc(), cellIdentityGsm.getMnc(), cellIdentityGsm.getLac(), cellIdentityGsm.getCid())) {
                    if (sbFilter.length() > 0) sbFilter.append(";");
                    sbFilter.append(cellIdentityGsm.getMcc() + ",");
                    sbFilter.append(cellIdentityGsm.getMnc() + ",");
                    sbFilter.append(cellIdentityGsm.getLac() + ",");
                    sbFilter.append(cellIdentityGsm.getCid() + ",");
                    sbFilter.append(cellInfoGsm.getCellSignalStrength().getDbm());
                }
                if (sbAll.length() > 0) sbAll.append(";");
                sbAll.append(cellIdentityGsm.getMcc() + ",");
                sbAll.append(cellIdentityGsm.getMnc() + ",");
                sbAll.append(cellIdentityGsm.getLac() + ",");
                sbAll.append(cellIdentityGsm.getCid() + ",");
                sbAll.append(cellInfoGsm.getCellSignalStrength().getDbm());

            } else if (cellInfo instanceof CellInfoWcdma) {
                CellInfoWcdma cellInfoWcdma = (CellInfoWcdma) cellInfo;
                cellInfoWcdma.getCellSignalStrength();
                CellIdentityWcdma cellIdentityWcdma = cellInfoWcdma.getCellIdentity();
                if (isAvailableCell("umts", cellIdentityWcdma.getMcc(), cellIdentityWcdma.getMnc(), cellIdentityWcdma.getLac(), cellIdentityWcdma.getCid())) {
                    if (sbFilter.length() > 0) sbFilter.append(";");
                    sbFilter.append(cellIdentityWcdma.getMcc() + ",");
                    sbFilter.append(cellIdentityWcdma.getMnc() + ",");
                    sbFilter.append(cellIdentityWcdma.getLac() + ",");
                    sbFilter.append(cellIdentityWcdma.getCid() + ",");
                    sbFilter.append(cellInfoWcdma.getCellSignalStrength().getDbm());
                }
                if (sbAll.length() > 0) sbAll.append(";");
                sbAll.append(cellIdentityWcdma.getMcc() + ",");
                sbAll.append(cellIdentityWcdma.getMnc() + ",");
                sbAll.append(cellIdentityWcdma.getLac() + ",");
                sbAll.append(cellIdentityWcdma.getCid() + ",");
                sbAll.append(cellInfoWcdma.getCellSignalStrength().getDbm());

            }
        }
        FileIOUtils.writeFileFromString(mContext.getExternalFilesDir(Environment.DIRECTORY_DOCUMENTS) + "/LbsServiceBak/AllCellInfo.txt", TimeUtils.getNowString() + ": " + sbAll.toString() + "\n\r", true);
        FileIOUtils.writeFileFromString(mContext.getExternalFilesDir(Environment.DIRECTORY_DOCUMENTS) + "/LbsServiceBak/FilterCellInfo.txt", TimeUtils.getNowString() + ": " + sbFilter.toString() + "\n\r", true);
        Log.d(TAG, "filter bs=" + sbFilter.toString());
        Log.d(TAG, "all bs=" + sbAll.toString());
        return sbFilter.toString();
    }

    /**
     * 判断是否是可用的GSM、UMTS、LET基站信息
     *
     * @param netType  网络类型
     * @param mcc      移动国家码
     * @param mnc      移动网号
     * @param lacOrTac 2G和3G的lac或LTe的tac
     * @param ci       基站编号
     * @return
     */
    private boolean isAvailableCell(String netType, int mcc, int mnc, int lacOrTac, int ci) {
        if (mcc == MAX_VALUE || mnc == MAX_VALUE) {
            return false;
        }
        //5G 范围1~16777215，暂不考虑
        if (lacOrTac < 1 || lacOrTac > 65535) {
            return false;
        }
        if (netType.equals("gsm")) {
            if (ci < 1 || ci > 65525) {
                return false;
            }
        } else {
            if (ci < 1 || ci > 268435455) {
                return false;
            }
        }
        return true;
    }

    /**
     * 判断是否是可用的cdma基站
     *
     * @param sid sid
     * @param nid nid
     * @param bid bid
     * @return
     */
    private boolean isAvailableCdmaCell(int sid, int nid, int bid) {
        if (sid > 32767) {
            return false;
        }
        if (nid > 65535) {
            return false;
        }
        if (bid < 1 || bid > 65535) {
            return false;
        }
        return true;
    }

    /**
     * 请求位置信息时若网络类型为cdma时必填，其他类型选填
     * 值是gsm、umts、lte、cdma其中之一，
     * gsm：移动、联通2G
     * umts：移动、联通3G
     * lte:4G
     * cdma：电信
     *
     * @return 网络类型
     */
    private String getNetworktype() {
        String type = "";
        switch (telephonyManager.getNetworkType()) {
            //CDMA
            case NETWORK_TYPE_CDMA:
                type = "cdma";
                break;
            //4G网络
            case NETWORK_TYPE_LTE:
                type = "lte";
                break;
            // 3G网络
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
            case TelephonyManager.NETWORK_TYPE_UMTS:
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
            case TelephonyManager.NETWORK_TYPE_HSDPA:
            case TelephonyManager.NETWORK_TYPE_HSUPA:
            case TelephonyManager.NETWORK_TYPE_HSPA:
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
            case TelephonyManager.NETWORK_TYPE_EHRPD:
            case TelephonyManager.NETWORK_TYPE_HSPAP:
                type = "umts";
                break;
            default:
                type = "gsm";
                break;
        }
        Log.d(TAG, "network type:" + type);
        return type;
    }

    private void requestLocation(String type, String bs) {
        Map<String, Object> map = new HashMap<>();
        map.put("oid", assetsProperties.getString("oid", ""));
        map.put("key", assetsProperties.getString("key", ""));
        map.put("type", type);
        map.put("bs", bs);
        map.put("hex", "10");
        map.put("to", "2");
        map.put("output", "json");
        AppExecutors.getInstance().getNetworkIO().execute(() -> {
            String response = "";
            try {
                response = HttpRequest.post("http://api.gpsspg.com/bs/", map);
                if (!TextUtils.isEmpty(response)) {
                    BSLocation bsLocation = gson.fromJson(response, BSLocation.class);
                    boolean isLocateSuccess = bsLocation.getStatus() == 200;
                    if (mXdjaLocationListener != null) {
                        if (isLocateSuccess) {
                            if (bsLocation.getResult().size() > 0) {
                                lastUpdateTimeMills = System.currentTimeMillis();
                                xLocation = mTransformer.transform(bsLocation);
                                mXdjaLocationListener.onLocationChanged(xLocation);
                            } else {
                                mXdjaLocationListener.onLocationError("基站定位失败：" + bsLocation.getMsg());
                            }
                        } else {
                            mXdjaLocationListener.onLocationError("基站定位失败：" + bsLocation.getMsg());
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
                mXdjaLocationListener.onLocationError("基站定位失败：" + e.getMessage());
            }
            StringBuilder sb = new StringBuilder();
            sb.append(TimeUtils.getNowString() + ": \n\r");
            sb.append("request:" + bs + "\n\r");
            sb.append("response:" + response + "\n\r");
            FileIOUtils.writeFileFromString(mContext.getExternalFilesDir(Environment.DIRECTORY_DOCUMENTS) + "/LbsServiceBak/Location.txt", sb.toString(), true);

        });
    }


    private void requestLocationFromNet(String bs) {
        AppExecutors.getInstance().getNetworkIO().execute(() -> {
            String response = "";
            try {
//                response = HttpRequest.post("http://192.168.24.27:3210/gls/api/bs_location.do", bs);
                response = HttpRequest.post(String.format("http://%s:%d/gls/api/bs_location.do", serverIP, serverPort), bs);
                if (!TextUtils.isEmpty(response)) {
                    BSLocationT bsLocationT = gson.fromJson(response, BSLocationT.class);
                    String flag = bsLocationT.getFlag();
                    if (mXdjaLocationListener != null) {
                        if (TextUtils.equals("1", flag)) {
                            BSLocationT.Data data = bsLocationT.getData();
                            if (data != null) {
                                lastUpdateTimeMills = System.currentTimeMillis();
                                xLocation = mTransformer.transform(data);
                                if (mXdjaLocationListener != null)
                                    mXdjaLocationListener.onLocationChanged(xLocation);
                            }
                        } else {
                            mXdjaLocationListener.onLocationError("基站定位失败：" + bsLocationT.getMessage());
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
                mXdjaLocationListener.onLocationError("基站定位失败：" + e.getMessage());
            }
            StringBuilder sb = new StringBuilder();
            sb.append(TimeUtils.getNowString() + ": \n\r");
            sb.append("request:" + bs + "\n\r");
            sb.append("response:" + response + "\n\r");
            FileIOUtils.writeFileFromString(mContext.getExternalFilesDir(Environment.DIRECTORY_DOCUMENTS) + "/LbsServiceBak/Location.txt", sb.toString(), true);
        });
    }

    private void requestFromCellocation(String cl) {

        Map<String, Object> map = new HashMap<>();
        map.put("cl", cl.replace("\\|", ";"));
        AppExecutors.getInstance().getNetworkIO().execute(() -> {
                    String response = "";
                    try {
                        response = HttpRequest.post("http://api.cellocation.com:81/loc/", map);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    StringBuilder sb = new StringBuilder();
                    sb.append(TimeUtils.getNowString() + ": \n\r");
                    sb.append("request:" + cl + "\n\r");
                    sb.append("response:" + response + "\n\r");
                    FileIOUtils.writeFileFromString(mContext.getExternalFilesDir(Environment.DIRECTORY_DOCUMENTS) + "/LbsServiceBak/Cellocation.txt", sb.toString(), true);
                }
        );
    }

    private class UpdateLocationExecutor implements Runnable {
        @Override
        public void run() {
            getAllCellInfo();
        }
    }
}
