package com.xdja.lbs.provider;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.location.Criteria;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;

import android.util.Log;

import com.xdja.lbs.transform.NativeTransformer;


/**
 * 提供及管理GPS定位方式实现
 *
 * @author Guojie
 */
public class GpsProvider extends BaseProvider implements IProvider, LocationListener {

    private LocationManager locationManager;
    private Criteria criteria;

    public GpsProvider(@NonNull Context context, int priority) {
        super(context, priority);
        mTransformer = new NativeTransformer();
        timeoutMills = 2 * 60 * 1000;
//        timeoutMills = 30 * 1000;
    }

    @Override
    public void initProvide() {
        criteria = new Criteria();
        // Criteria是一组筛选条件
        criteria.setAccuracy(Criteria.ACCURACY_FINE);
        //设置定位精准度
        criteria.setAltitudeRequired(false);
        //是否要求海拔
        criteria.setBearingRequired(true);
        //是否要求方向
        criteria.setCostAllowed(true);
        //是否要求收费
        criteria.setSpeedRequired(true);
        //是否要求速度
        criteria.setPowerRequirement(Criteria.NO_REQUIREMENT);
        //设置电池耗电要求
        criteria.setBearingAccuracy(Criteria.ACCURACY_HIGH);
        //设置方向精确度
        criteria.setSpeedAccuracy(Criteria.ACCURACY_HIGH);
        //设置速度精确度
        criteria.setHorizontalAccuracy(Criteria.ACCURACY_HIGH);
        //设置水平方向精确度
        criteria.setVerticalAccuracy(Criteria.ACCURACY_HIGH);
        //设置垂直方向精确度
        criteria.setBearingAccuracy(Criteria.ACCURACY_HIGH);
        locationManager = (LocationManager) mContext.getSystemService(Context.LOCATION_SERVICE);
    }

    /**
     * GPS定位方式优先级最高，为避免GPS开关未打开开启服务时影响后续流程
     * 故不再调用super方法，修改isOpen和openTimemills初始化逻辑
     */
    @Override
    public boolean startLocation() {
        if (openTimeMills == 0) {
            openTimeMills = System.currentTimeMillis();
        }
        if (locationManager == null) {
            initProvide();
        }
        if (!locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
            Log.d(TAG, "GPS disable error");
            if (mXdjaLocationListener != null) {
                mXdjaLocationListener.onLocationError("开启GPS定位失败，失败原因：GPS未开启");
            }
            return false;
        }
        String provider = locationManager.getBestProvider(criteria, true);
        if (ActivityCompat.checkSelfPermission(mContext, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(mContext, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            Log.e(TAG, "ACCESS_FINE_LOCATION permission not granted error");
            if (mXdjaLocationListener != null) {
                mXdjaLocationListener.onLocationError("开启GPS定位失败，失败原因：未授权定位权限");
            }
            return false;
        }
        locationManager.requestLocationUpdates(provider, 5 * 1000, 0, this);
        isOpen = true;
        Log.d(TAG, getClass().getSimpleName() + " start");
        return true;
    }

    @Override
    public void stopLocation() {
        if (locationManager != null) {
            locationManager.removeUpdates(this);
        }
        super.stopLocation();
    }

    /**
     * 因GPS定位优先级最高，为避免GPS开关未打开开启服务时影响后续流程
     *
     * @return 故重写超时方法
     */
    @Override
    public boolean isTimeOut() {
        if (xLocation == null) {
            if (System.currentTimeMillis() - openTimeMills > timeoutMills) {
                return true;
            }
        } else {
            if (System.currentTimeMillis() - lastUpdateTimeMills > timeoutMills) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void onDestroy() {
        stopLocation();
        criteria = null;
        locationManager = null;
    }

    @Override
    public void onLocationChanged(Location location) {
        if (mXdjaLocationListener != null) {
            lastUpdateTimeMills = System.currentTimeMillis();
            xLocation = mTransformer.transform(location);
            mXdjaLocationListener.onLocationChanged(xLocation);
        }
    }

    @Override
    public void onStatusChanged(String provider, int status, Bundle extras) {

    }

    @Override
    public void onProviderEnabled(String provider) {
        Log.d(TAG, "provider enbale:" + provider);
    }

    @Override
    public void onProviderDisabled(String provider) {
        Log.d(TAG, "provider disabled:" + provider);
    }
}
