package com.xdja.lbs;

import android.annotation.SuppressLint;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.xdja.lbs.callback.XdjaLocationListener;
import com.xdja.lbs.location.UploadLocation;
import com.xdja.lbs.location.XLocation;
import com.xdja.lbs.provider.AMapProvider;
import com.xdja.lbs.provider.BaiduProvider;
import com.xdja.lbs.provider.BaseProvider;
import com.xdja.lbs.provider.CellProvider;
import com.xdja.lbs.provider.GpsProvider;
import com.xdja.lbs.tools.PhoneTools;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import static com.xdja.lbs.config.Constants.ACTION_CHANGE_INTERVAL;
import static com.xdja.lbs.config.Constants.ACTION_LOCATION_CHANGED;
import static com.xdja.lbs.config.Constants.ACTION_START_SERVICE;
import static com.xdja.lbs.config.Constants.ACTION_STOP_SERVICE;
import static com.xdja.lbs.config.Constants.EXTRA_LOCATION;
import static com.xdja.lbs.config.Constants.EXTRA_UPLOAD_INTERVAL;
import static com.xdja.lbs.config.Constants.HandlerMessage.MESSAGE_CHANGE_UPLOAD_INTERVAL;
import static com.xdja.lbs.config.Constants.HandlerMessage.MESSAGE_LOCATION_ERROR;
import static com.xdja.lbs.config.Constants.HandlerMessage.MESSAGE_LOCATION_UPDATE;
import static com.xdja.lbs.config.Constants.HandlerMessage.MESSAGE_SERVICE_CONNECTION;

/**
 * @author Guojie
 */
public class XdjaLbsService extends Service implements XdjaLocationListener {
    private final String TAG = getClass().getSimpleName();
    private ScheduledExecutorService scheduledExecutorService;
    private ScheduledFuture uploadExecutorFuture, dispatchExecutorFuture;
    /**
     * 上传位置信息executor
     */
    private UploadLocationExecutor locationExecutor;
    /**
     * 调度定位方式executor
     */
    private DispatchProviderExecutor dispatchExecutor;
    private BaseProvider gpsProvider, aMapProvider, bdProvider, cellProvider;
    private int interval;
    private XLocation xLocation;
    private boolean isExecutorRunning, isDispatchRunning;
    private Gson gson;
    private LocationManager locationManager;
    private boolean isServing = false;
    private Messenger messenger;
    private Messenger messengerClient;
    private ServerHandler handler;
    private XdjaLocationClientOption option;
    private Map<Integer, BaseProvider> providerMap;

    @Override
    public void onCreate() {
        Log.d(TAG, "onCreate");
        super.onCreate();
        if (isServing) {
            Log.d(TAG, "location service can not start again");
        } else {
            gson = new GsonBuilder().create();
            locationManager = (LocationManager) getSystemService(Context.LOCATION_SERVICE);
            handler = new ServerHandler(Looper.getMainLooper(), this);
            messenger = new Messenger(handler);
            locationExecutor = new UploadLocationExecutor();
            dispatchExecutor = new DispatchProviderExecutor();
            gpsProvider = new GpsProvider(this, 0);
            cellProvider = new CellProvider(this, 1);
            bdProvider = new BaiduProvider(this, 2);
            aMapProvider = new AMapProvider(this, 3);
            providerMap = new HashMap<>();
            providerMap.put(gpsProvider.getPriority(), gpsProvider);
            providerMap.put(cellProvider.getPriority(), cellProvider);
            providerMap.put(bdProvider.getPriority(), bdProvider);
            providerMap.put(aMapProvider.getPriority(), aMapProvider);
            isServing = true;
        }
    }

    @Override
    public IBinder onBind(Intent intent) {
        Log.d(TAG, "onBind");
        return messenger.getBinder();
    }

    @Override
    public void onRebind(Intent intent) {
        Log.d(TAG, "onRebind");
        super.onRebind(intent);
    }

    @Override
    public boolean onUnbind(Intent intent) {
        Log.d(TAG, "onUnbind");
        stopService();
        return super.onUnbind(intent);
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        Log.d(TAG, "onStartCommand");
        if (intent == null || intent.getAction() == null) {
            return super.onStartCommand(intent, flags, startId);
        }
        if (intent.getAction() != null) {
            Bundle bundle = intent.getExtras();
            switch (intent.getAction()) {
                case ACTION_START_SERVICE:
                    if (bundle != null) {
                        interval = bundle.getInt(EXTRA_UPLOAD_INTERVAL, 2);
                    }
                    if (isExecutorRunning) uploadExecutorFuture.cancel(true);
                    if (isDispatchRunning) dispatchExecutorFuture.cancel(true);
                    enableProvider(gpsProvider, true);
                    scheduledExecutorService = LbsExecutors.getInstance().getMultiTaskScheduleExecutorService();
                    uploadExecutorFuture = scheduledExecutorService.scheduleAtFixedRate(locationExecutor, 0, interval <= 2 ? 2 : interval, TimeUnit.SECONDS);
                    isExecutorRunning = true;
                    dispatchExecutorFuture = scheduledExecutorService.scheduleAtFixedRate(dispatchExecutor, 2, 2, TimeUnit.SECONDS);
                    isDispatchRunning = true;
                    break;
                case ACTION_CHANGE_INTERVAL:
                    int intervalChange = intent.getIntExtra(EXTRA_UPLOAD_INTERVAL, 2);
                    if (isExecutorRunning) {
                        uploadExecutorFuture.cancel(true);
                        uploadExecutorFuture = scheduledExecutorService.scheduleAtFixedRate(locationExecutor, 0, intervalChange <= 2 ? 2 : intervalChange, TimeUnit.SECONDS);
                        isExecutorRunning = true;
                    }
                    break;
                case ACTION_STOP_SERVICE:
                    stopService();
                    break;
                default:
                    break;
            }
        }
        return super.onStartCommand(intent, flags, startId);
    }

    /**
     * 设置定位参数
     *
     * @param option 参数配置
     */
    public void setOption(XdjaLocationClientOption option) {
        if (option != null) {
            this.option = option;
            CellProvider cellProvider = (CellProvider) this.cellProvider;
            cellProvider.setServerIP(option.getCellLocationIp());
            cellProvider.setServerPort(option.getCellLocationPort());
            gpsProvider.setTimeoutMills(option.getGpsTimeout() == 0 ? 2 * 60 * 1000 : option.getGpsTimeout());
        }
    }

    public void startLocation() {
        enableProvider(gpsProvider, true);
        //调度定位方式线程
        if (isDispatchRunning) dispatchExecutorFuture.cancel(true);
        scheduledExecutorService = LbsExecutors.getInstance().getSingleScheduleExecutorService();
        dispatchExecutorFuture = scheduledExecutorService.scheduleAtFixedRate(dispatchExecutor, 2, 2, TimeUnit.SECONDS);
        isDispatchRunning = true;
        if (option.isUploadLocation()) {
            //上传定位线程
            interval = option.getUploadInterval();
            if (isExecutorRunning) uploadExecutorFuture.cancel(true);
            uploadExecutorFuture = scheduledExecutorService.scheduleAtFixedRate(locationExecutor, 0, interval <= 2 ? 2 : interval, TimeUnit.SECONDS);
            isExecutorRunning = true;
        }
    }

    public void changeUploadInterval() {
        if (isExecutorRunning) {
            interval = option.getUploadInterval();
            uploadExecutorFuture.cancel(true);
            scheduledExecutorService = LbsExecutors.getInstance().getMultiTaskScheduleExecutorService();
            uploadExecutorFuture = scheduledExecutorService.scheduleAtFixedRate(locationExecutor, 0, interval <= 2 ? 2 : interval, TimeUnit.SECONDS);
            isExecutorRunning = true;
        }
    }

    /**
     * 开启及关闭定位来源
     *
     * @param baseProvider 需要启用或关闭的provider
     * @param enable       开启或关闭
     */
    private void enableProvider(BaseProvider baseProvider, boolean enable) {
        if (enable) {
            baseProvider.setLocationListener(this);
            boolean ret = baseProvider.startLocation();
            Log.i(TAG, "enableProvider 开启定位方式：" + baseProvider.getClass().getSimpleName() + " 开启结果：" + ret);
        } else {
            baseProvider.stopLocation();
            baseProvider.removeLocationListener();
            Log.i(TAG, "enableProvider 关闭定位方式：" + baseProvider.getClass().getSimpleName());
        }
    }


    @Override
    public void onLocationChanged(XLocation xLocation) {
        Log.d(TAG, "onLocationChanged,location provider:" + xLocation.getLocationSource());
        this.xLocation = xLocation;
        Intent intent = new Intent();
        intent.setAction(ACTION_LOCATION_CHANGED);
        intent.putExtra(EXTRA_LOCATION, gson.toJson(xLocation));
        sendBroadcast(intent);
        try {
            if (handler != null && messengerClient != null) {
                Message message = handler.obtainMessage(MESSAGE_LOCATION_UPDATE);
                message.obj = xLocation;
                messengerClient.send(message);
            }
        } catch (RemoteException e) {
            e.printStackTrace();
        }

    }

    @Override
    public void onLocationError(String errMessage) {
        Log.d(TAG, "onLocationError:" + errMessage);
        try {
            Message message = handler.obtainMessage(MESSAGE_LOCATION_ERROR);
            message.obj = errMessage;
            if (messengerClient != null)
                messengerClient.send(message);
        } catch (RemoteException e) {
            e.printStackTrace();
        }
    }

    private void stopService() {
        gpsProvider.onDestroy();
        cellProvider.onDestroy();
        aMapProvider.onDestroy();
        bdProvider.onDestroy();
        if (isExecutorRunning) {
            uploadExecutorFuture.cancel(true);
            isExecutorRunning = false;
        }
        if (isDispatchRunning) {
            dispatchExecutorFuture.cancel(true);
            isDispatchRunning = false;
        }
        if (handler != null) {
            handler.removeCallbacksAndMessages(null);
        }
        LbsExecutors.getInstance().terminated();
        dispatchExecutor = null;
        locationExecutor = null;
        handler = null;
        isServing = false;
        stopSelf();
    }

    @Override
    public void onDestroy() {
        Log.d(TAG, "onDestroy");
        super.onDestroy();
    }

    private class UploadLocationExecutor implements Runnable {
        @SuppressLint("MissingPermission")
        @Override
        public void run() {
            if (xLocation != null) {
                try {
                    UploadLocation uploadLocation = new UploadLocation();
                    uploadLocation.setImei(PhoneTools.getIMEI(XdjaLbsService.this));
                    uploadLocation.setLon(xLocation.getLongitude());
                    uploadLocation.setLat(xLocation.getLatitude());
                    uploadLocation.setCountry(xLocation.getCountry());
                    uploadLocation.setProvince(xLocation.getProvince());
                    uploadLocation.setCity(xLocation.getCity());
                    uploadLocation.setDistrict(xLocation.getDistrict());
                    uploadLocation.setStreet(xLocation.getStreet());
                    uploadLocation.setDescription(xLocation.getDescription());
                    uploadLocation.setAddress(xLocation.getAddress());
                    uploadLocation.setSource(xLocation.getLocationSource());
                    uploadLocation.setTime(xLocation.getTime());
                    String message = gson.toJson(uploadLocation);
                    Log.d(TAG, "sendStr:" + message);
                    InetAddress inetAddress = InetAddress.getByName(option.getUploadIp());
                    byte[] buf = message.getBytes();
                    DatagramSocket datagramSocket = new DatagramSocket();
                    DatagramPacket datagramPacket = new DatagramPacket(buf, buf.length, inetAddress, option.getUploadPort());
                    datagramSocket.send(datagramPacket);
//                    byte[] receBuf = new byte[1024];
//                    DatagramPacket recePacket = new DatagramPacket(receBuf, receBuf.length);
//                    datagramSocket.receive(recePacket);
                    datagramSocket.close();
//                    String receStr = new String(recePacket.getData(), 0, recePacket.getLength());
//                    Log.d(TAG, "receStr:" + receStr);
                } catch (SocketException e) {
                    e.printStackTrace();
                } catch (UnknownHostException e) {
                    e.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private class DispatchProviderExecutor implements Runnable {

        @Override
        public void run() {
            Log.d(TAG, "thread name = " + Thread.currentThread().getName());
            if (!gpsProvider.isOpen() && locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
                enableProvider(gpsProvider, true);
            }
            dispatchProvider(gpsProvider);
        }
    }

    /**
     * 递归判断
     *
     * @param checkProvider 被检查是否超时的定位提供方式
     * @return
     */
    private void dispatchProvider(BaseProvider checkProvider) {
        Log.d(TAG, "checkProvider=" + checkProvider.getClass().getSimpleName());
        if (checkProvider.getPriority() > 2) {
            return;
        }
        if (checkProvider.isTimeOut()) {
            BaseProvider nextProvider = providerMap.get(checkProvider.getPriority() + 1);
            if (!nextProvider.isOpen()) enableProvider(nextProvider, true);
            dispatchProvider(nextProvider);
        } else {
            //关闭低优先级定位方式
            Iterator<Map.Entry<Integer, BaseProvider>> iterator = providerMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<Integer, BaseProvider> entry = iterator.next();
                BaseProvider baseProvider = entry.getValue();
                if (baseProvider.getPriority() > checkProvider.getPriority() && baseProvider.isOpen())
                    enableProvider(baseProvider, false);
            }
            return;
        }
        return;
    }

    private static class ServerHandler extends Handler {
        private final WeakReference<XdjaLbsService> serviceWeakReference;

        public ServerHandler(Looper looper, XdjaLbsService service) {
            super(looper);
            this.serviceWeakReference = new WeakReference(service);
        }

        @Override
        public void handleMessage(Message msg) {
            XdjaLbsService xdjaLbsService = serviceWeakReference.get();
            switch (msg.what) {
                case MESSAGE_SERVICE_CONNECTION:
                    if (xdjaLbsService != null) {
                        xdjaLbsService.setOption((XdjaLocationClientOption) msg.obj);
                        xdjaLbsService.messengerClient = msg.replyTo;
                        xdjaLbsService.startLocation();
                    }
                    break;
                case MESSAGE_CHANGE_UPLOAD_INTERVAL:
                    if (xdjaLbsService != null) {
                        xdjaLbsService.setOption((XdjaLocationClientOption) msg.obj);
                        xdjaLbsService.changeUploadInterval();
                    }
                    break;
                default:
                    break;
            }
        }
    }
}
