package com.xdja.lbs;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.util.Log;

import com.xdja.lbs.callback.XdjaLocationListener;
import com.xdja.lbs.location.XLocation;
import com.xdja.lbs.tools.AssetsProperties;

import java.lang.ref.WeakReference;

import static com.xdja.lbs.config.Constants.HandlerMessage.MESSAGE_CHANGE_UPLOAD_INTERVAL;
import static com.xdja.lbs.config.Constants.HandlerMessage.MESSAGE_LOCATION_ERROR;
import static com.xdja.lbs.config.Constants.HandlerMessage.MESSAGE_LOCATION_UPDATE;
import static com.xdja.lbs.config.Constants.HandlerMessage.MESSAGE_SERVICE_CONNECTION;

/**
 * @author Guojie
 */
public class XdjaLocationClient {
    private final String TAG = getClass().getSimpleName();
    private @NonNull
    Context context;
    private XdjaLocationClientOption clientOption;
    private XdjaLocationListener xdjaLocationListener;
    private LocationHandler handler;
    private Messenger messengerServer;
    private Messenger messengerClient;
    private boolean isBind;
    private AssetsProperties assetsProperties;
    private ServiceConnection serviceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName componentName, IBinder iBinder) {
            Log.d(TAG, "onServiceConnected");
            messengerServer = new Messenger(iBinder);
            Message message = Message.obtain();
            message.what = MESSAGE_SERVICE_CONNECTION;
            message.obj = clientOption;
            message.replyTo = messengerClient;
            try {
                messengerServer.send(message);
            } catch (RemoteException e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onServiceDisconnected(ComponentName componentName) {
            Log.d(TAG, "onServiceDisconnected");
            isBind = false;
        }
    };

    public XdjaLocationClient(@NonNull Context context) {
        if (context == null) {
            throw new IllegalArgumentException("Context参数不能为null");
        } else {
            this.context = context.getApplicationContext();
            this.assetsProperties = new AssetsProperties(context, "config.properties");
            this.clientOption = new XdjaLocationClientOption();
            this.handler = new LocationHandler(Looper.getMainLooper(), this);
            this.messengerClient = new Messenger(handler);
            initClientOption();
        }
        this.context = context;
    }

    public void setLocationOption(XdjaLocationClientOption clientOption) {
        if (clientOption == null) {
            throw new IllegalArgumentException("XdjaLocationClientOption 参数不能为null");
        } else {
            this.clientOption = clientOption;
        }
    }

    public XdjaLocationClientOption getLocationOption() {
        return this.clientOption;
    }

    private void initClientOption() {
        clientOption.setCellLocationIp(assetsProperties.getString("cellLocationIP", "127.0.0.1"));
        clientOption.setCellLocationPort(assetsProperties.getInt("cellLocationPort", 1080));
        clientOption.setUploadIp(assetsProperties.getString("uploadHost", "127.0.0.1"));
        clientOption.setUploadPort(assetsProperties.getInt("uploadPort", 1080));
        clientOption.setUploadInterval(2);
        clientOption.setUploadLocation(false);
    }

    public void changeUploadInterval(int seconds) {
        if (clientOption.isUploadLocation()) {
            clientOption.setUploadInterval(seconds);
            Message message = Message.obtain();
            message.what = MESSAGE_CHANGE_UPLOAD_INTERVAL;
            message.obj = clientOption;
            try {
                if (messengerServer != null) {
                    messengerServer.send(message);
                }
            } catch (RemoteException e) {
                e.printStackTrace();
            }
        }
    }

    public void startLocation() {
        if (!isBind) {
            this.handler = new LocationHandler(Looper.getMainLooper(), this);
            this.messengerClient = new Messenger(handler);
            Intent intent = new Intent(context, XdjaLbsService.class);
            isBind = context.bindService(intent, serviceConnection, Context.BIND_AUTO_CREATE);
            Log.d(TAG, "isBind=" + isBind);
        }
    }

    public void stopLocation() {
        if (isBind) {
            try {
                if (handler != null) {
                    handler.removeCallbacksAndMessages(null);
                    handler = null;
                }
                context.unbindService(serviceConnection);
            } catch (Exception e) {
                e.printStackTrace();
            }
            isBind = false;
        }

    }

    public void setLocationListener(XdjaLocationListener locationListener) {
        if (locationListener != null) {
            this.xdjaLocationListener = locationListener;
        }
    }

    public void unRegisterLocationListener() {
        xdjaLocationListener = null;
    }

    public void updateLocation(XLocation xLocation) {
        if (xdjaLocationListener != null) {
            xdjaLocationListener.onLocationChanged(xLocation);
        }
    }

    public void locationError(String message) {
        if (xdjaLocationListener != null) {
            xdjaLocationListener.onLocationError(message);
        }
    }

    private static class LocationHandler extends Handler {
        private final WeakReference<XdjaLocationClient> clientWeakReference;

        public LocationHandler(Looper looper, XdjaLocationClient client) {
            super(looper);
            this.clientWeakReference = new WeakReference(client);
        }

        @Override
        public void handleMessage(Message msg) {
            XdjaLocationClient xdjaLocationClient = clientWeakReference.get();
            switch (msg.what) {
                case MESSAGE_LOCATION_UPDATE:
                    if (xdjaLocationClient != null) {
                        xdjaLocationClient.updateLocation((XLocation) msg.obj);
                    }
                    break;
                case MESSAGE_LOCATION_ERROR:
                    if (xdjaLocationClient != null) {
                        xdjaLocationClient.locationError((String) msg.obj);
                    }
                    break;
                default:
                    break;
            }

        }
    }

}
