package com.xdja.lbs;

import android.os.Handler;
import android.os.Looper;

import com.xdja.lbs.tools.ThreadPoolManager;

import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

/**
 * 管理Lbs全局线程池
 *
 * @author Guojie
 */
public class LbsExecutors {
    /**
     * UI线程
     **/
    private Executor mainThread;
    /**
     * 磁盘IO线程池
     **/
    private ExecutorService diskIO;

    /**
     * 网络IO线程池
     **/
    private ExecutorService networkIO;
    /**
     * 单线程定时任务线程池
     */
    private ScheduledExecutorService singleScheduleExecutorService;
    /**
     * 多线程定时任务线程池
     */
    private ScheduledExecutorService multiTaskScheduleExecutorService;

    protected static LbsExecutors lbsExecutors;

    public static LbsExecutors getInstance() {
        if (lbsExecutors == null) {
            synchronized (LbsExecutors.class) {
                if (lbsExecutors == null) {
                    lbsExecutors = new LbsExecutors();
                }
            }
        }
        return lbsExecutors;
    }

    private LbsExecutors() {

    }

    /**
     * 磁盘IO线程池（单线程）
     * <p>
     * 和磁盘操作有关的进行使用此线程(如读写数据库,读写文件)
     * 禁止延迟,避免等待
     * 此线程不用考虑同步问题
     */
    public ExecutorService getDiskIO() {
        if (diskIO == null) {
            diskIO = ThreadPoolManager.newSingleThreadExecutor();
        }
        return diskIO;
    }

    /**
     * 网络IO线程池
     * <p>
     * 网络请求,异步任务等适用此线程
     * 不建议在这个线程 sleep 或者 wait
     */
    public ExecutorService getNetworkIO() {
        if (networkIO == null) {
            networkIO = ThreadPoolManager.newFixedThreadPool(5);
        }
        return networkIO;
    }

    public ScheduledExecutorService getSingleScheduleExecutorService() {
        if (singleScheduleExecutorService == null) {
            singleScheduleExecutorService = ThreadPoolManager.newSingleThreadScheduledExecutor();
        }
        return singleScheduleExecutorService;
    }

    public ScheduledExecutorService getMultiTaskScheduleExecutorService() {
        if (multiTaskScheduleExecutorService == null) {
            multiTaskScheduleExecutorService = ThreadPoolManager.newScheduledThreadPool(5);
        }
        return multiTaskScheduleExecutorService;
    }

    /**
     * UI线程
     * <p>
     * Android 的MainThread
     * UI线程不能做的事情这个都不能做
     */
    public Executor getMainThread() {
        if (mainThread == null) {
            mainThread = new MainThreadHandler();
        }
        return mainThread;
    }

    /**
     * 释放lbs所有线程
     */
    public void terminated() {
        if (networkIO != null && !networkIO.isShutdown()) {
            networkIO.shutdown();
            networkIO = null;
        }
        if (singleScheduleExecutorService != null && !singleScheduleExecutorService.isShutdown()) {
            singleScheduleExecutorService.shutdownNow();
            singleScheduleExecutorService = null;
        }
        if (multiTaskScheduleExecutorService != null && !multiTaskScheduleExecutorService.isShutdown()) {
            multiTaskScheduleExecutorService.shutdownNow();
            multiTaskScheduleExecutorService = null;
        }
        if (diskIO != null && !diskIO.isShutdown()) {
            diskIO.shutdown();
            diskIO = null;
        }
    }

    private static class MainThreadHandler implements Executor {
        private Handler mainHandler = new Handler(Looper.getMainLooper());

        @Override
        public void execute(Runnable command) {
            mainHandler.post(command);
        }
    }
}
