package com.xdja.log;

import android.content.Context;
import android.text.TextUtils;

import com.xdja.log.store.Store;

/**
 * 日志配置模块
 * Created by hyh on 2021/1/14
 */
public class LogConfiguration implements Cloneable {
    private static final String KEY_LOG_LEVEL = "logLevel";
    private static final String KEY_WRITE_LOG_TO_FILE = "writeLogToFile";
    private static final String KEY_PRINT_CONSOLE = "printConsole";
    private static final String KEY_COLLECT_CRASH = "collectCrash";
    private static final String KEY_ROOT_PATH = "rootPath";
    private static final String ROOT_PATH_NAME = "XDJALOG";
    public static final String LOG_NORMAL = "normal";
    public static final String LOG_CRASH = "crash";
    public static final String LOG_LOGCAT = "logcat";


    public Context context;
    /**
     * 日志级别。
     * 参看{@link LogLevel}。
     * 默认{@link LogLevel#ALL}
     */
    public int logLevel;
    /**
     * 保存日志根目录。
     * 默认路径:context.getExternalFilesDir( {@link LogConfiguration#ROOT_PATH_NAME} ).getAbsolutePath()
     */
    public String rootPath;
    /**
     * 是否拦截crash日志。
     * 默认:false
     */
    public boolean enableCollectCrash;
    /**
     * 是否写文件。
     * 默认:false
     */
    public boolean allowedWriteFile;
    /**
     * 是否写到控制台。
     * 默认: true
     */
    public boolean allowedPrintConsole;
    /**
     * 策略是否是一次性的。
     * 当为false时，表示调用{@link XdjaLog#updateLogConfiguration(LogConfiguration)}时，配置不持久化。
     * 持久化的目的是，应用进程结束后，再重新启动时，如果应用调用的是{@link XdjaLog#init(Context)}，
     * 则会根据上此调用{@link XdjaLog#updateLogConfiguration(LogConfiguration)}进行配置。
     * 默认: true
     */
    public boolean once;


    public LogConfiguration(Builder builder) {
        context = builder.context;
        logLevel = builder.logLevel;
        if (TextUtils.isEmpty(builder.rootPath)) {
            rootPath = context.getExternalFilesDir(ROOT_PATH_NAME).getAbsolutePath();
        } else {
            rootPath = builder.rootPath;
        }
        enableCollectCrash = builder.enableCollectCrash;
        allowedWriteFile = builder.allowedWriteFile;
        allowedPrintConsole = builder.allowedPrintConsole;
        once = builder.once;
    }

    void store() {
        Store.put(context, KEY_LOG_LEVEL, logLevel);
        Store.put(context, KEY_WRITE_LOG_TO_FILE, allowedWriteFile);
        Store.put(context, KEY_PRINT_CONSOLE, allowedPrintConsole);
        Store.put(context, KEY_COLLECT_CRASH, enableCollectCrash);
        Store.put(context, KEY_ROOT_PATH, rootPath);
    }

    @Override
    public int hashCode() {
        int result = 0;
        result = result + (enableCollectCrash ? 1 : 0);
        result = result + (allowedWriteFile ? 2 : 0);
        result = result + (allowedPrintConsole ? 4 : 0);
        result = result + (once ? 8 : 0);
        result = result + logLevel * 100;
        result = result + rootPath.hashCode() * 13;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LogConfiguration)) {
            return false;
        }
        LogConfiguration other = (LogConfiguration) obj;
        if (other.allowedPrintConsole == allowedPrintConsole &&
                other.allowedWriteFile == allowedWriteFile &&
                other.enableCollectCrash == enableCollectCrash &&
                other.once == once &&
                other.logLevel == logLevel &&
                other.rootPath.equals(rootPath)) {
            return true;
        }
        return false;
    }

    @Override
    protected Object clone() {
        Cloneable o = null;
        try {
            o = (Cloneable) super.clone();
        } catch (CloneNotSupportedException e) {
            e.printStackTrace();
        }

        return o;
    }

    public static class Builder {
        private static final int DEFAULT_LOG_LEVEL = LogLevel.ALL.level;
        private int logLevel;
        private String rootPath;
        private boolean enableCollectCrash;
        private boolean allowedWriteFile;
        private boolean allowedPrintConsole;
        private boolean once = false;
        private Context context;

        public Builder(Context context) {
            this.context = context;

            logLevel = Store.getInt(context, KEY_LOG_LEVEL, DEFAULT_LOG_LEVEL);
            allowedWriteFile = Store.getBoolean(context, KEY_WRITE_LOG_TO_FILE, false);
            allowedPrintConsole = Store.getBoolean(context, KEY_PRINT_CONSOLE, true);
            enableCollectCrash = Store.getBoolean(context, KEY_COLLECT_CRASH, true);
            rootPath = Store.getString(context, KEY_ROOT_PATH, context.getExternalFilesDir(ROOT_PATH_NAME).getAbsolutePath());
        }

        public Builder logLevel(int logLevel) {
            this.logLevel = logLevel;
            return this;
        }

        public Builder rootPath(String rootPath) {
            this.rootPath = rootPath;
            return this;
        }

        public Builder enableCollectCrash(boolean enableCollectCrash) {
            this.enableCollectCrash = enableCollectCrash;
            return this;
        }

        public Builder allowedWriteFile(boolean allowedWriteFile) {
            this.allowedWriteFile = allowedWriteFile;
            return this;
        }

        public Builder allowedPrintConsole(boolean allowedPrintConsole) {
            this.allowedPrintConsole = allowedPrintConsole;
            return this;
        }

        public Builder once(boolean once) {
            this.once = once;
            return this;
        }

        public LogConfiguration build() {
            return new LogConfiguration(this);
        }

    }
}
