package com.xdja.log;

import android.content.Context;
import android.util.Log;

import java.io.File;


/**
 * 日志打印模块的入口类
 * Created by hyh on 2021/1/14
 */
public class XdjaLog {

    static LogConfiguration sLogConfiguration;

    static Logger logger;

    static XdjaLogcat xdjaLogCat;

    static XdjaCrashLog xdjaCrashLog;

    static boolean initFlag = false;

    /**
     * 使用默认配置初始化
     * <p>
     * {@link #init(LogConfiguration)} 和本方法只生效一次
     *
     * @param context
     */
    public static void init(Context context) {
        if (context == null) {
            throw new NullPointerException("content is empty");
        }
        init(new LogConfiguration.Builder(context).build());
    }

    /**
     * 使用LogConfiguration初始化
     * <p>
     * {@link #init(Context)} 和本方法只生效一次
     *
     * @param configuration
     */
    public static void init(LogConfiguration configuration) {
        if (initFlag) {
            return;
        }
        initFlag = true;
        if (configuration == null) {
            throw new NullPointerException("LogConfiguration is empty");
        }
        sLogConfiguration = configuration;
        xdjaCrashLog = new XdjaCrashLog(sLogConfiguration);
        logger = new Logger(sLogConfiguration);
        xdjaLogCat = new XdjaLogcat(sLogConfiguration);
        updateLogConfiguration(configuration);
    }

    /**
     * 获取当前策略
     *
     * @return 如果未初始化，则返回为null
     */
    public static LogConfiguration getCurLogConfiguration() {
        if (!initFlag) {
            return null;
        }
        return (LogConfiguration) sLogConfiguration.clone();
    }

    /**
     * 更新当前策略。
     * 当{@link LogConfiguration#once}为true时，会存储{@link LogConfiguration}中的参数。
     * 注：调用此接口之前，需要先调用{@link XdjaLog#init}
     *
     * @param configuration
     * @return
     */
    public static boolean updateLogConfiguration(LogConfiguration configuration) {
        if (!initFlag) {
            return false;
        }
        if (configuration == null) {
            throw new NullPointerException("LogConfiguration is empty");
        }
        if (!configuration.equals(sLogConfiguration)) {
            sLogConfiguration = configuration;
            if (!sLogConfiguration.once) {
                sLogConfiguration.store();
            }
            logger.updateConfigLog(configuration);
            xdjaCrashLog.updateConfigLog(configuration);
            xdjaLogCat.updateConfigLog(configuration);
        }
        return true;
    }

    /**
     * 获取写日志的路径。
     * 也可以使用{@link #getCurLogConfiguration()}，得到{@link LogConfiguration#rootPath}
     *
     * @return 如果未初始化，则返回为空
     */
    public static String getPathForWriteLog() {
        if (!initFlag) {
            return null;
        }
        return sLogConfiguration.rootPath;
    }

    /**
     * 开启logcat日志输出。
     * 注：此接口为一次性接口，当进程重启后，不会自动进行。
     */
    public static void startLogCatOutput() {
        if (!initFlag) {
            return;
        }
        xdjaLogCat.startLogCatOutput();
    }

    /**
     * 关闭logcat日志输出
     */
    public static void stopLogCatOutput() {
        if (!initFlag) {
            return;
        }
        xdjaLogCat.stopLogCatOutput();
    }

    /**
     * 压缩所有日志。<p>
     * 注意：<p>
     * 1.需要先调用{@link XdjaLog#init(Context)} 或 {@link XdjaLog#init(LogConfiguration)}。否则会抛出异常。<p>
     * 2.使用完返回的File之后，要尽量删除，否则会导致占用的存储空间越来越大。<p>
     *
     * @return 压缩后的文件。如果日志目录下没有文件，或者异常，会返回null
     */
    public static File zipAllLog() {
        return XdjaLogZip.zipAllLog();
    }

    /**
     * v级别 日志输出
     *
     * @param tag
     * @param message
     */
    public static void v(String tag, String message) {
        if (!isInit()) {
            Log.v(tag, message);
            return;
        }
        logger.v(tag, message);
    }

    /**
     * d级别 日志输出
     *
     * @param tag
     * @param message
     */
    public static void d(String tag, String message) {
        if (!isInit()) {
            Log.d(tag, message);
            return;
        }
        logger.d(tag, message);
    }

    /**
     * i级别 日志输出
     *
     * @param tag
     * @param message
     */
    public static void i(String tag, String message) {
        if (!isInit()) {
            Log.i(tag, message);
            return;
        }
        logger.i(tag, message);
    }

    /**
     * w级别 日志输出
     *
     * @param tag
     * @param message
     */
    public static void w(String tag, String message) {
        if (!isInit()) {
            Log.w(tag, message);
            return;
        }
        logger.w(tag, message);
    }

    /**
     * e级别 日志输出
     *
     * @param tag
     * @param message
     */
    public static void e(String tag, String message) {
        if (!isInit()) {
            Log.e(tag, message);
            return;
        }
        logger.e(tag, message);
    }

    private static boolean isInit() {
        return initFlag;
    }

}
