package com.xdja.log;

import android.content.Context;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 压缩所有日志文件
 * <p>
 * 内部根据{@link LogConfiguration#rootPath}，获此目录下所有的文件夹，并进行压缩，
 * 压缩完成后，会删除所有压缩的日志，
 * 除了当前正在被打开的除外；
 *
 * @author zhangxiaolong@xdja.com <br/>
 * @date 2021/1/15 <br/>
 */
class XdjaLogZip {

    /**
     * 压缩所有日志。
     * 注意：需要先调用{@link XdjaLog#init(Context)} 或 {@link XdjaLog#init(LogConfiguration)}。
     * 否则会抛出异常。
     *
     * @return 压缩后的文件。如果日志目录下没有文件，或者异常，会返回null
     */
    public static File zipAllLog() {
        LogConfiguration curLogConfiguration = XdjaLog.getCurLogConfiguration();
        if (curLogConfiguration == null) {
            throw new RuntimeException("XdjaLog is not init()!");
        }
        String rootPath = curLogConfiguration.rootPath;
        String time = getTimeStr("yyyy-MM-dd hh:mm:ss");
        File zipFile = new File(rootPath, time + ".zip");
        ZipOutputStream zos = null;
        try {
            zos = new ZipOutputStream(new FileOutputStream(zipFile));
            ArrayList<File> compressFileList = new ArrayList<>();
            File[] files = new File(rootPath).listFiles();
            if (files == null) {
                return null;
            }
            for (File sourceFile : files) {
                if (sourceFile.isFile()) {
                    continue;
                }
                compress(sourceFile, zos, sourceFile.getName(), compressFileList);
            }
            deleteCompressFile(compressFileList);
        } catch (Exception e) {
            throw new RuntimeException("zip error from ZipUtils", e);
        } finally {
            if (zos != null) {
                try {
                    zos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return zipFile;
    }

    /**
     * 递归压缩方法
     *
     * @param sourceFile 源文件
     * @param zos        zip输出流
     * @param name       压缩后的名称
     * @throws Exception
     */

    private static void compress(File sourceFile,
                                 ZipOutputStream zos,
                                 String name,
                                 List<File> compressFileList) throws Exception {

        byte[] buf = new byte[4 * 1024];
        if (sourceFile.isFile()) {
            // 向zip输出流中添加一个zip实体，构造器中name为zip实体的文件的名字
            zos.putNextEntry(new ZipEntry(name));
            // copy文件到zip输出流中
            int len;
            FileInputStream in = new FileInputStream(sourceFile);
            while ((len = in.read(buf)) != -1) {
                zos.write(buf, 0, len);
            }
            // Complete the entry
            zos.closeEntry();
            in.close();
            compressFileList.add(sourceFile);
        } else {
            File[] listFiles = sourceFile.listFiles();
            if (listFiles == null || listFiles.length == 0) {
                // 需要保留原来的文件结构时,需要对空文件夹进行处理
                // 空文件夹的处理
                zos.putNextEntry(new ZipEntry(name + "/"));
                // 没有文件，不需要文件的copy
                zos.closeEntry();
            } else {
                for (File file : listFiles) {
                    // 判断是否需要保留原来的文件结构
                    // 注意：file.getName()前面需要带上父文件夹的名字加一斜杠,
                    // 不然最后压缩包中就不能保留原来的文件结构,即：所有文件都跑到压缩包根目录下了
                    compress(file, zos, name + "/" + file.getName(), compressFileList);
                }
            }
        }
    }

    /**
     * 删除压缩的文件
     * <p>
     * 除了当前正在被打开的除外；
     *
     * @param list
     */
    private static void deleteCompressFile(List<File> list) {
        for (File file : list) {
            File renameFile = new File(file.getAbsolutePath() + "_logtmp");
            if (file.renameTo(renameFile)) {
                renameFile.delete();
            }
        }
    }

    /**
     * 根据dateFormat返回日期字符串
     *
     * @param dateFormat
     * @return
     */
    private static String getTimeStr(String dateFormat) {
        SimpleDateFormat sdf = new SimpleDateFormat(dateFormat, Locale.CHINA);
        sdf.setTimeZone(TimeZone.getDefault());
        String time = sdf.format(new Date(System.currentTimeMillis()));
        return time;
    }
}
