package com.xdja.log.logcat;

import com.xdja.log.LogLevel;
import com.xdja.log.internal.DefaultsFactory;
import com.xdja.log.printer.file.FilePrinter;
import com.xdja.log.printer.file.backup.BackupStrategy;
import com.xdja.log.printer.file.clean.CleanStrategy;
import com.xdja.log.printer.file.naming.FileNameGenerator;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

/**
 * @author hyh
 * data on 2021/1/19
 */
public class LogcatFileOutput {

    private String folderPath;

    private FileNameGenerator fileNameGenerator;

    BackupStrategy backupStrategy;

    CleanStrategy cleanStrategy;

    public LogcatFileOutput(Builder builder) {
        folderPath = builder.folderPath;
        fileNameGenerator = builder.fileNameGenerator;
        backupStrategy = builder.backupStrategy;
        cleanStrategy = builder.cleanStrategy;
        checkLogFolder();
    }

    /**
     * Make sure the folder of log file exists.
     */
    private void checkLogFolder() {
        File folder = new File(folderPath);
        if (!folder.exists()) {
            folder.mkdirs();
        }
    }

    public void logcatFileOperation(ILogcatEvent iLogcatEvent) {
        cleanLogFilesIfNecessary();
        File targetFile = getCurrentLogcatPath();
        if (backupStrategy.shouldBackup(targetFile)) {
            iLogcatEvent.close();
            File backupFile = new File(folderPath, backupName());
            if (backupFile.exists()) {
                backupFile.delete();
            }
            targetFile.renameTo(backupFile);
            iLogcatEvent.open();
        }
    }

    SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm", Locale.CHINA);

    private String backupName() {
        sdf.setTimeZone(TimeZone.getDefault());
        return sdf.format(new Date(System.currentTimeMillis())) + ".bak";
    }

    /**
     * Clean log files if should clean follow strategy
     */
    private void cleanLogFilesIfNecessary() {
        File logDir = new File(folderPath);
        File[] files = logDir.listFiles();
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (cleanStrategy.shouldClean(file)) {
                file.delete();
            }
        }
    }

    public File getCurrentLogcatPath() {
        return new File(folderPath, fileNameGenerator.generateFileName(LogLevel.ERROR.getLevel(), System.currentTimeMillis()));
    }


    /**
     * Builder for {@link FilePrinter}.
     */
    public static class Builder {
        /**
         * The folder path of log file.
         */
        String folderPath;

        /**
         * The file name generator for log file.
         */
        FileNameGenerator fileNameGenerator;

        /**
         * The backup strategy for log file.
         */
        BackupStrategy backupStrategy;

        /**
         * The clean strategy for log file.
         */
        CleanStrategy cleanStrategy;


        /**
         * Construct a builder.
         *
         * @param folderPath the folder path of log file
         */
        public Builder(String folderPath) {
            this.folderPath = folderPath;
        }

        /**
         * Set the file name generator for log file.
         *
         * @param fileNameGenerator the file name generator for log file
         * @return the builder
         */
        public Builder fileNameGenerator(FileNameGenerator fileNameGenerator) {
            this.fileNameGenerator = fileNameGenerator;
            return this;
        }

        /**
         * Set the backup strategy for log file.
         *
         * @param backupStrategy the backup strategy for log file
         * @return the builder
         */
        public Builder backupStrategy(BackupStrategy backupStrategy) {
            this.backupStrategy = backupStrategy;
            return this;
        }

        /**
         * Set the clean strategy for log file.
         *
         * @param cleanStrategy the clean strategy for log file
         * @return the builder
         * @since 1.5.0
         */
        public Builder cleanStrategy(CleanStrategy cleanStrategy) {
            this.cleanStrategy = cleanStrategy;
            return this;
        }


        /**
         * Build configured {@link FilePrinter} object.
         *
         * @return the built configured {@link FilePrinter} object
         */
        public LogcatFileOutput build() {
            fillEmptyFields();
            return new LogcatFileOutput(this);
        }

        private void fillEmptyFields() {
            if (fileNameGenerator == null) {
                fileNameGenerator = DefaultsFactory.createFileNameGenerator();
            }
            if (backupStrategy == null) {
                backupStrategy = DefaultsFactory.createBackupStrategy();
            }
            if (cleanStrategy == null) {
                cleanStrategy = DefaultsFactory.createCleanStrategy();
            }
        }
    }

    public interface ILogcatEvent {
        void open();

        void close();
    }

}
