package com.xdja.cssp.as.auth;

import java.io.InputStream;
import java.security.PublicKey;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.UUID;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.message.BasicHeader;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import com.xdja.cssp.as.auth.cache.SignatureNonce;
import com.xdja.cssp.as.auth.exception.DuplicateRequestException;
import com.xdja.cssp.as.auth.exception.InvalidApiVersionException;
import com.xdja.cssp.as.auth.exception.InvalidDateException;
import com.xdja.cssp.as.auth.exception.InvalidSnException;
import com.xdja.cssp.as.auth.exception.NotMatchSignatureException;
import com.xdja.cssp.as.auth.exception.NotSupportSignAlgoException;
import com.xdja.cssp.as.auth.exception.RequestTimeoutException;
import com.xdja.cssp.as.auth.exception.VerifySignatureException;
import com.xdja.cssp.as.auth.model.Request;
import com.xdja.cssp.as.auth.sort.ComparatorHeader;
import com.xdja.cssp.as.auth.util.ByteUtils;
import com.xdja.cssp.as.auth.util.CertUtil;
import com.xdja.cssp.as.auth.util.SignUtils;
import com.xdja.cssp.as.service.ILoginService;
import com.xdja.cssp.as.service.model.Cert;
import com.xdja.cssp.restful.auth.exception.AuthException;
import com.xdja.cssp.restful.exception.ApiException;
import com.xdja.cssp.restful.exception.BadRequestException;
import com.xdja.platform.common.lite.kit.prop.PropKit;
import com.xdja.platform.rpc.consumer.refer.DefaultServiceRefer;

/**
 * 客户端身份认证拦截器
 * @author wyf
 *
 */
@Aspect
@Component
public class AuthClientAspect {
	
	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
	private ILoginService service = DefaultServiceRefer.getServiceRefer(ILoginService.class);
	
	/**
	 * 服务标识
	 */
	private String hostId;
	/**
	 * API版本号
	 */
	private String apiVersion; 
	/**
	 * 客户端请求时间距服务器时间最大间隔时间
	 */
	private long timeout;
	
	@Around("@annotation(com.xdja.cssp.as.auth.annotation.AuthClient)")
	public Object authClient(ProceedingJoinPoint jp) throws Throwable {
		logger.debug("开始验证客户端身份");
		
		//获取请求信息
		HttpServletRequest httpRequest = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
		
		//生成请求对象供控制层调用
		Request request = new Request();
		//生成请求id
		String requestId = genRequestId();
		request.setId(requestId);
		
		hostId = getHostId();
		apiVersion = getApiVersion();
		timeout = getTimeout();
//		this.getTimeout(jp);
		
		//验证API版本
		String version = httpRequest.getHeader(Constants.VERSION_HEADER_NAME);
		
		this.commonCheckParams(version, requestId, "API版本号为空");
		
		if (apiVersion != null && !version.equals(apiVersion)) {
			throw new InvalidApiVersionException(requestId, requestId,
					AuthException.INVALID_API_VERSION, "API版本号与服务要求不符");
		}

		//验证时间戳过期
		String timestamp = httpRequest.getHeader(Constants.TIMESTAMP_HEADER_NAME);
		
		this.commonCheckParams(timestamp, requestId, "请求时间戳为空");
		
		if(timeout != 0){
			try {
				Date date = Constants.parseTimestamp(timestamp);
				long minusTime = System.currentTimeMillis() - date.getTime();
				if (minusTime > timeout) {
					throw new RequestTimeoutException(hostId, requestId, AuthException.REQUEST_TIMEOUT ,"请求时间戳超时(可能由于客户端时间不正确导致，请先校准客户端时间)");
				}
			} catch (ParseException e) {
				logger.error("请求时间戳格式转换异常", e);
				throw new InvalidDateException(hostId, requestId, AuthException.INVALID_DATE ,"请求时间戳格式非法", e);
			}
		}
	
		
		//校验挑战值，防止重放攻击
		String signatureNonce = httpRequest.getHeader(Constants.SIGNATURE_NONCE_HEADER_NAME);
		
		this.commonCheckParams(signatureNonce, requestId, "随机数为空");
		
		if (SignatureNonce.signatureNonce.equals(signatureNonce)) { 
			throw new DuplicateRequestException(hostId, requestId, AuthException.DUPLICATE_REQUEST ,"重复的请求");
		} else {
			SignatureNonce.signatureNonce = signatureNonce;
		}
		
		//验证签名算法
		String signatureMethod = httpRequest.getHeader(Constants.SIGNATURE_METHOD_HEADER_NAME);
		
		this.commonCheckParams(signatureMethod, requestId, "签名方式为空");
		
		if (!signatureMethod.equals(Constants.HTTP_HEADER_SIGN_METHOD_RSA)
				&& !signatureMethod.equals(Constants.HTTP_HEADER_SIGN_METHOD_SM2)) {
			throw new NotSupportSignAlgoException(hostId, requestId, AuthException.NOT_SUPPORT_SIGN_ALGO, "不支持的签名算法");
		}
		
		request.setSignatureAlgo(signatureMethod);
		
		String authorization = httpRequest.getHeader(Constants.AUTHORIZATION_HEADER_NAME);
		
		this.commonCheckParams(authorization, requestId, "签名信息为空");
		
		String signatureSn = httpRequest.getHeader(Constants.SIGNATURE_SN_HEADER_NAME);
		
		this.commonCheckParams(signatureSn, requestId, "终端签名证书SN为空");
		
		Cert cert;
		int caAlg;
		if (signatureMethod.equals(Constants.HTTP_HEADER_SIGN_METHOD_SM2)) {
			caAlg = 2;
		} else {
			caAlg = 1;
		}
		
		cert = service.queryCert(signatureSn, caAlg);
		if (null == cert) {
			throw new InvalidSnException(hostId, requestId, AuthException.INVALID_SN, "证书不存在");
		}
		request.setCardNo(cert.getCardNo());
		request.setSignSn(cert.getSn());
		
		String queryString = httpRequest.getQueryString();
		String method = httpRequest.getMethod();
		String uri = httpRequest.getServletPath();
		request.setMethod(method);
		request.setUri(uri);
		
		List<Header> list = new ArrayList<Header>();
		@SuppressWarnings("unchecked")
		Enumeration<String> headers = httpRequest.getHeaderNames();
		String headerName;
		while (headers.hasMoreElements()) {
			headerName = headers.nextElement();
			if (headerName.startsWith(Constants.HEADER_NAME_START)) {
				list.add(new BasicHeader(headerName, httpRequest.getHeader(headerName)));
			}
		}
		ComparatorHeader comparatorHeader = new ComparatorHeader();
		Collections.sort(list, comparatorHeader);
		
		String requestBody;
		boolean flag = false;
		try {
			InputStream in = httpRequest.getInputStream();
			byte[] contentIn = ByteUtils.inputStreamToBytes(in);
			requestBody = new String(contentIn, "UTF-8");
			request.setBody(contentIn);

			String canonicalizeRequest = Constants.generateCanonicalizeRequest(method, uri, queryString, list, requestBody);

			String certBase64 = cert.getCert();
			PublicKey publicKey = CertUtil.getCertFromStr(certBase64).getPublicKey();
			
			flag = SignUtils.verifySignature(signatureMethod, publicKey, canonicalizeRequest.getBytes("UTF-8"), Base64.decodeBase64(authorization));
		} catch (Exception e) {
			logger.error("验证请求签名异常", e);
			throw new VerifySignatureException(hostId, requestId, AuthException.VERIFY_SIGNATURE_ERROR ,"验证请求签名异常", e);
		}
		if (!flag) {
			throw new NotMatchSignatureException(hostId, requestId, AuthException.NOT_MATCH_SIGNATURE ,"验证请求签名不匹配");
		}
		
		Object[] args = jp.getArgs();
		Object item = null;
		for (int i = 0; i < args.length; i++) {
			item = args[i];
			if (null != item && item.getClass() == Request.class) {
				args[i] = request;
				break;
			}
		}
		
		logger.debug("验证客户端身份通过");
		return jp.proceed(args);
	}

	/**
	 * 获取服务标识
	 * @param jp
	 */
	/*private void getHostId(ProceedingJoinPoint jp) {
		//服务标识
		hostId = "未设置服务标识";
		Method method = ClassUtils.getMethodIfAvailable(jp.getTarget().getClass(), "getHostId");
		if (null != method) {
			try {
				Object result = method.invoke(jp.getTarget());
				if (null != result) {
					hostId = result.toString();
					logger.debug("服务器标识：{}", hostId);
				} else {
					logger.warn("{}中的方法：public String getHostId(){}未返回服务标识", jp.getTarget().getClass());
				}
			} catch (IllegalAccessException | IllegalArgumentException
					| InvocationTargetException e) {
				logger.error("从{}中获取服务标识失败", jp.getTarget().getClass(), e);
			}
		} else {
			logger.warn("请在{}中添加方法：public String getHostId(){}返回服务标识", jp.getTarget().getClass());
		}
	}*/
	
	/**
	 * 获取请求最大间隔时间
	 * @param jp
	 */
	/*private void getTimeout(ProceedingJoinPoint jp) {
		Method method = ClassUtils.getMethodIfAvailable(jp.getTarget().getClass(), "getTimeout");
		if (null != method) {
			try {
				Object result = method.invoke(jp.getTarget());
				if (null != result) {
					timeout = Long.parseLong(result.toString());
					logger.debug("请求最大间隔时间：{}", timeout);
				} else {
					logger.warn("{}中的方法：public long getTimeout(){}未返回请求最大间隔时间", jp.getTarget().getClass());
				}
			} catch (IllegalAccessException | IllegalArgumentException
					| InvocationTargetException e) {
				logger.error("从{}中获取请求最大间隔时间失败", jp.getTarget().getClass(), e);
			}
		} else {
			logger.warn("请在{}中添加方法：public long getTimeout(){}返回请求最大间隔时间", jp.getTarget().getClass());
		}
	}*/

	/**
	 * 获取API版本号
	 * @param jp
	 */
	/*private void getApiVersion(ProceedingJoinPoint jp) {
		Method method = ClassUtils.getMethodIfAvailable(jp.getTarget().getClass(), "getApiVersion");
		if (null != method) {
			try {
				Object result = method.invoke(jp.getTarget());
				if (null != result) {
					apiVersion = result.toString();
					logger.debug("API版本号：{}", apiVersion);
				} else {
					logger.warn("{}中的方法：public String getApiVersion(){}未返回API版本号", jp.getTarget().getClass());
				}
			} catch (IllegalAccessException | IllegalArgumentException
					| InvocationTargetException e) {
				logger.error("从{}中获取API版本号失败", jp.getTarget().getClass(), e);
			}
		} else {
			logger.warn("请在{}中添加方法：public String getApiVersion(){}返回API版本号", jp.getTarget().getClass());
		}
	}*/

	/**
	 * 通用验证参数是否为空
	 * @param value 待验证值
	 * @param requestId 请求ID
	 * @param errMsg 错误消息描述
	 */
	private void commonCheckParams(String value, String requestId, String errMsg) {
		if (StringUtils.isBlank(value)) {
			throw new BadRequestException(hostId, requestId,
					ApiException.REQUEST_PARAMS_ERROR, errMsg);
		}
	}
	
	/**
	 * 生成请求id
	 * @return
	 */
	private String genRequestId() {
		return UUID.randomUUID().toString();
	}
	
	public String getHostId() {
		return PropKit.getProp("system.properties").get("host.id");
	}
	
	private String getApiVersion(){
		return PropKit.getProp("system.properties").get("api.version");
	}
	
	private Long getTimeout() {
		return PropKit.getProp("system.properties").getLong("request.timeout", 0l);
		
	}

}
