package com.xdja.cssp.as.auth.util;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.SecureRandom;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECGenParameterSpec;

import org.bouncycastle.asn1.sm2.SM2ObjectIdentifiers;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import com.xdja.cssp.sm2cipher.sm2.ISM2Cipher;
import com.xdja.cssp.sm2cipher.sm2.cipher.Base64Util;
import com.xdja.cssp.sm2cipher.sm2.cipher.SM2CipherImpl;

/**
 * 
 * 
 * Project Name：atfs-auth
 * ClassName：SM2Utils
 * Description：提供sm2加密、解密、数字签名和签名验证接口
 * @author: 谢文超
 * @date: 2015-5-5 上午10:00:02
 * note:
 *
 */
public class SM2Utils {
	
	static {
		if (Security.getProvider("BC") == null) {
			Security.addProvider(new BouncyCastleProvider());
		}
	}
	
	/**
	 * 
	 * 方法描述：使用sm2算法公钥加密数据
	 * @author: 谢文超
	 * @date: 2015-5-5 上午10:42:35
	 * @param publicKey  公钥  不允许为空
	 * @param plainText  明文 不允许为空
	 * @return byte[] 密文
	 */
	public static byte[] encrypt(ECPublicKey publicKey, byte[] plainText) {
		if (publicKey == null || plainText == null ||
				plainText.length <= 0) {
			throw new IllegalArgumentException("param error: publicKey,plainText can not be null");
		}
		byte[] cipherText = null;
		byte[] pubkeydata = new byte[65];
		//assembel pubkeydata
		pubkeydata[0] = 0x04;
		byte[] xbyte = publicKey.getW().getAffineX().toByteArray();
		byte[] ybyte = publicKey.getW().getAffineY().toByteArray();
		System.arraycopy(xbyte, xbyte.length > 32  ? 1 : 0, pubkeydata, 1, 32);
		System.arraycopy(ybyte, ybyte.length > 32  ? 1 : 0, pubkeydata, 32 + 1 , 32);
		
		ISM2Cipher cipher = new SM2CipherImpl();
		String base64key = new String(Base64Util.encode(pubkeydata));
		cipherText = cipher.encrypt_gb(base64key, plainText);
		
		return cipherText;
	}
	
	/**
	 * 
	 * 方法描述：使用sm2加密证书加密数据
	 * @author: 谢文超
	 * @date: 2015-5-5 上午10:42:35
	 * @param certificate 加密证书  不允许为空
	 * @param plainText  明文 不允许为空
	 * @return byte[] 密文
	 */
	public static byte[] encrypt(X509Certificate certificate, byte[] plainText) {
		if (certificate == null || plainText == null ||
				plainText.length <= 0) {
			throw new IllegalArgumentException("param error: certificate,plainText can not be null");
		}
		byte[] cipherText = null;
		byte[] pubkeydata = new byte[65];
		//assembel pubkeydata
		ECPublicKey publicKey = (ECPublicKey) certificate.getPublicKey();
		pubkeydata[0] = 0x04;
		byte[] xbyte = publicKey.getW().getAffineX().toByteArray();
		byte[] ybyte = publicKey.getW().getAffineY().toByteArray();
		System.arraycopy(xbyte, xbyte.length > 32  ? 1 : 0, pubkeydata, 1, 32);
		System.arraycopy(ybyte, ybyte.length > 32  ? 1 : 0, pubkeydata, 32 + 1 , 32);
		
		ISM2Cipher cipher = new SM2CipherImpl();
		String base64key = new String(Base64Util.encode(pubkeydata));
		cipherText = cipher.encrypt_gb(base64key, plainText);
		
		return cipherText;
	}
	
	/**
	 * 
	 * 方法描述：使用sm2算法私钥解密数据
	 * @author: 谢文超
	 * @date: 2015-5-5 上午10:42:35
	 * @param publicKey  私钥  不允许为空
	 * @param plainText  密文 不允许为空
	 * @return byte[] 明文
	 */
	public static byte[] decrypt(ECPrivateKey privateKey, byte[] cipherText) {
		if (privateKey == null || cipherText == null ||
				cipherText.length <= 0) {
			throw new IllegalArgumentException("param error: privateKey,cipherText can not be null");
		}
		byte[] plainText = null;
		byte[] prikeydata = new byte[32];
		
		//assembel prikeydata
		byte[] temp = privateKey.getS().toByteArray();
		System.arraycopy(temp, temp.length > 32 ? 1 : 0, prikeydata, 0, 32);
		
		ISM2Cipher cipher = new SM2CipherImpl();
		String base64key = new String(Base64Util.encode(prikeydata));
		plainText = cipher.decrypt_gb(base64key, cipherText);
		
		return plainText;
	}
	
	
	/**
	 * 
	 * 方法描述：使用sm2算法私钥签名数据
	 * @author: 谢文超
	 * @date: 2015-5-5 上午11:24:11
	 * @param privateKey	私钥
	 * @param original		签名原文
	 * @return byte[]  签名数据
	 * @throws IllegalArgumentException
	 */
	public static byte[] sign(ECPrivateKey privateKey, byte[] original) 
			throws IllegalArgumentException {
		if (privateKey == null || original == null ||
				original.length <= 0) {
			throw new IllegalArgumentException("param error: privateKey,original can not be null");
		}
		ISM2Cipher cipher = new SM2CipherImpl();
		byte[] signData = cipher.sign(privateKey, original);;
		
		return signData;
	}
	
	/**
	 * 
	 * 方法描述：使用sm2算法公钥验证数字签名
	 * @author: 谢文超
	 * @date: 2015-5-5 上午11:25:41
	 * @param publicKey	公钥
	 * @param sign	签名数据
	 * @param original	签名原文
	 * @return  true-通过 false-未通过
	 * @throws IllegalArgumentException
	 */
	public static boolean verify(ECPublicKey publicKey, byte[] sign, byte[] original) 
			throws IllegalArgumentException {
		if (publicKey == null || original == null || sign == null ||
				original.length <= 0 || sign.length <= 0) {
			throw new IllegalArgumentException("param error: publicKey,sign,original can not be null");
		}
		
		boolean flag = false;
		ISM2Cipher cipher = new SM2CipherImpl();
		flag = cipher.verify(publicKey, sign, original);
		
		return flag;
	}
	
	/**
	 * 
	 * 方法描述：使用sm2算法签名证书验证数字签名
	 * @author: 谢文超
	 * @date: 2015-5-5 上午11:25:41
	 * @param certificate	签名证书
	 * @param sign	签名数据
	 * @param original	签名原文
	 * @return  true-通过 false-未通过
	 * @throws IllegalArgumentException
	 */
	public static boolean verify(X509Certificate certificate, byte[] sign, byte[] original) 
			throws IllegalArgumentException {
		if (certificate == null || original == null || sign == null ||
				original.length <= 0 || sign.length <= 0) {
			throw new IllegalArgumentException("param error: certificate,sign,original can not be null");
		}
		boolean flag = false;
		ISM2Cipher cipher = new SM2CipherImpl();
		flag = cipher.verify((ECPublicKey) certificate.getPublicKey(), sign, original);
		
		return flag;
	}
	
	public static void main(String[] args) {
		try {
			KeyPairGenerator pair = KeyPairGenerator.getInstance("ECSM2", "BC");
			pair.initialize(new ECGenParameterSpec(SM2ObjectIdentifiers.sm2256.getId()), new SecureRandom());
			KeyPair keyPair  = pair.generateKeyPair();
			ECPrivateKey privateKey = (ECPrivateKey) keyPair.getPrivate();
			System.out.println(privateKey);
			System.out.println(privateKey.getEncoded().length);
			ECPublicKey publicKey = (ECPublicKey) keyPair.getPublic();
			System.out.println(publicKey);
			System.out.println(publicKey.getEncoded().length);
			System.out.println(new String(Base64Util.encode(publicKey.getEncoded())).length());
			System.out.println(new String(Base64Util.encode(publicKey.getEncoded())));
			String plainText = "hello1ss11";
			byte[] cipherText = encrypt(publicKey, plainText.getBytes());
			System.out.println( "size: " + cipherText.length + "  " + new String(decrypt(privateKey, cipherText)));
			
			byte[] sign = sign(privateKey, plainText.getBytes());
			System.out.println("size: " + sign.length + " " + verify(publicKey, sign, plainText.getBytes()));
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
}
