package com.xdja.cssp.ras.business.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xdja.cssp.acs.IAssetService;
import com.xdja.cssp.acs.ICertService;
import com.xdja.cssp.acs.bean.asset.Asset;
import com.xdja.cssp.acs.bean.asset.Type;
import com.xdja.cssp.acs.bean.cert.Cert;
import com.xdja.cssp.acs.bean.cert.CertStatus;
import com.xdja.cssp.ras.business.vo.AccountAsset;
import com.xdja.cssp.ras.business.vo.AccountInfo;
import com.xdja.cssp.ras.cache.AccountCache;
import com.xdja.cssp.ras.service.Constant;
import com.xdja.cssp.ras.service.bean.auth.DeviceBind;
import com.xdja.cssp.ras.service.bean.auth.Regist;
import com.xdja.cssp.ras.service.bean.auth.RegistMobile;
import com.xdja.cssp.ras.service.bean.enums.DeviceBindStatus;
import com.xdja.cssp.ras.service.bean.enums.DeviceTestStatus;
import com.xdja.cssp.ras.service.bean.enums.RegistMobileStatus;
import com.xdja.cssp.ras.service.bean.enums.RegistStatus;
import com.xdja.cssp.ras.service.bean.result.DeviceBindResult;
import com.xdja.cssp.ras.service.bean.result.RegistMobileResult;
import com.xdja.cssp.ras.service.bean.result.RegistResult;
import com.xdja.platform.common.lite.kit.StrKit;
import com.xdja.platform.microservice.db.Dao;
import com.xdja.platform.microservice.db.tx.IAtom;
import com.xdja.platform.rpc.consumer.refer.DefaultServiceRefer;

/**
 * 公共业务信息实现类
 * @author liqi
 * @date 2015-2-4
 */
public final class CommonBusiness {
	
	
	//注册账号语句
	private static final String SQL_REG_ACCOUNT = "insert into t_account_info(c_account, c_password, c_register_time) values(?, ?, ?)";
	private static final String SQL_REG_ACCOUNT_EXTENTION = "insert into t_account_extention(c_account, c_mobile) values(?,?)";
	//注册资产语句
	private static final String SQL_REG_ASSERT = "insert into t_account_asset(c_account, c_asset_identify, c_device_name, n_asset_type, c_card_no, n_time) values(?, ?, ?, ?, ?, ?)";
	
	private static final Logger logger = LoggerFactory.getLogger(CommonBusiness.class);
	private static final Dao dao = Dao.use(Constant.DB_RAS);
	
	private static final int MESSAGE_STATUS_MODIFY = 2;
	private static final int MESSAGE_STATUS_ADD = 1;
	
	/**
	 * 账号变更通知2015-05-07
	 * @param accountList
	 * @param statusType
	 * @return 
	 */
	public static boolean isMessageSendSucess(List<String> accountList, int statusType) {
		try {
			SCNoticeUtil.sendUserModifyNotice(statusType, accountList);
		} catch (Exception e) {
			return false;
		}
		
		return true;
	}
	
	
	/**
	 * 注册设备检测
	 * @param device 设备注册信息
	 * @return 设备注册检测结果
	 * @throws Exception 异常信息
	 * @update 2015-04-12 13:29 
	 * @reason 一批手机终端没有对应的USBKey，因此，在注册时不再校验USBKEy，同时绑定时，有则绑定，无则不绑定
	 * @rd liqi 
	 * 
	 * 
	 */
	public static DeviceTestStatus registTestOnly(DeviceBind device) throws Exception {
		
		List<Asset> deviceList = getAssets(device.getDeviceId());
		
		if(deviceList == null || deviceList.size() <= 0 ) {
			
			logger.error("资产数据库中没有找到对应设备：" + device.getDeviceId());			
			return DeviceTestStatus.DEVICE_NOT_EXIST;
		}
		 
		List<Asset> cardList = getAssets(deviceList.get(0).getRelatedAsset());
		
		if(cardList == null || cardList.size() <= 0) {
			
			logger.error("资产数据库中没有找到对应设备的USBKey:" + device.getDeviceId());
			
			/**
			 *   @update 2015-04-12 13:29 
			 *   @reason 一批手机终端没有对应的USBKey，因此，在注册时不再校验USBKEy，同时绑定时，有则绑定，无则不绑定
			 *   @rd liqi   
			 */
			//return DeviceTestStatus.CARD_NOT_EXIST;
		}
		
		return DeviceTestStatus.SUCCESS;
	}
	
	
	
	/**
	 * 注册设备
	 * @param device 设备注册信息
	 * @return 设备注册结果
	 * @throws Exception 异常信息
	 * @update 2015-04-12 13:40 
	 * @reason 一批手机终端没有对应的USBKey，因此，在注册时不再校验USBKEy，同时绑定时，有则绑定，无则不绑定
	 * @rd liqi 
	 */
	public static DeviceBindResult registDeviceOnly(DeviceBind device) throws Exception {
		
		List<Asset> deviceList = getAssets(device.getDeviceId());
		
		if(deviceList == null || deviceList.size() <= 0 ) {
			
			logger.error("资产数据库中没有找到对应设备：" + device.getDeviceId());			
			return new DeviceBindResult(DeviceBindStatus.DEVICE_NOT_EXIST);
		}
		
		List<Asset> cardList = getAssets(deviceList.get(0).getRelatedAsset());

		if(cardList == null || cardList.size() <= 0) {
			
			logger.error("资产数据库中没有找到对应设备的USBKey:" + device.getDeviceId());
			
			/**
			 *   @update 2015-04-12 13:29 
			 *   @reason 一批手机终端没有对应的USBKey，因此，在注册时不再校验USBKEy，同时绑定时，有则绑定，无则不绑定
			 *   @rd liqi  
			 */
			//return new DeviceBindResult(DeviceBindStatus.CARD_NOT_EXIST);
		} 
		
		
		
		String deviceName = getDeviceName(device.getAccount(), deviceList.get(0).getType());
		long regTime = System.currentTimeMillis();
		final Object[][] param = new Object[1 + cardList.size() ][6];
		param[0][0] = device.getAccount();
		param[0][1] = deviceList.get(0).getIdentity();
		param[0][2] = deviceName;
		param[0][3] = deviceList.get(0).getType().value;
		param[0][4] = deviceList.get(0).getCardNo();
		param[0][5] = regTime;

		int i = 1;
		for(Asset as : cardList) {
			param[i][0] = device.getAccount();
			param[i][1] = as.getIdentity();
			param[i][2] = "";
			param[i][3] = as.getType().value;
			param[i][4] = as.getCardNo();
			param[i][5] = regTime;
		    i++;
		}
		
		final String deviceId = device.getDeviceId().trim();
		final String cardSn = device.getCardNo().trim() + "-" + device.getCertSn().trim();
		
		dao.tx(new IAtom(){
			
			@Override
			public boolean run() throws SQLException { 
				
				dao.batch(SQL_REG_ASSERT , param); 
				
				String oldRegist = RedisUtil.mdmHget(deviceId);
				if (StringUtils.isBlank(oldRegist)) {
					oldRegist = cardSn;
					
				} else {
					oldRegist += "," + cardSn;
				}
				//保证事务完整性
				RedisUtil.mdmHSet(deviceId, oldRegist);
				 
				return true;
			}  
		});
		
		List<String> accountList = new ArrayList<String>(1);
		accountList.add(device.getAccount());
		if(!isMessageSendSucess(accountList, MESSAGE_STATUS_MODIFY)) {
			 //TO-DO 
		}   
		
		return new DeviceBindResult(DeviceBindStatus.SUCCESS, device.getAccount(), deviceName);
	}
	
	/**
	 * 获取设备名称
	 * @param account 账号
	 * @param type 设备类型
	 * @return
	 * @throws Exception
	 */
	public static String getDeviceName(String account, Type type) throws Exception {
		long serial = dao.queryForLong("select count(0) from t_account_asset where c_account = ? and n_asset_type = ?", account, type.value).longValue();
 		return String.format("%s-%s", StrKit.firstCharToUpperCase(type.toString().toLowerCase()), serial < 9l ? ("0" + String.valueOf(serial + 1)) : String.valueOf(serial + 1) );
	}	
	
	/**
	 * 注册账号和设备信息
	 * @param regist
	 * @return
	 * @throws Exception
	 */
	public static RegistResult registAccountDevice(final Regist regist) throws Exception {
		//1  注册账号
		//2  获取设备信息
		//3  获取设备及证书信息
		
		List<Asset> deviceList = getAssets(regist.getDeviceId());
		
		if(deviceList == null || deviceList.size() <= 0 ) {
			
			logger.error("资产数据库中没有找到对应设备：" + regist.getDeviceId());			
			return new RegistResult(RegistStatus.DEVICE_NOT_EXIST);
		}
		
		Asset regAsset = deviceList.get(0);
		
		List<Asset> cardList = getAssets(regAsset.getRelatedAsset());
		
		if(cardList == null || cardList.size() <= 0) {
			
			logger.error("资产数据库中没有找到设备 {} 关联的设备", regist.getDeviceId());
			/**
			 *   @update 2015-04-12 14:00 
			 *   @reason 一批手机终端没有对应的USBKey，因此，在注册时不再校验USBKEy，同时绑定时，有则绑定，无则不绑定
			 *   @rd liqi  
			 *   start  1
			 *   
			 */
			//return new RegistResult(RegistStatus.CARD_NOT_EXIST);
			/**
			 *   @update 2015-04-12 14:00 
			 *   @reason 一批手机终端没有对应的USBKey，因此，在注册时不再校验USBKEy，同时绑定时，有则绑定，无则不绑定
			 *   @rd liqi  
			 *   end  1
			 *   
			 */
		}
		
		String deviceName = null;
		String relationDeviceName = null;
		if (regAsset.getType().value == Type.PHONE.value
				|| regAsset.getType().value == Type.PAD.value) {
			deviceName = getDeviceName(regist.getAccount(), regAsset.getType());
		}
		
		final long time = System.currentTimeMillis();
		
		final Object[][] param = new Object[1 + cardList.size()][6];
		param[0][0] = regist.getAccount();
		param[0][1] = regAsset.getIdentity();
		param[0][2] = deviceName;
		param[0][3] = regAsset.getType().value;
		param[0][4] = regAsset.getCardNo();
		param[0][5] = time;
		
		if (!cardList.isEmpty()) {
			if (cardList.get(0).getType().value == Type.PHONE.value
					|| cardList.get(0).getType().value == Type.PAD.value) {
				relationDeviceName = getDeviceName(regist.getAccount(), cardList.get(0).getType());
			}
			
			param[1][0] = regist.getAccount();
			param[1][1] = cardList.get(0).getIdentity();
			param[1][2] = relationDeviceName;
			param[1][3] = cardList.get(0).getType().value;
			param[1][4] = cardList.get(0).getCardNo();
			param[1][5] = time;
		}
		
		final String deviceId = regAsset.getType().value == Type.PHONE.value || regAsset.getType().value == Type.PAD.value
				? regist.getDeviceId().trim()
				: (!cardList.isEmpty() && (cardList.get(0).getType().value == Type.PHONE.value
					|| cardList.get(0).getType().value == Type.PAD.value) ? cardList.get(0).getIdentity() : "");
				
		final String cardSn = regAsset.getType().value == Type.PHONE.value || regAsset.getType().value == Type.PAD.value
				? regist.getCardNo().trim() + "-" + regist.getCertSn().trim()
				: (!cardList.isEmpty() && (cardList.get(0).getType().value == Type.PHONE.value
					|| cardList.get(0).getType().value == Type.PAD.value) 
					? cardList.get(0).getIdentity() + "-" + getCertSnByCardNo(cardList.get(0).getIdentity()) : "");
	
		
		dao.tx(new IAtom(){
			
			@Override
			public boolean run() throws SQLException {
				
				dao.insert(SQL_REG_ACCOUNT, regist.getAccount(), regist.getPwd(), time);
				dao.insert(SQL_REG_ACCOUNT_EXTENTION, regist.getAccount(), null);
				dao.batch(SQL_REG_ASSERT , param);
				
				//保证事务完整性
				if (StringUtils.isNotBlank(deviceId)) {
					RedisUtil.mdmHSet(deviceId, cardSn);
				}
				return true;
			}  
		});
		
		List<String> accountList = new ArrayList<String>(1);
		accountList.add(regist.getAccount());
		if(!isMessageSendSucess(accountList, MESSAGE_STATUS_ADD)) {
			 //TO-DO 
		}
		
		return new RegistResult(RegistStatus.SUCCESS, regist.getAccount(),
				StringUtils.isNotBlank(deviceName) ? deviceName : (StringUtils.isNotBlank(relationDeviceName) ? relationDeviceName : ""));
	}
	
	/**
	 * 根据证书认证返回结果拼装返回结果
	 * @param certState 证书认证结果
	 * @param regist 注册信息
	 * @return 注册结果
	 */
	public static RegistResult checkRegistCert(CertStatus certState, Regist regist) {
		
//		SUCCESS	1	检测成功
//		CARDNOTEXISTS	2	安全卡不存在
//		CERTNOTEXISTS	3	证书不存在
//		DISACCORD	4	卡号与SN不对应
//		CERTBLOCK	5	证书被冻结
//		CERTREVOKE	6	证书被吊销
//		   SUCCESS = new CertStatus("SUCCESS", 0, 1, "\u68C0\u6D4B\u6210\u529F");
//	        CARDNOTEXISTS = new CertStatus("CARDNOTEXISTS", 1, 2, "\u5B89\u5168\u5361\u4E0D\u5B58\u5728");
//	        CERTNOTEXISTS = new CertStatus("CERTNOTEXISTS", 2, 3, "\u8BC1\u4E66\u4E0D\u5B58\u5728");
//	        DISACCORD = new CertStatus("DISACCORD", 3, 4, "\u5361\u53F7\u4E0ESN\u4E0D\u5BF9\u5E94");
//	        CERTBLOCK = new CertStatus("CERTBLOCK", 4, 5, "\u8BC1\u4E66\u88AB\u51BB\u7ED3");
//	        CERTREVOKE = new CertStatus("CERTREVOKE", 5, 6, "\u8BC1\u4E66\u88AB\u540A\u9500");
		
		RegistResult result;
		switch(certState) {
			case CARD_NOT_EXISTS:
				logger.error("CertStatus.CARDNOTEXISTS:" + regist.getCardNo());
				result = new RegistResult(RegistStatus.CARD_NOT_EXIST);
				break;
			case CARD_SN_DISACCORD:
				logger.error("CertStatus.DISACCORD:" + regist.getCardNo() + " " +regist.getCertSn());
				result = new RegistResult(RegistStatus.CARD_SN_NOT_ACCORD);
				break;
			case CERT_FREEZE:
				logger.error("CertStatus.CERTBLOCK:" + regist.getCardNo() + " " +regist.getCertSn());
				result = new RegistResult(RegistStatus.CERT_FREEZE);
				break;
			case CERT_REVOKED: 
				logger.error("CertStatus.CERTREVOKE:" + regist.getCardNo() + " " +regist.getCertSn());
				result = new RegistResult(RegistStatus.CERT_REVOKED);
				break;
			default: 
				logger.error("CertStatus.SUCCESS:" + regist.getCardNo() + " " +regist.getCertSn());
				result = new RegistResult();
				break;
				
		}
		return result;
	}
	
	/**
	 * 组装证书认证结果
	 * @param certState 认证结果
	 * @param regist 注册信息
	 * @return
	 */
	public static RegistMobileResult bulidCheckCertResult(CertStatus certState, RegistMobile regist) {
		
	//		SUCCESS	1	检测成功
	//		CARDNOTEXISTS	2	安全卡不存在
	//		CERTNOTEXISTS	3	证书不存在
	//		DISACCORD	4	卡号与SN不对应
	//		CERTBLOCK	5	证书被冻结
	//		CERTREVOKE	6	证书被吊销
		
		RegistMobileResult result;
		switch(certState) {
			case CARD_NOT_EXISTS:
				logger.error("CertStatus.CARDNOTEXISTS:" + regist.getCardNo());
				result = new RegistMobileResult(RegistMobileStatus.CARD_NOT_EXIST);
				break;
			case CARD_SN_DISACCORD:
				logger.error("CertStatus.DISACCORD:" + regist.getCardNo() + " " +regist.getCertSn());
				result = new RegistMobileResult(RegistMobileStatus.CARD_SN_NOT_ACCORD);
				break;
			case CERT_FREEZE:
				logger.error("CertStatus.CERTBLOCK:" + regist.getCardNo() + " " +regist.getCertSn());
				result = new RegistMobileResult(RegistMobileStatus.CERT_FREEZE);
				break;
			case CERT_REVOKED: 
				logger.error("CertStatus.CERTREVOKE:" + regist.getCardNo() + " " +regist.getCertSn());
				result = new RegistMobileResult(RegistMobileStatus.CERT_REVOKED);
				break;
			default: 
				logger.error("CertStatus.SUCCESS:" + regist.getCardNo() + " " +regist.getCertSn());
				result = new RegistMobileResult();
				break;
				
		}
		
		return result;
	}
	
	/**
	 * 根据证书认证返回结果拼装返回结果
	 * @param certState 证书认证结果
	 * @param regist 注册信息
	 * @return 注册结果
	 */
	public static DeviceBindResult checkDeviceCert(CertStatus certState, DeviceBind deviceBind) {
		
		DeviceBindResult result;
		switch(certState) {
			case CARD_NOT_EXISTS:
				logger.error("CertStatus.CARDNOTEXISTS:" + deviceBind.getCardNo());
				result = new DeviceBindResult(DeviceBindStatus.CARD_NOT_EXIST);
				break;
			case CARD_SN_DISACCORD:
				logger.error("CertStatus.DISACCORD:" + deviceBind.getCardNo() + " " + deviceBind.getCertSn());
				result = new DeviceBindResult(DeviceBindStatus.CARD_SN_NOT_ACCORD);
				break;
			case CERT_FREEZE:
				logger.error("CertStatus.CERTBLOCK:" + deviceBind.getCardNo() + " " + deviceBind.getCertSn());
				result = new DeviceBindResult(DeviceBindStatus.CERT_FREEZE);
				break;
			case CERT_REVOKED: 
				logger.error("CertStatus.CERTREVOKE:" + deviceBind.getCardNo() + " " + deviceBind.getCertSn());
				result = new DeviceBindResult(DeviceBindStatus.CERT_REVOKED);
				break;
			default: 
				logger.error("CertStatus.SUCCESS:" + deviceBind.getCardNo() + " " + deviceBind.getCertSn());
				result = new DeviceBindResult();
				break;
				
		}
		return result;
	}
	
	/**
	 * 根据账号返回密码
	 * @param account
	 * @return
	 * @throws SQLException
	 */
	public static AccountInfo getAccount(String account) throws SQLException {
		return dao.queryForObject(AccountInfo.class, "select n_id as id, c_account as account, c_password as pwd from t_account_info where c_account = ?", account);
	}
	
	/**
	 * 根据注册信息或者用户资产信息
	 * @param regist
	 * @return
	 * @throws SQLException
	 */
    public static List<AccountAsset> getAccountAsset(String deviceId, String cardNo) throws SQLException {
		return dao.queryForList(AccountAsset.class, "select n_id as id, c_account as account, c_asset_identify as assetIdentify, c_device_name as deviceName, n_asset_type as assetType, c_card_no as cardNo  from t_account_asset where c_asset_identify = ? or c_card_no = ?", deviceId, cardNo);
	}
	
    /**
     * 根据账号获取证书卡号
     * @param accounts 账号列表
     * @return 
     * @throws SQLException
     */
    public static List<AccountAsset> getAccountsCardNo(List<String> accounts) throws SQLException {
		StringBuffer sqlBuffer = new StringBuffer(1024); 
		
		for(String s : accounts) {
			sqlBuffer.append(",'").append(s.trim()).append("'");
		} 
		sqlBuffer.substring(1);
		
    	return dao.queryForList(AccountAsset.class, "select c_account as account, c_card_no as cardNo  from t_account_asset where c_account in ( " + sqlBuffer.toString() + ")");
    }
    
     
    
    /**
     * 根据注册信息或者用户资产信息
     * @param account 账号
     * @param cardNo 卡号
     * @return 账号资产
     * @throws SQLException
     */
    public static AccountAsset getAssetByAccountCart(String account, String cardNo) throws SQLException {
		return dao.queryForObject(AccountAsset.class, "select n_id as id, c_account as account, c_asset_identify as assetIdentify, c_device_name as deviceName, n_asset_type as assetType, c_card_no as cardNo  from t_account_asset where c_account = ? and c_card_no = ?", account, cardNo);
	}
	
    /**
     * 根据证书获取账号
     * @param cardNo
     * @return
     * @throws SQLException
     */
    public static String getAccountByCart(String cardNo) throws SQLException {
		return dao.queryForStr("select c_account as account from t_account_asset where c_card_no = ?", cardNo);
	}
    
	/**
	 * 根据资产标识获取资产信息
	 * @param as 资产标识
	 * @return 资产列表
	 */
	public static List<Asset> getAssets(String... as) {
		IAssetService assetService = DefaultServiceRefer.getServiceRefer(IAssetService.class);
		return assetService.queryAssets(as);
	}
	
//	public static List<PublicKey> getCertsBySn(List<String> snLists) {
//		ICertService certService = DefaultServiceRefer.getServiceRefer(ICertService.class);
//		return null;
//	}
//	
//	public static List<Cert> getCertsByCard(List<Card> cardLists) {
//		ICertService certService = DefaultServiceRefer.getServiceRefer(ICertService.class);
//		return null;
//	}
	
	/**
	 * 验证设备是否存在
	 * @param regist
	 * @return
	 * @throws Exception
	 */
	public static boolean isDevExists(String deviceId) throws Exception { 
		IAssetService assetService = DefaultServiceRefer.getServiceRefer(IAssetService.class);
		return assetService.exists(deviceId); 
	}
	
	/**
	 * 获取证书状态
	 * @param regist 注册参数
	 * @return 
	 * 	SUCCESS	1	检测成功
	 *  CARDNOTEXISTS	2	安全卡不存在
	 *  CERTNOTEXISTS	3	证书不存在
	 *  DISACCORD	4	卡号与SN不对应
	 *  CERTBLOCK	5	证书被冻结
	 *  CERTREVOKE	6	证书被吊销
	 * @throws Exception
	 */
	public static CertStatus getCardStatus(String cardNo, String certSn, int caAlg) { 
		ICertService certService = DefaultServiceRefer.getServiceRefer(ICertService.class);
		return certService.checkCertStatus(cardNo, certSn, caAlg);
	}
	
	/**
	 * 根据卡号获取RSA签名证书SN
	 * @param cardNo
	 * @return
	 */
	public static String getCertSnByCardNo(String cardNo) {
		ICertService certService = DefaultServiceRefer.getServiceRefer(ICertService.class);
		
		List<String> cardNos = new ArrayList<>();
		cardNos.add(cardNo);
		
		Map<String, List<Cert>> result = certService.queryCertByCardNos(cardNos, 1, 2);
		if (null == result || result.isEmpty()) {
			return "";
		}
		
		return result.get(cardNo).get(0).getSn();
	}
	
	/**
	 * 验证手机号是否注册
	 * @param mobile 手机号
	 * @return true-已注册；false-未注册
	 * @throws SQLException 
	 */
	public static boolean checkMobileRegist(String mobile) throws SQLException {
		String sql = "SELECT COUNT(0) FROM t_account_extention WHERE c_mobile = ?";
		return dao.queryForLong(sql, mobile) > 0;
	}
	
	/**
	 * 注册手机和设备信息
	 * @param regist
	 * @return
	 * @throws Exception
	 */
	public static RegistMobileResult registMobileDevice(final RegistMobile regist) throws Exception {
		final String account = AccountCache.bulidAccount();
		
		try {
			List<Asset> deviceList = getAssets(regist.getDeviceId());
			
			if (null == deviceList || deviceList.isEmpty()) {
				logger.error("资产数据库中没有找到对应设备：" + regist.getDeviceId());			
				return new RegistMobileResult(RegistMobileStatus.DEVICE_NOT_EXIST);
			}
			
			Asset regAsset = deviceList.get(0);
			
			List<Asset> cardList = getAssets(regAsset.getRelatedAsset());
			
			String deviceName = null;
			String relationDeviceName = null;
			if (regAsset.getType().value == Type.PHONE.value
					|| regAsset.getType().value == Type.PAD.value) {
				deviceName = getDeviceName(account, regAsset.getType());
			}
			
			final long time = System.currentTimeMillis();
			
			final Object[][] param = new Object[1 + cardList.size() ][6];
			param[0][0] = account;
			param[0][1] = regAsset.getIdentity();
			param[0][2] = deviceName;
			param[0][3] = regAsset.getType().value;
			param[0][4] = regAsset.getCardNo();
			param[0][5] = time;
			
			if (!cardList.isEmpty()) {
				Asset relatedAsset = cardList.get(0);
				
				if (relatedAsset.getType().value == Type.PHONE.value
						|| relatedAsset.getType().value == Type.PAD.value) {
					relationDeviceName = getDeviceName(account, relatedAsset.getType());
				}
				
				param[1][0] = account;
				param[1][1] = relatedAsset.getIdentity();
				param[1][2] = relationDeviceName;
				param[1][3] = relatedAsset.getType().value;
				param[1][4] = relatedAsset.getCardNo();
				param[1][5] = time;
			}
			
			final String deviceId = regAsset.getType().value == Type.PHONE.value || regAsset.getType().value == Type.PAD.value
					? regist.getDeviceId().trim()
					: (!cardList.isEmpty() && (cardList.get(0).getType().value == Type.PHONE.value
						|| cardList.get(0).getType().value == Type.PAD.value) ? cardList.get(0).getIdentity() : "");
					
			final String cardSn = regAsset.getType().value == Type.PHONE.value || regAsset.getType().value == Type.PAD.value
					? regist.getCardNo().trim() + "-" + regist.getCertSn().trim()
					: (!cardList.isEmpty() && (cardList.get(0).getType().value == Type.PHONE.value
						|| cardList.get(0).getType().value == Type.PAD.value) 
						? cardList.get(0).getIdentity() + "-" + getCertSnByCardNo(cardList.get(0).getIdentity()) : "");
		
			
			dao.tx(new IAtom(){
				
				@Override
				public boolean run() throws SQLException {
					
					dao.insert(SQL_REG_ACCOUNT, account, regist.getPwd(), time);
					dao.insert(SQL_REG_ACCOUNT_EXTENTION, account, regist.getMobile());
					dao.batch(SQL_REG_ASSERT , param);
					
					//保证事务完整性
					if (StringUtils.isNotBlank(deviceId)) {
						RedisUtil.mdmHSet(deviceId, cardSn);
					}
					return true;
				}  
			});
			
			//发送SC消息
			List<String> accountList = new ArrayList<String>(1);
			accountList.add(account);
			isMessageSendSucess(accountList, MESSAGE_STATUS_ADD);
			
			return new RegistMobileResult(RegistMobileStatus.SUCCESS, account,
					StringUtils.isNotBlank(deviceName) ? deviceName
							: (StringUtils.isNotBlank(relationDeviceName) ? relationDeviceName : ""));
		} catch (Exception e) {
			if (StringUtils.isNotBlank(account)) {
				AccountCache.remove(account);
			}
			throw e;	
		}
	}
	
	/**
	 * 根据手机号查询账户信息
	 * @param mobile 手机号
	 * @return 账户信息
	 * @throws SQLException
	 */
	public static AccountInfo getAccountByMobile(String mobile) throws SQLException {
		StringBuffer sql = new StringBuffer();
		sql.append("SELECT info.n_id as id, info.c_account as account, info.c_password as pwd FROM t_account_info info");
		sql.append(" JOIN t_account_extention extention ON info.c_account = extention.c_account");
		sql.append(" WHERE extention.c_mobile = ?");
		
		return dao.queryForObject(AccountInfo.class, sql.toString(), mobile);
	}
	
	/**
	 * 根据卡号查询对应账户
	 * @param cardNo 卡号
	 * @return 对应账户
	 * @throws SQLException
	 */
	public static String getAccountByCardNo(String cardNo) throws SQLException {
		String sql = "SELECT c_account FROM t_account_asset WHERE c_card_no = ?";
		return dao.queryForStr(sql, cardNo);
	}
	
	/**
	 * 查询系统所有账户
	 * @return
	 * @throws SQLException
	 */
	public static List<String> getAccounts() throws SQLException {
		String sql = "SELECT c_account FROM t_account_info";

		return dao.queryForList(sql, new ResultSetHandler<List<String>>() {
			@Override
			public List<String> handle(ResultSet rs) throws SQLException {
				List<String> accounts = new ArrayList<>();

				while (rs.next()) {
					accounts.add(rs.getString("c_account"));
				}

				return accounts;
			}
		}, new Object[] {});
	}
	
	/**
	 * 验证账号是否存在
	 * @param account 待验证账号
	 * @return true-存在；false-不存在
	 * @throws SQLException
	 */
	public static boolean checkAccountExist(String account) throws SQLException {
		String sql = "SELECT COUNT(0) FROM t_account_info WHERE c_account = ?";
		return dao.queryForLong(sql, account) > 0;
	}
}
