package com.xdja.cssp.ras.business.impl;

import static com.xdja.cssp.ras.business.impl.CommonBusiness.getAccount;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.getAccountByCart;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.getAssetByAccountCart;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.getCardStatus;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.getAccountByMobile;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.getAccountByCardNo;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.xdja.cssp.acs.bean.cert.CertStatus;
import com.xdja.cssp.ras.business.interfaces.ILoginBusiness;
import com.xdja.cssp.ras.business.vo.AccountAsset;
import com.xdja.cssp.ras.business.vo.AccountInfo;
import com.xdja.cssp.ras.service.bean.auth.AccountCardAuth;
import com.xdja.cssp.ras.service.bean.auth.CardAuth;
import com.xdja.cssp.ras.service.bean.auth.MobileAuth;
import com.xdja.cssp.ras.service.bean.auth.UserAuth;
import com.xdja.cssp.ras.service.bean.enums.AccountCardAuthStatus;
import com.xdja.cssp.ras.service.bean.enums.CardAuthStatus;
import com.xdja.cssp.ras.service.bean.enums.ChiperAuthStatus;
import com.xdja.cssp.ras.service.bean.enums.MobileAuthStatus;
import com.xdja.cssp.ras.service.bean.enums.UserAuthStatus;
import com.xdja.cssp.ras.service.bean.result.CardAuthResult;
import com.xdja.platform.microservice.ServiceException;

/**
 * 登录认证业务类
 * @author liqi
 * @date 2015-2-5
 */
@Service
public class LoginBusinessImpl implements ILoginBusiness {

	private static final Logger logger = LoggerFactory.getLogger(LoginBusinessImpl.class);
			
	@Override
	public UserAuthStatus userAuth(UserAuth auth) throws Exception {
		
		AccountInfo account = getAccount(auth.getAccount());
		 
		//1 验证账户是否存在
		if(account == null || StringUtils.isBlank(account.getAccount())) { 
			logger.error("账户不存在:" + auth.getAccount()); 
		 
			return UserAuthStatus.ACCOUNT_NOT_EXIST;
			
		} else if (!auth.getPwd().equals(account.getPwd())) {
			logger.error("密码错误:" + account.getAccount()); 
			 
			return UserAuthStatus.PASSWORD_VALIDATE;
		} 
 		
		if(logger.isDebugEnabled()) {
			logger.debug(String.format("获取account信息开始:account=[%s];cardNo=[%s]", auth.getAccount(), auth.getCardNo()));
		}
		
		AccountAsset asset = getAssetByAccountCart(auth.getAccount(), auth.getCardNo());
		
		if (null == asset) {
			
			logger.error(String.format("帐号与安全卡不匹配:account=[%s];cardNao=[%s]", account.getAccount(), auth.getCardNo()));
			 
			return UserAuthStatus.ACCOUNT_CARD_NOT_ACCORD;
		}
		
		CertStatus status = getCardStatus(auth.getCardNo(), auth.getCertSn(), auth.getCaAlg());
 		UserAuthStatus userStatus;
 		
		switch(status) {
			case CARD_NOT_EXISTS:
				userStatus = UserAuthStatus.CARD_NOT_EXIST;
				break;
			case CARD_SN_DISACCORD:
				userStatus = UserAuthStatus.ACCOUNT_CARD_NOT_ACCORD;
				break;
			case CERT_FREEZE:
				userStatus = UserAuthStatus.CERT_FREEZE;
				break;
			case CERT_REVOKED:
				userStatus = UserAuthStatus.CERT_REVOKED;
				break; 
			default:
				userStatus = UserAuthStatus.SUCCESS;
				break;
				
		} 
			
		logger.info(String.format("账号[%s]安全卡[%s]认证结果:[%s]", auth.getAccount(), auth.getCardNo(), userStatus.msg));

		return userStatus;
	}

	@Override
	public ChiperAuthStatus chipAuth(CardAuth auth) throws Exception {
		
		ChiperAuthStatus chiperStatus;
		
		CertStatus status = getCardStatus(auth.getCardNo(), auth.getCertSn(), auth.getCaAlg());
		 
		switch(status) {
			case CARD_NOT_EXISTS:
				chiperStatus = ChiperAuthStatus.CARD_NOT_EXIST;
				break;
			case CARD_SN_DISACCORD:
				chiperStatus = ChiperAuthStatus.CARD_SN_NOT_ACCORD;
				break;
			case CERT_FREEZE: 
				chiperStatus = ChiperAuthStatus.CERT_FREEZE;
				break;
			case CERT_REVOKED:
				chiperStatus = ChiperAuthStatus.CERT_REVOKED;
				break;  
			default: 
				chiperStatus = ChiperAuthStatus.SUCCESS;
 				break; 
		} 	 
		
		logger.info(String.format("安全卡[%s]认证结果:[%s]", auth.getCardNo(), chiperStatus.msg ));
		
		return chiperStatus;
	}

	@Override
	public CardAuthResult cardAuth(CardAuth auth) throws Exception {
		
		CardAuthResult result = new CardAuthResult();
		//验证证书状态
		CertStatus status = getCardStatus(auth.getCardNo(), auth.getCertSn(), auth.getCaAlg());
		 
		switch(status) {
			case CARD_NOT_EXISTS: 
				 result.setStatus(CardAuthStatus.CARD_NOT_EXIST);
				break;
			case CARD_SN_DISACCORD:
				result.setStatus(CardAuthStatus.CARD_SN_NOT_ACCORD);
				break;
			case CERT_FREEZE:
				result.setStatus(CardAuthStatus.CERT_FREEZE);
				break;
			case CERT_REVOKED:
				result.setStatus(CardAuthStatus.CERT_REVOKED);
				break;  
			case SUCCESS:
				String account = getAccountByCart(auth.getCardNo());
				if(StringUtils.isBlank(account)) {  
					result.setStatus(CardAuthStatus.CERT_NOT_REGIST);
				} else {
					result.setAccount(account);
					result.setStatus(CardAuthStatus.SUCCESS);
				}
				break;
		}  
		
		logger.info(String.format("安全卡[%s]认证结果:[%s]", auth.getCardNo(), result.getStatus().msg));
		return result;
	}

	@Override
	public MobileAuthStatus mobileAuth(MobileAuth auth) {
		try {
			AccountInfo account = getAccountByMobile(auth.getMobile());

			if (null == account || StringUtils.isBlank(account.getAccount())) {
				return MobileAuthStatus.MOBILE_NOT_EXIST;
			}

			CertStatus status = getCardStatus(auth.getCardNo(), auth.getCertSn(), auth.getCaAlg());

			switch (status) {
				case CARD_NOT_EXISTS:
					return MobileAuthStatus.CARD_NOT_EXIST;
				case CARD_SN_DISACCORD:
					return MobileAuthStatus.CARD_SN_NOT_ACCORD;
				case CERT_FREEZE:
					return MobileAuthStatus.CERT_FREEZE;
				case CERT_REVOKED:
					return MobileAuthStatus.CERT_REVOKED;
			}

			if (!auth.getPwd().equals(account.getPwd())) {
				return MobileAuthStatus.PASSWORD_VALIDATE;
			}

			String cardAccount = getAccountByCardNo(auth.getCardNo());

			if (StringUtils.isBlank(cardAccount)) {
				return MobileAuthStatus.CARD_NOT_REGISTER;
			}

			if (!account.getAccount().equals(cardAccount)) {
				return MobileAuthStatus.MOBILE_CARD_NOT_ACCORD;
			}
		} catch (Exception e) {
			logger.error("手机认证时出错", e);
			throw ServiceException.create("手机认证时出错", e);
		}

		return MobileAuthStatus.SUCCESS;
	}
	
	@Override
	public AccountCardAuthStatus accountCardAuth(AccountCardAuth auth) {
		try {
			AccountInfo account = getAccount(auth.getAccount());
			 
			if (null == account || StringUtils.isBlank(account.getAccount())) {
				logger.debug("账户不存在 ：{}" , auth.getAccount());
				return AccountCardAuthStatus.ACCOUNT_NOT_EXIST;
			} 
	 		
			logger.debug("获取用户信息:account=[{}];cardNo=[{}]", auth.getAccount(), auth.getCardNo());
			
			AccountAsset asset = getAssetByAccountCart(auth.getAccount(), auth.getCardNo());
			
			if (null == asset) {
				logger.error("帐号与安全卡不匹配:account=[{}];cardNao=[{}]", account.getAccount(), auth.getCardNo());
				return AccountCardAuthStatus.ACCOUNT_CARD_NOT_ACCORD;
			}
			
			CertStatus status = getCardStatus(auth.getCardNo(), auth.getCertSn(), auth.getCaAlg());
			
			AccountCardAuthStatus accountCardStatus;
	 		
			switch(status) {
				case CARD_NOT_EXISTS:
					accountCardStatus = AccountCardAuthStatus.CARD_NOT_EXIST;
					break;
				case CARD_SN_DISACCORD:
					accountCardStatus = AccountCardAuthStatus.ACCOUNT_CARD_NOT_ACCORD;
					break;
				case CERT_FREEZE:
					accountCardStatus = AccountCardAuthStatus.CERT_FREEZE;
					break;
				case CERT_REVOKED:
					accountCardStatus = AccountCardAuthStatus.CERT_REVOKED;
					break; 
				default:
					accountCardStatus = AccountCardAuthStatus.SUCCESS;
					break;
					
			} 
				
			logger.info("账号[{}]安全卡[{}]认证结果:[{}]", auth.getAccount(), auth.getCardNo(), accountCardStatus.msg);
	
			return accountCardStatus;
		} catch (Exception e) {
			logger.error("账号安全卡认证时出错", e);
			throw ServiceException.create("账号安全卡认证时出错", e);
		}
	}
	
}
