package com.xdja.cssp.ras.business.impl;

import static com.xdja.cssp.ras.business.impl.CommonBusiness.checkDeviceCert;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.checkRegistCert;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.getAccount;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.getAccountAsset;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.getCardStatus;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.isDevExists;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.registAccountDevice;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.registDeviceOnly;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.registTestOnly;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.checkMobileRegist;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.bulidCheckCertResult;
import static com.xdja.cssp.ras.business.impl.CommonBusiness.registMobileDevice;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.xdja.cssp.acs.bean.cert.CertStatus;
import com.xdja.cssp.ras.business.interfaces.IRegistBusiness;
import com.xdja.cssp.ras.business.vo.AccountAsset;
import com.xdja.cssp.ras.business.vo.AccountInfo;
import com.xdja.cssp.ras.cache.AccountCache;
import com.xdja.cssp.ras.service.bean.auth.DeviceBind;
import com.xdja.cssp.ras.service.bean.auth.Regist;
import com.xdja.cssp.ras.service.bean.auth.RegistMobile;
import com.xdja.cssp.ras.service.bean.enums.DeviceBindStatus;
import com.xdja.cssp.ras.service.bean.enums.DeviceTestStatus;
import com.xdja.cssp.ras.service.bean.enums.RegistMobileStatus;
import com.xdja.cssp.ras.service.bean.enums.RegistStatus;
import com.xdja.cssp.ras.service.bean.result.DeviceBindResult;
import com.xdja.cssp.ras.service.bean.result.RegistMobileResult;
import com.xdja.cssp.ras.service.bean.result.RegistResult;
import com.xdja.platform.microservice.ServiceException;

/**
 * 注册信息业务实现类
 * @author michael
 * @date 2015-2-4
 */
@Service
public class RegistBusinessImpl implements IRegistBusiness {
	 
	private static final Logger logger = LoggerFactory.getLogger(RegistBusinessImpl.class);
 	
	
	@Override
	public RegistResult regist(Regist regist) throws Exception {

		try {
			// 1 验证账户是否存在
			if (AccountCache.checkAccountExist(regist.getAccount())) {
				logger.error("账户已存在:" + regist.getAccount());
				return new RegistResult(RegistStatus.ACCOUNT_EXIST);
			}

			// 2 验证资产是否存在
			if (!isDevExists(regist.getDeviceId())) {
				logger.error("该设备在资产库不存在:" + regist.getDeviceId());
				return new RegistResult(RegistStatus.DEVICE_NOT_EXIST);
			}

			// 3 验证证书是否有效
			CertStatus certState = getCardStatus(regist.getCardNo(), regist.getCertSn(), regist.getCaAlg());
			if (certState != CertStatus.SUCCESS) {

				return checkRegistCert(certState, regist);
			}

			// 校验设备是否已经注册
			List<AccountAsset> accountAsset = getAccountAsset(regist.getDeviceId(), regist.getCardNo());

			if (accountAsset != null && accountAsset.size() > 0) {

				for (AccountAsset aas : accountAsset) {
					if (aas.getAssetIdentify().equals(regist.getDeviceId())) {
						return new RegistResult(RegistStatus.DEVICE_REGISTED);
					}

					if (aas.getCardNo().equals(regist.getCardNo())) {
						return new RegistResult(RegistStatus.CARD_REGISTED);
					}
				}

			}

			return registAccountDevice(regist);
		} catch (Exception e) {
			logger.error("注册账户失败，将从账号缓存中删除账号，账号：{}", regist.getAccount());
			AccountCache.remove(regist.getAccount());
			throw e;
		}
	}
	
	

	@Override
	public DeviceBindResult registDevice(DeviceBind deviceBind) throws Exception {
		// 1 验证账号是否存在
		// 2 验证设备是否有效
		// 3验证证书是否有效
		// 4验证设备是否已经注册
		// 5验证证书是否已经注册
		// 6注册设备及证书
		 
		//2 验证资产是否存在
		if(!isDevExists(deviceBind.getDeviceId())) {
			logger.error("该设备在资产库不存在:" + deviceBind.getDeviceId());
			return new DeviceBindResult(DeviceBindStatus.DEVICE_NOT_EXIST);
		}
		
		//3 验证证书是否有效
		CertStatus certState = getCardStatus(deviceBind.getCardNo(), deviceBind.getCertSn(), deviceBind.getCaAlg());
		if(certState != CertStatus.SUCCESS) {
			
			return checkDeviceCert(certState, deviceBind);
		}  
		
		//4 校验设备是否已经注册
		List<AccountAsset> accountAsset = getAccountAsset(deviceBind.getDeviceId(), deviceBind.getCardNo());
		
		if(accountAsset != null && accountAsset.size() > 0) {
			
			for(AccountAsset aas : accountAsset) {
				if(aas.getAssetIdentify().equals(deviceBind.getDeviceId())) {
					return new DeviceBindResult(DeviceBindStatus.DEVICE_REGISTED);
				}
				
				if(aas.getCardNo().equals(deviceBind.getCardNo())) {
					return new DeviceBindResult(DeviceBindStatus.CARD_REGISTED);
				}
			}
			
		} 
		
		AccountInfo account =  getAccount(deviceBind.getAccount());
		
		//1 验证账户是否存在
		if (account == null || StringUtils.isBlank(account.getAccount())) { 
			logger.error("账号不存在:" + deviceBind.getAccount());
			return new DeviceBindResult(DeviceBindStatus.ACCOUNT_NOT_EXIST);
		} else if (!account.getPwd().equals(deviceBind.getPwd())) {
			return new DeviceBindResult(DeviceBindStatus.PASSWORD_VALIDATE);
		}
		
 		return registDeviceOnly(deviceBind);
	}



	@Override
	public DeviceTestStatus registTest(DeviceBind deviceBind) throws Exception {
		// 1 验证账号是否存在
		// 2 验证设备是否有效
		// 3验证证书是否有效
		// 4验证设备是否已经注册
		// 5验证证书是否已经注册
		// 6注册设备及证书
			 
		//2 验证资产是否存在
		if(!isDevExists(deviceBind.getDeviceId())) {
			logger.error("该设备在资产库不存在:" + deviceBind.getDeviceId());
			return DeviceTestStatus.DEVICE_NOT_EXIST;
		}
		
		//3 验证证书是否有效
		CertStatus certState = getCardStatus(deviceBind.getCardNo(), deviceBind.getCertSn(), deviceBind.getCaAlg());
		if(certState != CertStatus.SUCCESS) { 
			DeviceTestStatus result ;
			switch(certState) {
				case CARD_NOT_EXISTS:
					logger.error("CertStatus.CARDNOTEXISTS:" + deviceBind.getCardNo());
					result = DeviceTestStatus.CARD_NOT_EXIST;
					break;
				case CARD_SN_DISACCORD:
					logger.error("CertStatus.DISACCORD:" + deviceBind.getCardNo() + " " + deviceBind.getCertSn());
					result = DeviceTestStatus.CARD_SN_NOT_ACCORD;
					break;
				case CERT_FREEZE:
					logger.error("CertStatus.CERTBLOCK:" + deviceBind.getCardNo() + " " + deviceBind.getCertSn());
					result = DeviceTestStatus.CERT_FREEZE;
					break;
				case CERT_REVOKED: 
					logger.error("CertStatus.CERTREVOKE:" + deviceBind.getCardNo() + " " + deviceBind.getCertSn());
					result = DeviceTestStatus.CERT_REVOKED;
					break;
				default: 
					logger.error("ERROR STATUS CertStatus.SUCCESS:" + deviceBind.getCardNo() + " " + deviceBind.getCertSn());
					result = DeviceTestStatus.SUCCESS;
					break;
					
			}
			return result;
		}  
		
		//4 校验设备是否已经注册
		List<AccountAsset> accountAsset = getAccountAsset(deviceBind.getDeviceId(), deviceBind.getCardNo());
		
		if(accountAsset != null && accountAsset.size() > 0) {
			
			for(AccountAsset aas : accountAsset) {
				if(aas.getAssetIdentify().equals(deviceBind.getDeviceId())) {
					return DeviceTestStatus.DEVICE_REGISTED;
				}
				
				if(aas.getCardNo().equals(deviceBind.getCardNo())) {
					return DeviceTestStatus.CARD_REGISTED;
				}
			}
			
		} 
		
		AccountInfo account =  getAccount(deviceBind.getAccount());
		
		//1 验证账户是否存在
		if (account == null || StringUtils.isBlank(account.getAccount())) { 
			logger.error("账号不存在:" + deviceBind.getAccount());
			return DeviceTestStatus.ACCOUNT_NOT_EXIST;
		} else if (!account.getPwd().equals(deviceBind.getPwd())) {
			return DeviceTestStatus.PASSWORD_VALIDATE;
		}
		
		
 		return registTestOnly(deviceBind);
	}

	@Override
	public boolean checkMobile(String mobile) {
		try {
			return checkMobileRegist(mobile);
		} catch (Exception e) {
			logger.error(String.format("验证手机号是否注册时出错，手机号：%s", mobile), e);
			throw ServiceException.create(String.format("验证手机号是否注册时出错，手机号：%s", mobile), e);
		}
	}

	@Override
	public RegistMobileResult registMobile(RegistMobile regist) {
		try {
			if (checkMobileRegist(regist.getMobile())) {
				logger.error("手机号已存在，手机号：{}", regist.getMobile());
				return new RegistMobileResult(RegistMobileStatus.MOBILE_EXIST);
			}
			
			//2 验证资产是否存在
			if(!isDevExists(regist.getDeviceId())) {
				logger.error("该设备在资产库不存在:" + regist.getDeviceId());
				return new RegistMobileResult(RegistMobileStatus.DEVICE_NOT_EXIST);
			}
			
			//3 验证证书是否有效
			CertStatus certState = getCardStatus(regist.getCardNo(), regist.getCertSn(), regist.getCaAlg());
			if(certState != CertStatus.SUCCESS) {
				return bulidCheckCertResult(certState, regist);
			}  
			
			//校验设备是否已经注册
			List<AccountAsset> accountAsset = getAccountAsset(regist.getDeviceId(), regist.getCardNo());
			
			if (accountAsset != null && accountAsset.size() > 0) {
				for (AccountAsset aas : accountAsset) {
					if (aas.getAssetIdentify().equals(regist.getDeviceId())) {
						return new RegistMobileResult(RegistMobileStatus.DEVICE_REGISTED);
					}
	
					if (aas.getCardNo().equals(regist.getCardNo())) {
						return new RegistMobileResult(RegistMobileStatus.CARD_REGISTED);
					}
				}
			}
			
			return registMobileDevice(regist);
		} catch(Exception e) {
			logger.error("手机注册时出错", e);
			throw ServiceException.create("手机注册时出错", e);
		}
	}

	 
}
