package com.xdja.pki.itsca.oer.asn1;

import com.xdja.pki.itsca.oer.asn1.base.Enumerated;
import com.xdja.pki.itsca.oer.utils.ByteArrayUtils;
import org.bouncycastle.util.BigIntegers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigInteger;

/**
 * EccCurve ::= ENUMERATED {
 *     sgdsm2,
 *     nistP256,
 *     brainpoolP256r1,
 *     ...
 * }
 */
public class EccCurve extends Enumerated {

    private static Logger logger = LoggerFactory.getLogger(EccCurve.class);

    public static final Value SGD_SM2 = new Value(0, "SGD_SM2");
    public static final Value NIST_P_256 = new Value(1, "NIST_P_256");
    public static final Value BRAINPOOL_P_256_R1 = new Value(2, "BRAINPOOL_P_256_R1");

    public EccCurve(Value value) {
        super(value);
        this.addValueToList(SGD_SM2);
        this.addValueToList(NIST_P_256);
        this.addValueToList(BRAINPOOL_P_256_R1);
    }

    public static EccCurve getInstance(byte[] data) throws Exception {
        ByteArrayUtils.printHexBinary(logger, " EccCurve start data ", data);
        BigInteger choice = BigIntegers.fromUnsignedByteArray(data, 0, 1);
        EccCurve eccCurve ;
        if (choice.intValue() == 0){
            logger.debug("EccCurve choice SGD_SM2");
            eccCurve = new EccCurve(EccCurve.SGD_SM2);
        }else if (choice.intValue() == 1){
            logger.debug("EccCurve choice NIST_P_256");
            eccCurve = new EccCurve(EccCurve.NIST_P_256);
        }else if (choice.intValue() == 2){
            logger.debug("EccCurve choice BRAINPOOL_P_256_R1");
            eccCurve = new EccCurve(EccCurve.BRAINPOOL_P_256_R1);
        }else {
            logger.error("EccCurve choice is " + choice.intValue());
            throw new Exception("不支持的类型");
        }
        byte[] goal = new byte[data.length - 1];
        System.arraycopy(data, 1, goal, 0 ,goal.length);
        eccCurve.setGoal(goal);
        ByteArrayUtils.printHexBinary(logger, " EccCurve lave data ", eccCurve.getGoal());
        return eccCurve;
    }
}
