package com.xdja.pki.itsca.oer.utils;

import org.bouncycastle.asn1.*;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.ECKeyParameters;
import org.bouncycastle.crypto.params.ParametersWithID;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.crypto.signers.SM2Signer;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.BigIntegers;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.interfaces.ECPublicKey;
import java.util.Random;

public class BCUtils {
	static {
		if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
			Security.addProvider(new BouncyCastleProvider());
		}
	}

	public static byte[] generateRandomByBC(int len) throws NoSuchAlgorithmException {
		SecureRandom random = SecureRandom.getInstance("SHA1PRNG");
		return random.generateSeed(len);
	}

    /**
     * 生成 SM2 密钥对
     *
     * @return SM2 密钥对
     * @throws NoSuchProviderException            找不到相应的 Provider
     * @throws NoSuchAlgorithmException           找不到相应的算法
     * @throws InvalidAlgorithmParameterException KeyPairGenerator 初始化异常
     */
    public static KeyPair generateSM2KeyPairByBC() throws NoSuchProviderException, NoSuchAlgorithmException, InvalidAlgorithmParameterException {
        return X509Utils.generateSM2KeyPair();
    }


    private static final String SM4_ALGORITHM = "SM4";
	/************************************************************************************
	 *                                      加密解密                                    *
	 *                   GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3             *
	 ************************************************************************************/

	/**
	 * 使用 BC 进行加密
	 *
	 * @param publicKey  加密者公钥
	 * @param base64Data BASE64 编码 待加密消息
	 * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
	 */
	public static String sm2EncryptByBC(PublicKey publicKey, String base64Data) throws CryptoException, IOException {
		byte[] data = ByteArrayUtils.base64Decode(base64Data);
		//will throw IOException
		AsymmetricKeyParameter keyParameter = X509Utils.convertECPublicKeyParameters(publicKey);
		byte[] cipher = sm2EncryptByBC((ECKeyParameters) keyParameter, new SecureRandom(), data, 0, data.length);
		return ByteArrayUtils.base64Encode(cipher);
	}

	/**
	 * 使用 BC 进行解密
	 *
	 * @param privateKey 加密者私钥
	 * @param base64Data BASE64 编码 加密消息 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
	 * @return BASE64 编码 加密原文
	 */
	public static String sm2DecryptByBC(PrivateKey privateKey, String base64Data) throws IOException, InvalidCipherTextException {
		//will throw IOException
		AsymmetricKeyParameter keyParameter = X509Utils.convertECPrivateKeyKeyParameters(privateKey);
		byte[] data = ByteArrayUtils.base64Decode(base64Data);
		//will throw InvalidCipherTextException
		byte[] out = sm2DecryptByBC((ECKeyParameters) keyParameter, data, 0, data.length);
		return ByteArrayUtils.base64Encode(out);
	}

	public static String sm2DecryptByBC(PrivateKey privateKey, byte[] data) throws IOException, InvalidCipherTextException {
		//will throw IOException
		AsymmetricKeyParameter keyParameter = X509Utils.convertECPrivateKeyKeyParameters(privateKey);

		//will throw InvalidCipherTextException
		byte[] out = sm2DecryptByBC((ECKeyParameters) keyParameter, data, 0, data.length);
		return ByteArrayUtils.base64Encode(out);
	}

	/************************************************************************************
	 *                                      加密解密                                    *
	 *                      GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文               *
	 ************************************************************************************/

	/**
	 * 使用 BC 进行加密
	 *
	 * @param publicKey  加密者公钥
	 * @param base64Data BASE64 编码 待加密消息
	 * @return BASE64 编码 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
	 */
	public static String sm2EncryptASN1ByBC(PublicKey publicKey, String base64Data) throws CryptoException, IOException {
		byte[] data = ByteArrayUtils.base64Decode(base64Data);
		//will throw IOException
		byte[] cipher = sm2EncryptASN1ByBC(publicKey, data);
		return ByteArrayUtils.base64Encode(cipher);
	}

	/**
	 * 使用 BC 进行加密
	 *
	 * @param publicKey 加密者公钥
	 * @param data      BASE64 编码 待加密消息
	 * @return 二进制 编码 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
	 */
	public static byte[] sm2EncryptASN1ByBC(PublicKey publicKey, byte[] data) throws CryptoException, IOException {
		//will throw IOException
		AsymmetricKeyParameter keyParameter = X509Utils.convertECPublicKeyParameters(publicKey);
		byte[] cipher = sm2EncryptASN1ByBC((ECKeyParameters) keyParameter, new SecureRandom(), data, 0, data.length);
		return cipher;
	}

	/**
	 * 使用 BC 进行解密
	 *
	 * @param privateKey 加密者私钥
	 * @param base64Data BASE64 编码 加密消息
	 * @return BASE64 编码 加密原文
	 */
	public static String sm2DecryptASN1ByBC(PrivateKey privateKey, String base64Data) throws IOException, InvalidCipherTextException {
		byte[] data = ByteArrayUtils.base64Decode(base64Data);
		//will throw InvalidCipherTextException
		byte[] out = sm2DecryptASN1ByBC(privateKey, data);
		return ByteArrayUtils.base64Encode(out);
	}

	/**
	 * 使用 BC 进行解密
	 *
	 * @param privateKey 加密者私钥
	 * @return 加密原文
	 */
	public static byte[] sm2DecryptASN1ByBC(PrivateKey privateKey, byte[] data) throws IOException, InvalidCipherTextException {
		//will throw IOException
		AsymmetricKeyParameter keyParameter = X509Utils.convertECPrivateKeyKeyParameters(privateKey);
		//will throw InvalidCipherTextException
		byte[] out = sm2DecryptASN1ByBC((ECKeyParameters) keyParameter, data, 0, data.length);
		return out;
	}

	/**
	 * SM2 加密 输出 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
	 *
	 * @param keyParameter 密钥配置
	 * @param secureRandom 随机数
	 * @param input        输入数据
	 * @param inOff        输入起始位置
	 * @param length       长度
	 * @return 输出 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
	 * @throws InvalidCipherTextException
	 */
	public static byte[] sm2EncryptByBC(ECKeyParameters keyParameter, SecureRandom secureRandom, byte[] input, int inOff, int length) throws InvalidCipherTextException {
		SM2Engine sm2Engine = new SM2Engine();
		sm2Engine.init(true, new ParametersWithRandom(keyParameter, secureRandom));
		return sm2Engine.processBlock(input, inOff, length);
	}

	/**
	 * SM2 解密 输入 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
	 *
	 * @param keyParameter 密钥配置
	 * @param input        输入数据
	 * @param inOff        输入起始位置
	 * @param length       长度
	 * @return 明文
	 * @throws InvalidCipherTextException
	 */
	public static byte[] sm2DecryptByBC(ECKeyParameters keyParameter, byte[] input, int inOff, int length) throws InvalidCipherTextException {
		SM2Engine sm2Engine = new SM2Engine();
		sm2Engine.init(false, keyParameter);
		return sm2Engine.processBlock(input, inOff, length);
	}

	/**
	 * SM2 加密 输出 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
	 *
	 * @param keyParameter 密钥配置
	 * @param input        输入数据
	 * @param inOff        输入起始位置
	 * @param length       长度
	 * @return 输出 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
	 * @throws InvalidCipherTextException
	 */
	public static byte[] sm2EncryptASN1ByBC(ECKeyParameters keyParameter, SecureRandom secureRandom, byte[] input, int inOff, int length) throws InvalidCipherTextException, IOException {
		SM2Engine sm2Engine = new SM2Engine();
		//这里用来生成密文，与rsa不同的是不需要用pkcs#1结构体，传入参数为：公钥+随机数
		sm2Engine.init(true, new ParametersWithRandom(keyParameter, secureRandom));

		byte[] ciphertext = sm2Engine.processBlock(input, inOff, length);

		ECCurve curve = keyParameter.getParameters().getCurve();
		int curveLength = (curve.getFieldSize() + 7) / 8;

		byte[] c1 = new byte[curveLength * 2 + 1];
		System.arraycopy(ciphertext, inOff, c1, 0, c1.length);

		ECPoint c1P = curve.decodePoint(c1);
		byte[] xb = c1P.getXCoord().getEncoded();
		byte[] yb = c1P.getYCoord().getEncoded();

		SM3Digest digest = new SM3Digest();

		byte[] c2 = new byte[length];
		System.arraycopy(ciphertext, ciphertext.length - length - digest.getDigestSize(), c2, 0, length);

		byte[] c3 = new byte[digest.getDigestSize()];
		System.arraycopy(ciphertext, ciphertext.length - digest.getDigestSize(), c3, 0, digest.getDigestSize());

		ASN1SM2Cipher asn1SM2Cipher = new ASN1SM2Cipher(xb, yb, c3, c2);
		byte[] result = asn1SM2Cipher.toASN1Primitive().getEncoded();

		return result;
	}

	/**
	 * SM2 解密 输入 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
	 *
	 * @param keyParameter 密钥配置
	 * @param input        输入数据
	 * @return 明文
	 * @throws InvalidCipherTextException
	 */
	public static byte[] sm2DecryptASN1ByBC(ECKeyParameters keyParameter, byte[] input) throws InvalidCipherTextException {
		return sm2DecryptASN1ByBC(keyParameter, input, 0, input.length);
	}

	/**
	 * SM2 解密 输入 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
	 *
	 * @param keyParameter 密钥配置
	 * @param input        输入数据
	 * @param inOff        输入起始位置
	 * @param length       长度
	 * @return 明文
	 * @throws InvalidCipherTextException
	 */
	public static byte[] sm2DecryptASN1ByBC(ECKeyParameters keyParameter, byte[] input, int inOff, int length) throws InvalidCipherTextException {
		byte[] sm2CipherASN1 = new byte[length];
		System.arraycopy(input, inOff, sm2CipherASN1, 0, length);

		ASN1SM2Cipher asn1 = ASN1SM2Cipher.getInstance(sm2CipherASN1);

		//generate c1
		ECCurve curve = keyParameter.getParameters().getCurve();
		BigInteger x = asn1.getxCoordinate();
		BigInteger y = asn1.getyCoordinate();
		ECPoint point = curve.createPoint(x, y);
		byte[] c1 = point.getEncoded(false);

		//get c2
		byte[] c2 = asn1.getCipherText();

		//get c3
		byte[] c3 = asn1.getHash();

		byte[] ciphertext = Arrays.concatenate(c1, c2, c3);

		SM2Engine sm2Engine = new SM2Engine();
		//这里如果要是 decrypt 解密的话 需要传入 private key
		sm2Engine.init(false, keyParameter);

		byte[] M = sm2Engine.processBlock(ciphertext, 0, ciphertext.length);

		return M;
	}


	/************************************************************************************
	 *                                      SM2签名验签                                    *
	 ************************************************************************************/

	/**
	 * 使用 BC 进行签名
	 *
	 * @param privateKey 签名者私钥
	 * @param base64Data BASE64 编码 待签名消息
	 * @return BASE64 编码 签名数据
	 */
	public static String sm2SignByBC(PrivateKey privateKey, String base64Data) throws CryptoException, IOException {
		byte[] data = ByteArrayUtils.base64Decode(base64Data);
		byte[] sig = sm2SignByBC(privateKey, data);
		return ByteArrayUtils.base64Encode(sig);
	}

	/**
	 * 使用 BC 进行签名
	 *
	 * @param privateKey 签名者私钥
	 * @param data       待签名消息
	 * @return 签名数据
	 */
	public static byte[] sm2SignByBC(PrivateKey privateKey, byte[] data) throws CryptoException, IOException {
		//will throw IOException
		AsymmetricKeyParameter keyParameter = X509Utils.convertECPrivateKeyKeyParameters(privateKey);
		SM2Signer signer = new SM2Signer();
		ParametersWithRandom parametersWithRandom = new ParametersWithRandom(keyParameter, new SecureRandom());
		ParametersWithID parametersWithID = new ParametersWithID(parametersWithRandom, "China".getBytes());
		signer.init(true, parametersWithID);
		signer.update(data, 0, data.length);
		//will throw CryptoException
		return signer.generateSignature();
	}

	public static byte[][] sm2SignDerDecode(byte[] sign) throws IOException {
		ASN1Sequence var2 = ASN1Sequence.getInstance(ASN1Primitive.fromByteArray(sign));
		if (var2.size() != 2) {
			return null;
		} else {
			BigInteger var3 = ASN1Integer.getInstance(var2.getObjectAt(0)).getValue();
			BigInteger var4 = ASN1Integer.getInstance(var2.getObjectAt(1)).getValue();
			return new byte[][]{BigIntegers.asUnsignedByteArray(var3), BigIntegers.asUnsignedByteArray(var4)};
		}
	}

	public static byte[] sm2SignDerEncode(byte[] r, byte[] s) throws IOException {
		ASN1EncodableVector v = new ASN1EncodableVector();
		v.add(new ASN1Integer(BigIntegers.fromUnsignedByteArray(r)));
		v.add(new ASN1Integer(BigIntegers.fromUnsignedByteArray(s)));
		return new DERSequence(v).getEncoded(ASN1Encoding.DER);
	}

	/**
	 * 使用 BC 进行验签
	 *
	 * @param publicKey  签名者公钥
	 * @param base64Data BASE64 编码 签名原文
	 * @param base64Sign BASE64 编码 签名消息
	 * @return BASE64 编码 签名数据
	 */
	public static boolean sm2VerifyByBC(PublicKey publicKey, String base64Data, String base64Sign) throws IOException {
		byte[] data = ByteArrayUtils.base64Decode(base64Data);
		byte[] sign = ByteArrayUtils.base64Decode(base64Sign);
		return sm2VerifyByBC(publicKey, data, sign);
	}

	/**
	 * 使用 BC 进行验签
	 *
	 * @param publicKey 签名者公钥
	 * @param data      签名原文
	 * @param sign      签名消息
	 * @return 签名数据
	 */
	public static boolean sm2VerifyByBC(PublicKey publicKey, byte[] data, byte[] sign) throws IOException {
		//will throw IOException
		AsymmetricKeyParameter keyParameter = X509Utils.convertECPublicKeyParameters(publicKey);
		SM2Signer signer = new SM2Signer();
		ParametersWithID parametersWithID = new ParametersWithID(keyParameter, "China".getBytes());
		signer.init(false, parametersWithID);
		signer.update(data, 0, data.length);
		return signer.verifySignature(sign);
	}


	/************************************************************************************
	 *                                      SM3 HASH                                    *
	 ************************************************************************************/

	/**
	 * 使用 BC 摘要算法
	 *
	 * @param base64Data BASE64 原始数据
	 * @return BASE64 编码 摘要数据
	 */
	public static String sm3DigestByBC(String base64Data) {
		byte[] data = ByteArrayUtils.base64Decode(base64Data);
		byte[] digest = sm3DigestByBC(data);
		return ByteArrayUtils.base64Encode(digest);
	}

	/**
	 * 使用 BC 摘要算法
	 *
	 * @param data 原始数据
	 * @return BASE64 编码 摘要数据
	 */
	public static byte[] sm3DigestByBC(byte[] data) {
//        MessageDigest instance = MessageDigest.getInstance("SM3", BouncyCastleProvider.PROVIDER_NAME);
//        byte[] result = instance.digest(data);
//        return ByteArrayUtils.base64Encode(result);
		SM3Digest sm3Digest = new SM3Digest();
		sm3Digest.update(data, 0, data.length);
		byte[] out = new byte[sm3Digest.getDigestSize()];
		sm3Digest.doFinal(out, 0);
		return out;
	}

	/**
	 * 使用 BC 先把公钥进行摘要计算 在把得到的结果的需要进行摘要计算的数据合并进行 摘要算法
	 *
	 * @param publicKey  SM2公钥
	 * @param base64Data BASE64 原始数据
	 * @return BASE64 编码 摘要数据
	 */
	public static String sm3DigestByBCWithPublicKey(ECPublicKey publicKey, String base64Data) {
		byte[] data = ByteArrayUtils.base64Decode(base64Data);
		byte[] digest = sm3DigestByBCWithPublicKey(publicKey, data);
		return ByteArrayUtils.base64Encode(digest);
	}

	/**
	 * 使用 BC  先把公钥进行摘要计算 在把得到的结果的需要进行摘要计算的数据合并进行 摘要算法
	 *
	 * @param publicKey SM2公钥
	 * @param data      原始数据
	 * @return BASE64 编码 摘要数据
	 */
	public static byte[] sm3DigestByBCWithPublicKey(ECPublicKey publicKey, byte[] data) {
		SM3Digest digest = new SM3Digest();
		byte[] userID = ByteArrayUtils.hexDecode("31323334353637383132333435363738"); //default value
		byte[] x = publicKey.getW().getAffineX().toByteArray();
		byte[] y = publicKey.getW().getAffineY().toByteArray();

		int len = userID.length * 8;
		digest.update((byte) (len >> 8 & 0xFF));
		digest.update((byte) (len & 0xFF));
		digest.update(userID, 0, userID.length);

		//组装 Z 信息
		ECParameterSpec ecParameterSpec = ECNamedCurveTable.getParameterSpec(X509Utils.ECC_SM2_NAME);
		ECCurve curve = ecParameterSpec.getCurve();
		//得到椭圆心
		ECPoint G = ecParameterSpec.getG();

		addFieldElement(digest, curve.getA().toBigInteger());
		addFieldElement(digest, curve.getB().toBigInteger());
		addFieldElement(digest, G.getAffineXCoord().toBigInteger());
		addFieldElement(digest, G.getAffineYCoord().toBigInteger());
		addFieldElement(digest, BigIntegers.fromUnsignedByteArray(x));
		addFieldElement(digest, BigIntegers.fromUnsignedByteArray(y));

		byte[] z = new byte[digest.getDigestSize()];

		digest.doFinal(z, 0);

		digest.update(z, 0, z.length);
		digest.update(data, 0, data.length);

		byte[] out = new byte[digest.getDigestSize()];
		digest.doFinal(out, 0);


		return out;
	}

	private static void addFieldElement(Digest digest, BigInteger p) {
		byte[] in = ByteArrayUtils.asUnsignedByteArray32(p);
		digest.update(in, 0, in.length);
	}
	/************************************************************************************
	 *                                      SM4加解密                                *
	 ************************************************************************************/

	/**
	 * 使用 BC 进行加密 NoPadding填充  传入明文为16整数倍
	 *
	 * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
	 * @param base64Data BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
	 * @return BASE64 编码 密文消息
	 */
	public static String sm4EncryptByBCWithNoPadding(String base64Key, String base64Data) throws Exception {
		return ByteArrayUtils.base64Encode(symmetricECBEncryptByBC(true,
				EncryptTypeByBC.SM4_ECB_NoPadding,
				ByteArrayUtils.base64Decode(base64Key),
				ByteArrayUtils.base64Decode(base64Data)));
	}

	/**
	 * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
	 *
	 * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
	 * @param base64Data BASE64 编码 待加密消息
	 * @return BASE64 编码 密文消息
	 */
	public static String sm4EncryptByBCWithPKCS5Padding(String base64Key, String base64Data) throws Exception {
		return ByteArrayUtils.base64Encode(symmetricECBEncryptByBC(true,
				EncryptTypeByBC.SM4_ECB_PKCS5Padding,
				ByteArrayUtils.base64Decode(base64Key),
				ByteArrayUtils.base64Decode(base64Data)));
	}

	/**
	 * 使用 BC 进行加密 PKCS7Padding填充  传入明文为16整数倍
	 *
	 * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
	 * @param base64Data BASE64 编码 待加密消息
	 * @return BASE64 编码 密文消息
	 */
	public static String sm4EncryptByBCWithPKCS7Padding(String base64Key, String base64Data) throws Exception {
		return ByteArrayUtils.base64Encode(symmetricECBEncryptByBC(true,
				EncryptTypeByBC.SM4_ECB_PKCS7Padding,
				ByteArrayUtils.base64Decode(base64Key),
				ByteArrayUtils.base64Decode(base64Data)));
	}

	/**
	 * 使用 BC 进行解密 NoPadding填充
	 *
	 * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
	 * @param base64Data BASE64 编码 加密消息 转换为byte[]数组后长度为16整数倍
	 * @return BASE64 编码 密文消息
	 */
	public static String sm4DecryptByBCWithNoPadding(String base64Key, String base64Data) throws Exception {
		return ByteArrayUtils.base64Encode(symmetricECBEncryptByBC(false,
				EncryptTypeByBC.SM4_ECB_NoPadding,
				ByteArrayUtils.base64Decode(base64Key),
				ByteArrayUtils.base64Decode(base64Data)));
	}

	/**
	 * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
	 *
	 * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
	 * @param base64Data BASE64 编码 加密消息
	 * @return BASE64 编码 密文消息
	 */
	public static String sm4DecryptByBCWithPKCS5Padding(String base64Key, String base64Data) throws Exception {
		return ByteArrayUtils.base64Encode(symmetricECBEncryptByBC(false,
				EncryptTypeByBC.SM4_ECB_PKCS5Padding,
				ByteArrayUtils.base64Decode(base64Key),
				ByteArrayUtils.base64Decode(base64Data)));
	}

	/**
	 * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
	 *
	 * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
	 * @param base64Data BASE64 编码 加密消息
	 * @return BASE64 编码 密文消息
	 */
	public static String sm4DecryptByBCWithPKCS7Padding(String base64Key, String base64Data) throws Exception {
		return ByteArrayUtils.base64Encode(symmetricECBEncryptByBC(false,
				EncryptTypeByBC.SM4_ECB_PKCS7Padding,
				ByteArrayUtils.base64Decode(base64Key),
				ByteArrayUtils.base64Decode(base64Data)));
	}

	/**
	 * 使用 padding模式填充的加解密公用类
	 *
	 * @param forEncryption 是否为加密
	 * @param encryptType   加密填充类型
	 * @param key           16位byte[] 会话密钥
	 * @param data          二进制byte[]数据 待加密消息或已加密消息
	 * @return byte[] 明文消息
	 */
	public static byte[] symmetricECBEncryptByBC(
			boolean forEncryption, EncryptTypeByBC encryptType,
			byte[] key, byte[] data) throws Exception {
		Cipher cipher = Cipher.getInstance(encryptType.getName(), BouncyCastleProvider.PROVIDER_NAME);
		Key keySpec = new SecretKeySpec(key, SM4_ALGORITHM);

		if (forEncryption) {
			cipher.init(Cipher.ENCRYPT_MODE, keySpec);
		} else {
			cipher.init(Cipher.DECRYPT_MODE, keySpec);
		}
		return cipher.doFinal(data);
	}

	public enum EncryptTypeByBC {
		/**
		 * ECB加密模式无填充
		 */
		SM4_ECB_NoPadding("SM4/ECB/NoPadding"),

		/**
		 * ECB加密模式PKCS5Padding填充
		 */
		SM4_ECB_PKCS5Padding("SM4/ECB/PKCS5Padding"),

		/**
		 * ECB加密模式PKCS7Padding填充
		 */
		SM4_ECB_PKCS7Padding("SM4/ECB/PKCS7Padding"),

		/**
		 * CBC加密模式无填充
		 */
		SM4_CBC_NoPadding("SM4/CBC/NoPadding"),

		/**
		 * CBC加密模式PKCS5Padding填充
		 */
		SM4_CBC_PKCS5Padding("SM4/CBC/PKCS5Padding"),


		/**
		 * CBC加密模式PKCS7Padding填充
		 */
		SM4_CBC_PKCS7Padding("SM4/CBC/PKCS7Padding");


		private String name;

		EncryptTypeByBC(String name) {

			this.name = name;
		}

		public String getName() {
			return name;
		}
	}


}
