package com.xdja.pki.itsca.oer.asn1;

import com.xdja.pki.itsca.oer.asn1.base.Enumerated;
import com.xdja.pki.itsca.oer.utils.ByteArrayUtils;
import org.bouncycastle.util.BigIntegers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigInteger;

/***
 * -- 6.2.2.4 主题信息
 *
 * SubjectType ::= ENUMERATED {
 *     enrollmentCredential,
 *     authorizationTicket,
 *     authorizationAuthority,
 *     enrollmentAuthority,
 *     rootCa,
 *     crlSigner
 * }
 */
public class SubjectType extends Enumerated {
    private static Logger logger = LoggerFactory.getLogger(SubjectType.class);
    /**
     * 当与认证机构通信时，ITS设备应使用SubjectType为enrollmentCredential的SubjectInfo的证书
     * 此类证书禁止用于签发ITS设备注册证书（授权证书）或用于与其他ITS设备认证通信
     * 注册证书
     */
    public static final Value ENROLLMENT_CREDENTIAL = new Value(0, "ENROLLMENT_CREDENTIAL");
    /**
     * 当与其他ITS设备通信时，ITS设备应使用SubjectType为authorizationTicket的SubjectInfo的证书
     * 此类证书禁止用于签发ITS设备注册证书（授权证书）
     * 授权证书
     */
    public static final Value AUTHORIZATION_TICKET = new Value(1, "AUTHORIZATION_TICKET");
    /**
     * 授权机构为ITS设备签发授权证书时，SubjectType应当为authorizationAuthority
     * 授权CA证书
     */
    public static final Value AUTHORIZATION_AUTHORITY = new Value(2, "AUTHORIZATION_AUTHORITY");
    /**
     * 授权机构为ITS设备签发注册证书时，SubjectType应当为enrollmentAuthority
     * 注册CA证书
     */
    public static final Value ENROLLMENT_AUTHORITY = new Value(3, "ENROLLMENT_AUTHORITY");
    /**
     * 认证机构为其他认证机构签署证书时，SubjectType应当为rootCa
     * 根CA证书
     */
    public static final Value ROOT_CA = new Value(4, "ROOT_CA");
    /**
     * 证书撤销列表的签名者，crlSigner
     * 证书撤销CA证书
     */
    public static final Value CRL_SIGNER = new Value(5, "CRL_SIGNER");
    /**
     * 假名证书，pseudonymCertificate
     */
    public static final Value PSEUDONYM_CERTIFICATE = new Value(6, "PSEUDONYM_CERTIFICATE");
    /**
     * 假名CA证书 PCA证书，pseudonymAuthority
     */
    public static final Value PSEUDONYM_AUTHORITY = new Value(7, "PSEUDONYM_AUTHORITY");

    public static SubjectType getInstance(byte[] data) throws Exception {
        ByteArrayUtils.printHexBinary(logger, "SubjectType start data", data);
        BigInteger type = BigIntegers.fromUnsignedByteArray(data, 0, 1);
        SubjectType subjectType;
        if (type.intValue() == 0) {
            logger.debug("SubjectType choice ENROLLMENT_CREDENTIAL");
            subjectType = new SubjectType(SubjectType.ENROLLMENT_CREDENTIAL);
        } else if (type.intValue() == 1) {
            logger.debug("SubjectType choice AUTHORIZATION_TICKET");
            subjectType = new SubjectType(SubjectType.AUTHORIZATION_TICKET);
        } else if (type.intValue() == 2) {
            logger.debug("SubjectType choice AUTHORIZATION_AUTHORITY");
            subjectType = new SubjectType(SubjectType.AUTHORIZATION_AUTHORITY);
        } else if (type.intValue() == 3) {
            logger.debug("SubjectType choice ENROLLMENT_AUTHORITY");
            subjectType = new SubjectType(SubjectType.ENROLLMENT_AUTHORITY);
        } else if (type.intValue() == 4) {
            logger.debug("SubjectType choice ROOT_CA");
            subjectType = new SubjectType(SubjectType.ROOT_CA);
        } else if (type.intValue() == 5) {
            logger.debug("SubjectType choice CRL_SIGNER");
            subjectType = new SubjectType(SubjectType.CRL_SIGNER);
        }else if (type.intValue() == 6) {
            logger.debug("SubjectType choice PSEUDONYM_CERTIFICATE");
            subjectType = new SubjectType(SubjectType.PSEUDONYM_CERTIFICATE);
        }else if (type.intValue() == 7) {
            logger.debug("SubjectType choice PSEUDONYM_AUTHORITY");
            subjectType = new SubjectType(SubjectType.PSEUDONYM_AUTHORITY);
        } else {
            logger.error("SubjectType choice is error " + type.intValue());
            throw new Exception("不支持的类型");
        }
        byte[] goal = new byte[data.length - 1];
        System.arraycopy(data, 1, goal, 0, goal.length);
        subjectType.setGoal(goal);
        ByteArrayUtils.printHexBinary(logger, "SubjectType start data", subjectType.getGoal());
        return subjectType;
    }

    public SubjectType(Value value) {
        super(value);
        this.addValueToList(ENROLLMENT_CREDENTIAL);
        this.addValueToList(AUTHORIZATION_TICKET);
        this.addValueToList(AUTHORIZATION_AUTHORITY);
        this.addValueToList(ENROLLMENT_AUTHORITY);
        this.addValueToList(ROOT_CA);
        this.addValueToList(CRL_SIGNER);
        this.addValueToList(PSEUDONYM_CERTIFICATE);
        this.addValueToList(PSEUDONYM_AUTHORITY);
    }
}
