package com.xdja.pki.itsca.oer.utils;

import com.xdja.pki.gmssl.crypto.utils.GMSSLSM3DigestUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.Certificate;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaCertStore;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cms.CMSException;
import org.bouncycastle.cms.CMSProcessableByteArray;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.CMSSignedDataGenerator;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.util.PrivateKeyFactory;
import org.bouncycastle.crypto.util.PublicKeyFactory;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.jcajce.util.DefaultJcaJceHelper;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPrivateKeySpec;
import org.bouncycastle.jce.spec.ECPublicKeySpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.ContentVerifier;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.Store;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.security.provider.certpath.X509CertificatePair;

import java.io.*;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.*;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.util.*;

public class X509Utils {

    private static Logger logger = LoggerFactory.getLogger(X509Utils.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    public static final int ECC_KEY_BITS = 256;
    public static final String ECC_ALGORITHM_NAME = "EC";
    public static final String ECC_SM2_NAME = "sm2p256v1";
    public static final String X509_TYPE_NAME = "X.509";

    /************************************************************************************
     *                                SM2 生成公私钥
     ************************************************************************************/

    public static KeyPair generateSM2KeyPair() throws NoSuchProviderException, NoSuchAlgorithmException, InvalidAlgorithmParameterException {
        KeyPairGenerator kpGen = KeyPairGenerator.getInstance(ECC_ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME);
        kpGen.initialize(new ECGenParameterSpec(ECC_SM2_NAME), new SecureRandom());
        return kpGen.generateKeyPair();
    }

    /**
     * GM/T 0009-2012 7.1 密钥数据格式 SM2PublicKey :: = BIT STRING 内容为 04 || X || Y 分别标识公钥的 X 分量 Y 分量
     *
     * @param base64 base64 编码的 sm2 公钥
     * @return 公钥
     * @throws Exception
     */
    public static PublicKey convertSM2PublicKeyBitString(String base64) throws Exception {
        return convertSM2PublicKeyBitString(ByteArrayUtils.base64Decode(base64));
    }

    public static PublicKey convertSM2PublicKeyBitString(byte[] data) throws Exception {
        int digestLen = 32;
        byte[] x = new byte[digestLen];
        System.arraycopy(data, 1, x, 0, digestLen);
        byte[] y = new byte[digestLen];
        System.arraycopy(data, digestLen + 1, y, 0, digestLen);
        return convertSM2PublicKey(x, y);
    }

    public static String convertBitStringSM2PublicKey(ECPublicKey publicKey) throws Exception {
        return convertBitStringSM2PublicKey(publicKey.getW().getAffineX(), publicKey.getW().getAffineY());
    }

    public static String convertBitStringSM2PublicKey(BigInteger x, BigInteger y) throws Exception {
        return convertBitStringSM2PublicKey(ByteArrayUtils.asUnsignedByteArray32(x), ByteArrayUtils.asUnsignedByteArray32(y));
    }

    public static String convertBitStringSM2PublicKey(byte[] x, byte[] y) throws Exception {
        if (x.length != 32 || y.length != 32) {
            throw new Exception("x or y length error x: " + x.length + " y: " + y.length);
        }
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        outputStream.write(0x04);
        outputStream.write(x);
        outputStream.write(y);
        return ByteArrayUtils.base64Encode(outputStream.toByteArray());
    }

    public static PublicKey convertSM2PublicKey(byte[] x, byte[] y) throws Exception {
        return convertSM2PublicKey(BigIntegers.fromUnsignedByteArray(x), BigIntegers.fromUnsignedByteArray(y));
    }

    public static PublicKey convertSM2PublicKey(BigInteger x, BigInteger y) throws Exception {
//        //this will be convert too long
//        X9ECParameters curveParams = ECUtil.getNamedCurveByOid(GMObjectIdentifiers.sm2p256v1);
//        X962Parameters params = new X962Parameters(curveParams);
//        AlgorithmIdentifier algorithmIdentifier = new AlgorithmIdentifier(X9ObjectIdentifiers.id_ecPublicKey, params);
//        ECCurve ecCurve = curveParams.getCurve();
//        ECPoint point = ecCurve.createPoint(new BigInteger(x), new BigInteger(y));
//        SubjectPublicKeyInfo info = new SubjectPublicKeyInfo(algorithmIdentifier, new X9ECPoint(point, true).toASN1Primitive());

//        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(info.getEncoded());
//        PublicKey publicKey = KeyFactory.getInstance(ECC_ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME).generatePublic(keySpec);
//        PublicKey publicKey = convertSM2PublicKey(info);
//
//        X9ECParameters x9 = org.bouncycastle.asn1.x9.ECNamedCurveTable.getByName(ECC_SM2_NAME);
//        ECCurve curve = x9.getCurve();

        ECParameterSpec ecParameterSpec = ECNamedCurveTable.getParameterSpec(ECC_SM2_NAME);
        ECCurve curve = ecParameterSpec.getCurve();
        ECPublicKeySpec ecPublicKeySpec = new ECPublicKeySpec(curve.createPoint(x, y), ecParameterSpec);
        return new BCECPublicKey(ECC_SM2_NAME, ecPublicKeySpec, BouncyCastleProvider.CONFIGURATION);
        // TODO: 2019/5/9 linux fail with
//        java.security.spec.InvalidKeySpecException: key spec not recognized
//        at org.bouncycastle.jcajce.provider.asymmetric.util.BaseKeyFactorySpi.engineGeneratePublic(Unknown Source)
//        at org.bouncycastle.jcajce.provider.asymmetric.ec.KeyFactorySpi.engineGeneratePublic(Unknown Source)
//        at java.security.KeyFactory.generatePublic(KeyFactory.java:328)

//        KeyFactory keyFactory = KeyFactory.getInstance(ECC_ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME);
//        return keyFactory.generatePublic(ecPublicKeySpec);
    }

    public static PrivateKey convertSM2PrivateKey(byte[] s) throws Exception {
        return convertSM2PrivateKey(new BigInteger(s));
    }

    public static PrivateKey convertSM2PrivateKey(BigInteger s) throws Exception {
        ECParameterSpec ecParameterSpec = ECNamedCurveTable.getParameterSpec(ECC_SM2_NAME);
        ECPrivateKeySpec privateKeySpec = new ECPrivateKeySpec(s, ecParameterSpec);
        KeyFactory keyFactory = KeyFactory.getInstance(ECC_ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME);
        return keyFactory.generatePrivate(privateKeySpec);
    }

    public static PublicKey convertSM2PublicKey(SubjectPublicKeyInfo info) throws IOException, InvalidKeySpecException, NoSuchAlgorithmException {
        KeyFactory keyFactory = new DefaultJcaJceHelper().createKeyFactory(info.getAlgorithm().getAlgorithm().getId());
        return keyFactory.generatePublic(new X509EncodedKeySpec(info.getEncoded()));
//        JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME);
//        return converter.getPublicKey(info);
    }

    /************************************************************************************
     *                                Read 读取
     ************************************************************************************/

    public static PEMParser readPem(String path) throws Exception {
        InputStream inputStream = ClassLoader.getSystemResourceAsStream(path);
        if (inputStream == null) {
            inputStream = new FileInputStream(path);
        }
        InputStreamReader reader = new InputStreamReader(inputStream);
        return new PEMParser(reader);
    }

    public static KeyPair readKeyPairFromPEM(String path) throws Exception {
        PEMParser pemParser = readPem(path);
        PEMKeyPair pemPair = (PEMKeyPair) pemParser.readObject();

        KeyPair keyPair = new JcaPEMKeyConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getKeyPair(pemPair);
        return keyPair;
    }

    public static X509Certificate readCertificateFromPEM(String path) throws Exception {
        PEMParser pemParser = readPem(path);
        X509CertificateHolder holder = (X509CertificateHolder) pemParser.readObject();

        CertificateFactory certificateFactory = CertificateFactory.getInstance(X509_TYPE_NAME, BouncyCastleProvider.PROVIDER_NAME);
        InputStream inputStream = new ByteArrayInputStream(holder.getEncoded());
        X509Certificate certificate = (X509Certificate) certificateFactory.generateCertificate(inputStream);
        inputStream.close();
        return certificate;
    }

    public static X509Certificate readCertificateFromCer(String path) throws Exception {
        InputStream inputStream = new FileInputStream(path);
        return readCertificateFromCerInputStream(inputStream);
    }

    public static X509Certificate readCertificateFromCerByte(byte[] encode) throws CertificateException, NoSuchProviderException {
        InputStream inputStream = new ByteArrayInputStream(encode);
        return readCertificateFromCerInputStream(inputStream);
    }

    public static X509Certificate readCertificateFromCerInputStream(InputStream inputStream) throws CertificateException, NoSuchProviderException {
        CertificateFactory factory = CertificateFactory.getInstance(X509_TYPE_NAME, BouncyCastleProvider.PROVIDER_NAME);
        return (X509Certificate) factory.generateCertificate(inputStream);
    }

    public static X509Certificate readCertificateFromP12(String filename, char[] pw) throws Exception {
        KeyStore kse = readKeyStoreFromP12(filename, pw);
        String alias = kse.aliases().nextElement();
        return (X509Certificate) kse.getCertificate(alias);
    }

    public static List<X509Certificate> readCertificatesFromP12(String filename, char[] pw) throws Exception {
        KeyStore kse = readKeyStoreFromP12(filename, pw);
        List<X509Certificate> list = new ArrayList<>();
        if (kse.aliases().hasMoreElements()) {
            String alias = kse.aliases().nextElement();
            list.add((X509Certificate) kse.getCertificate(alias));
        }
        return list;
    }

    public static PublicKey readPublicKeyFromCer(String path) throws Exception {
        X509Certificate certificate = readCertificateFromCer(path);
        return certificate.getPublicKey();
    }

    public static PublicKey readPublicKeyFromPem(String path) throws Exception {
        PEMParser pemParser = readPem(path);
        SubjectPublicKeyInfo info = SubjectPublicKeyInfo.getInstance(pemParser.readObject());
        return convertSM2PublicKey(info);
    }

    public static PublicKey readPublicKeyFromP12(String filename, char[] pw) throws Exception {
        KeyStore kse = readKeyStoreFromP12(filename, pw);
        String alias = kse.aliases().nextElement();
        return kse.getCertificate(alias).getPublicKey();
    }

    public static PrivateKey readPrivateKeyFromPem(String path) throws Exception {
        return readKeyPairFromPEM(path).getPrivate();
        //  PEMParser pemParser = readPem(path);
        //  PrivateKey privateKey = (PrivateKey) pemParser.readObject();
        //  return privateKey;
    }

    public static PrivateKey readPrivateKeyFromP12(String filename, char[] pw) throws Exception {
        KeyStore kse = readKeyStoreFromP12(filename, pw);
        String alias = kse.aliases().nextElement();
        return (PrivateKey) kse.getKey(alias, pw);
    }

    public static KeyStore readKeyStoreFromP12(String filename, char[] pw) throws Exception {
        KeyStore kse = KeyStore.getInstance("pkcs12", BouncyCastleProvider.PROVIDER_NAME);
        InputStream inputStream = ClassLoader.getSystemResourceAsStream(filename);
        if (inputStream == null) {
            inputStream = new FileInputStream(filename);
        }
        kse.load(inputStream, pw);
        inputStream.close();
        return kse;
    }

    public static List<X509Certificate> getCertsByCertChain(InputStream inputStream) throws CMSException, CertificateException, IOException {
        List<X509Certificate> certificateList = new ArrayList<>();
        CMSSignedData cmsSignedData = new CMSSignedData(inputStream);
        Store<X509CertificateHolder> store = cmsSignedData.getCertificates();

        Collection collection = store.getMatches(null);
        Iterator<X509CertificateHolder> it = collection.iterator();
        while (it.hasNext()) {
            X509CertificateHolder x509CertificateHolder = it.next();
            X509Certificate x509Certificate = new JcaX509CertificateConverter().setProvider("BC").getCertificate(x509CertificateHolder);
            certificateList.add(x509Certificate);
        }
        inputStream.close();
        return certificateList;
    }

    public static String createCertChainByCerts(List<X509Certificate> certificateList) {

        CMSSignedDataGenerator gen = new CMSSignedDataGenerator();
        try {
            CMSProcessableByteArray msg = new CMSProcessableByteArray("".getBytes());
            JcaCertStore jcaCertStore = new JcaCertStore(certificateList);
            gen.addCertificates(jcaCertStore);
            CMSSignedData signedData = gen.generate(msg);
            return new String(Base64.encode(signedData.getEncoded()));
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }


    /************************************************************************************
     *                                Write 写入
     ************************************************************************************/

    public static void writeSdfPrivateKey(String path, String name, PrivateKey privateKey) throws IOException {
        path = path + File.separator;
        FileWriter fileWritter = new FileWriter(path + name + ".pem", false);
        fileWritter.write(privateKey.toString());
        fileWritter.close();
    }

    public static void writeCertificateToCer(String path, String name, X509Certificate certificate) throws Exception {
        File dir = new File(path);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        String filename = path + name + ".cer";
        FileOutputStream out = new FileOutputStream(filename);
        out.write(certificate.getEncoded());
        out.close();
    }

    public static void writeCertificateToPem(String path, String name, X509Certificate certificate) throws Exception {
        writeObjectToPem(path, name, certificate);
    }

    public static void writePublicKeyToPem(String path, String name, PublicKey publicKey) throws Exception {
        writeObjectToPem(path, name, publicKey);
    }

    public static void writeObjectToPem(String path, String name, Object object) throws Exception {
        File dir = new File(path);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        String filename = path + name + ".pem";
        FileWriter writer = new FileWriter(filename);
        writePEM(object, writer);
        writer.close();
    }

    public static void writePEM(Object object, Writer writer) throws Exception {
        JcaPEMWriter pw = new JcaPEMWriter(writer);
        pw.writeObject(object);
        pw.flush();
        pw.close();
    }

    public static void writeKeyStoreToP12(KeyStore keyStore, char[] password, String path, String alias) throws Exception {
        KeyStore write = KeyStore.getInstance("PKCS12", BouncyCastleProvider.PROVIDER_NAME);
        write.load(null, null);
        write.setCertificateEntry(alias, keyStore.getCertificate(alias));
        if (keyStore.getKey(alias, password) != null) {
            write.setKeyEntry(alias, keyStore.getKey(alias, password), password, keyStore.getCertificateChain(alias));
        }
        FileOutputStream out = new FileOutputStream(path + alias + ".p12");
        write.store(out, password);
        out.close();
    }

    public static void writeKeyStoreToP12(KeyStore keyStore, char[] password, String p12file) throws Exception {
        KeyStore write = KeyStore.getInstance("PKCS12", BouncyCastleProvider.PROVIDER_NAME);
        write.load(null, null);
        Enumeration<String> aliases = keyStore.aliases();
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            if (keyStore.getCertificate(alias) != null) {
                write.setCertificateEntry(alias, keyStore.getCertificate(alias));
            }
            if (keyStore.getKey(alias, password) != null) {
                write.setKeyEntry(alias, keyStore.getKey(alias, password), password, keyStore.getCertificateChain(alias));
            }
        }
        FileOutputStream out = new FileOutputStream(p12file + ".p12");
        write.store(out, password);
        out.close();
    }
    /************************************************************************************
     *                                文件文件夹删除复制
     ************************************************************************************/

    /**
     * 复制目录文件夹
     *
     * @param sourcePath 复制的源文件夹路径
     * @param bakPath    目标（备份）文件夹路径
     */
    public static void copyDir(String sourcePath, String bakPath) throws IOException {
        File bakFile = new File(bakPath);
        if (!bakFile.exists()) {
            bakFile.mkdir();
        }
        File sourceFile = new File(sourcePath);
        String[] list = sourceFile.list();
        if (list != null) {
            for (String s : list) {
                if ((new File(sourcePath + File.separator + s)).isFile()) {
                    copyFile(sourcePath + File.separator + s, bakPath + File.separator + s);
                }
                if (new File(sourcePath + File.separator + s).isDirectory()) {
                    copyDir(sourcePath + File.separator + s, bakPath + File.separator + s);
                }
            }
        }
    }

    /**
     * 复制单个文件
     *
     * @param oldPath 源文件路径
     * @param newPath 目标文件路径
     */
    public static void copyFile(String oldPath, String newPath) throws IOException {
        File oldFile = new File(oldPath);
        FileInputStream in = new FileInputStream(oldFile);
        copyFile(in, newPath);
    }

    public static void copyFile(FileInputStream inputStream, String newPath) throws IOException {
        File file = new File(newPath);
        FileOutputStream out = new FileOutputStream(file);
        byte[] buffer = new byte[2097152];
        int readByte = 0;
        while ((readByte = inputStream.read(buffer)) != -1) {
            out.write(buffer, 0, readByte);
        }
        inputStream.close();
        out.close();
    }


    /**
     * 删除目录（文件夹）以及目录下的文件
     *
     * @param sPath 被删除目录的文件路径
     * @return 目录删除成功返回true，否则返回false
     */
    public static boolean deleteDirectory(String sPath) {
        //如果sPath不以文件分隔符结尾，自动添加文件分隔符
        if (!sPath.endsWith(File.separator)) {
            sPath = sPath + File.separator;
        }
        File dirFile = new File(sPath);
        //如果dir对应的文件不存在，或者不是一个目录，则退出
        if (!dirFile.exists() || !dirFile.isDirectory()) {
            return false;
        }
        boolean flag = true;
        //删除文件夹下的所有文件(包括子目录)
        File[] files = dirFile.listFiles();
        if (files != null) {
            for (int i = 0; i < files.length; i++) {
                //删除子文件
                if (files[i].isFile()) {
                    flag = deleteFile(files[i].getAbsolutePath());
                    if (!flag) {
                        break;
                    }
                } //删除子目录
                else {
                    flag = deleteDirectory(files[i].getAbsolutePath());
                    if (!flag) {
                        break;
                    }
                }
            }
        }
        if (!flag) {
            return false;
        }
        //删除当前目录
        return dirFile.delete();
    }

    /**
     * 删除文件
     *
     * @param sPath 被删除的文件路径
     * @return 目录删除成功返回true，否则返回false
     */
    public static boolean deleteFile(String sPath) {
        boolean flag = false;
        File file = new File(sPath);
        // 路径为文件且不为空则进行删除
        if (file.isFile() && file.exists()) {
            flag = file.delete();
        }
        return flag;
    }

    /************************************************************************************
     *                                Verify 校验
     ************************************************************************************/

    public static boolean verifyCert(X509Certificate cert, PublicKey publicKey) throws Exception {
        try {
            Certificate certificate = convertCertificate(cert);
            return verify(certificate.getSignatureAlgorithm(), publicKey,
                    certificate.getTBSCertificate().getEncoded(),
                    certificate.getSignature().getOctets());
        } catch (Exception e) {
            throw new Exception("exception processing signature: " + e, e);
        }
    }

    public static boolean verifyCRL(X509CRL crl, PublicKey publicKey) throws Exception {
        try {
            CertificateList certificateList = convertCRL(crl);
            return verify(certificateList.getSignatureAlgorithm(), publicKey,
                    certificateList.getTBSCertList().getEncoded(),
                    certificateList.getSignature().getOctets());
        } catch (Exception e) {
            throw new Exception("exception processing signature: " + e, e);
        }
    }

    public static boolean verifyCert(X509Certificate cert, List<X509Certificate> certificates) throws Exception {
        boolean flag = false;
        for (int i = 0; i < certificates.size(); i++) {
            flag = verifyCert(cert, certificates.get(i).getPublicKey());
            if (flag) {
                return true;
            }
        }
        return false;
    }

    public static boolean verify(AlgorithmIdentifier algorithmIdentifier, PublicKey publicKey, byte[] encoded, byte[] expected) throws Exception {
        ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
        ContentVerifier verifier = verifierProvider.get(algorithmIdentifier);
        OutputStream out = verifier.getOutputStream();
        out.write(encoded);
        out.close();
        return verifier.verify(expected);
    }

    /************************************************************************************
     *                                convert 类型转换
     ************************************************************************************/

    public static AsymmetricKeyParameter convertECPublicKeyParameters(PublicKey publicKey) throws IOException {
        return PublicKeyFactory.createKey(SubjectPublicKeyInfo.getInstance(publicKey.getEncoded()));
    }

    public static AsymmetricKeyParameter convertECPrivateKeyKeyParameters(PrivateKey privateKey) throws IOException {
        return PrivateKeyFactory.createKey(PrivateKeyInfo.getInstance(privateKey.getEncoded()));
    }

    public static SubjectPublicKeyInfo convertSubjectPublicKeyInfo(PublicKey publicKey) {
        return SubjectPublicKeyInfo.getInstance(publicKey.getEncoded());
    }

    public static Certificate convertCertificate(X509Certificate cert) throws CertificateEncodingException, IOException {
        return Certificate.getInstance(ASN1Primitive.fromByteArray(cert.getEncoded()));
    }

    public static X509Certificate convertCertificate(Certificate cert) throws Exception {
        CertificateFactory certificateFactory = CertificateFactory.getInstance(X509_TYPE_NAME, BouncyCastleProvider.PROVIDER_NAME);
        return (X509Certificate) certificateFactory.generateCertificate(new ByteArrayInputStream(cert.getEncoded()));
    }

    public static X509CRL convertCRL(CertificateList certificateList) throws Exception {
        CertificateFactory certificateFactory = CertificateFactory.getInstance(X509_TYPE_NAME, BouncyCastleProvider.PROVIDER_NAME);
        return (X509CRL) certificateFactory.generateCRL(new ByteArrayInputStream(certificateList.getEncoded()));
    }

    public static CertificateList convertCRL(X509CRL certificateList) throws Exception {
        return CertificateList.getInstance(ASN1Primitive.fromByteArray(certificateList.getEncoded()));
    }

    public static org.bouncycastle.x509.X509CertificatePair convertCertificatePair(X509CertificatePair pair) throws Exception {
        return new org.bouncycastle.x509.X509CertificatePair(pair.getForward(), pair.getReverse());
    }

    public static X509CertificatePair convertCertificatePair(org.bouncycastle.x509.X509CertificatePair pair) throws Exception {
        return new X509CertificatePair(pair.getForward(), pair.getReverse());
    }

    /************************************************************************************
     *                                check 检查
     ************************************************************************************/

    public static boolean isCACertificate(X509Certificate certificate) {
        int basic = certificate.getBasicConstraints();
        return basic >= 0;
    }

    public static boolean isDRL(X509CRL crl) {
        Set<String> critical = crl.getCriticalExtensionOIDs();
        return critical != null && critical.contains(Extension.deltaCRLIndicator.getId());
    }

    public static boolean isARL(X509CRL crl) throws IOException {
        byte[] ext = crl.getExtensionValue(Extension.issuingDistributionPoint.getId());
        if (ext == null) {
            return false;
        }
        ASN1InputStream aIn = new ASN1InputStream(ext);
        ASN1OctetString octs = (ASN1OctetString) aIn.readObject();

        aIn = new ASN1InputStream(octs.getOctets());
        ASN1Primitive idp = aIn.readObject();

        IssuingDistributionPoint p = IssuingDistributionPoint.getInstance(idp);
        return p.onlyContainsCACerts();
    }

}
