package com.xdja.ca.sdk;

import com.alibaba.fastjson.JSON;
import com.xdja.ca.utils.CMPSendHttpUtils;
import com.xdja.ca.utils.P10Utils;
import com.xdja.ca.utils.SdkCertUtils;
import com.xdja.ca.utils.SdkFileUtils;
import com.xdja.ca.vo.FreeText;
import com.xdja.ca.vo.TempInfo;
import com.xdja.ca.vo.UpdateCertInfo;
import com.xdja.ca.vo.UserCertInfo;
import org.apache.log4j.Logger;
import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.cmp.*;
import org.bouncycastle.asn1.crmf.*;
import org.bouncycastle.asn1.pkcs.CertificationRequest;
import org.bouncycastle.asn1.pkcs.CertificationRequestInfo;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;

/**
 * CMP对外保留的api方法
 */
public class CmpApi {

    protected static transient final Logger logger = Logger.getLogger(CmpApi.class.getClass());

    static final ASN1ObjectIdentifier smAlgorithm = new ASN1ObjectIdentifier("1.2.156.10197.1");
    static final ASN1ObjectIdentifier SM3withSM2 = smAlgorithm.branch("501");

    private static X509Certificate cacert;

//    static {
//        // 生成事务id  TODO 定义生成
//        SecureRandom secureRandom = new SecureRandom();
//        byte[] trans = new byte[16];
//        secureRandom.nextBytes(trans);
//        transId = trans;
//
//        // CA证书   TODO 从本地获取上级CA证书
//        String certStr = SdkFileUtils.readByBytes("");
//        CmpApi.cacert = SdkCertUtils.getCertFromStr(certStr);
//
//    }

    /**
     * 请求产生随机数
     */
    public byte[] sendRandomNumReq(byte[] transId, String url) {
        String random = CMPSendHttpUtils.sendGet(url, "transId=" + Base64.toBase64String(transId));
        if (random == null || random == "") {
            return null;
        }
        byte[] nonce = Base64.decode(random.getBytes());
        return nonce;
    }

    /**
     * 用户证书申请 - 发送证书创建请求(1)
     *
     * @param userCertDN   用户自定义的证书DN
     * @param base64P10    base64编码的证书申请p10文件内容
     * @param certValidity 证书有效期  (天)
     * @param tempInfo     模板相关信息结构体
     * @param raDN         RA的唯一标识DN
     * @param caDN         CA的唯一标识DN
     * @return
     */
    public Result sendIssuerCertReqMessages(String raDN, String caDN, byte[] transId, int certValidity, TempInfo tempInfo, String base64P10, String userCertDN, String url) {
        Result result = new Result();

        // 1. 获取recipNonce值
        byte[] recipNonce = this.sendRandomNumReq(transId, url);
        // 获取P10信息中的申请DN和公钥信息
//        String certDn = null;    // 暂时依赖于用户页面输入的DN  不适用P10中的DN
        byte[] publicKey = null;
        ASN1ObjectIdentifier signAlgo = null;
        String signValue = null;
        if (base64P10 != null && base64P10 != "") {
//            certDn = P10Utils.p10ToCertDn(base64P10);
            publicKey = P10Utils.p10ToPublicKey(base64P10).getEncoded();
            signAlgo = P10Utils.p10ToSignAlgo(base64P10);
            signValue = P10Utils.p10ToSignValue(base64P10);
        }

        long certReqId = new Date().getTime();
        // 封装证书请求对象 CertRequset
        CertRequest certRequest = this.genCertRequest(certValidity, userCertDN, publicKey, signAlgo, certReqId);
        if (certRequest == null) {
            result.setErrCode(111);
            result.setErrMsg("封装certRequest错误");
            return result;
        }

        // 封装证书请求消息对象 CertReqMessages
        CertReqMessages certReqMessages = this.genCertReqMessages(false, certRequest, signAlgo, signValue);
        if (certReqMessages == null) {
            result.setErrCode(111);
            result.setErrMsg("封装certReqMessages错误");
            return result;
        }

        // RA产生十六字节随机数 senderNonce
        byte[] senderNonce = new byte[16];
        SecureRandom secureRandom = new SecureRandom();
        secureRandom.nextBytes(senderNonce);

        // 封装header中的freeText属性
        FreeText freeText = new FreeText();
        freeText.setTempInfo(tempInfo);

        Map<String, Object> map = CommonVariable.getMap();
        BaseCMPInfo baseCMPInfo = (BaseCMPInfo) map.get(new String(Base64.encode(transId)));
        if (baseCMPInfo == null) {
            map.put(new String(Base64.encode(transId)), new BaseCMPInfo(senderNonce, recipNonce, transId, certReqId, 0));
        }
        // 封装PKIMessage消息格式体
        PKIMessage pkiMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_INIT_REQ, recipNonce, senderNonce, transId, certReqMessages, JSON.toJSONString(freeText));
        if (pkiMessage == null) {
            result.setErrCode(111);
            result.setErrMsg("封装PKIMessage错误");
            return result;
        }

        // 2. 发送证书申请请求
        byte[] caResp = this.sendCmpHttpPost(pkiMessage, url);
        if (caResp == null) {
            result.setErrCode(222);
            result.setErrMsg("没有收到CA的返回结果");
            return result;
        }
        // 3. 检查返回消息的header和签名信息
        Result checkCRGResult = this.checkCmpHeaderAndSign(caResp, raDN, caDN, senderNonce, transId);
        if (!checkCRGResult.isSuccess()) {
            // 封装ErrorMsgContent返回给CA
            ErrorMsgContent errorMsgContent = this.genErrorMsgContent(PKIStatus.keyUpdateWaiting, checkCRGResult.getErrCode(), checkCRGResult.getErrMsg());
            PKIMessage errorPKIMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_ERROR, recipNonce, senderNonce, transId, errorMsgContent, "");
            this.sendCmpHttpPost(errorPKIMessage, url);

            result.setErrCode(checkCRGResult.getErrCode());
            result.setErrMsg(checkCRGResult.getErrMsg());
            return result;
        }

        // 获取返回body中的证书信息
        Result resolveCmpRepResult = this.resolveVarietyRepMessage(caResp, transId, userCertDN);
        if (!resolveCmpRepResult.isSuccess()) {
            if (resolveCmpRepResult.getErrCode() == -1) {
                result.setErrCode(resolveCmpRepResult.getErrCode());
                result.setErrMsg(resolveCmpRepResult.getErrMsg());
                return result;
            } else {
                // 封装ErrorMsgContent返回给CA
                ErrorMsgContent errorMsgContent = this.genErrorMsgContent(PKIStatus.waiting, resolveCmpRepResult.getErrCode(), resolveCmpRepResult.getErrMsg());
                PKIMessage errorPKIMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_ERROR, recipNonce, senderNonce, transId, errorMsgContent, "");
                this.sendCmpHttpPost(errorPKIMessage, url);
                result.setErrCode(resolveCmpRepResult.getErrCode());
                result.setErrMsg(resolveCmpRepResult.getErrMsg());
                return result;
            }
        }

        result.setInfo(resolveCmpRepResult.getInfo());
        result.setSuccess(true);
        return result;
    }

    /**
     * 用户证书申请 - 发送证书更新请求
     *
     * @param certValidity   证书有效期  (天)
     * @param tempInfo       模板相关信息结构体
     * @param raDN           RA的唯一标识DN
     * @param caDN           CA的唯一标识DN
     * @param updateCertInfo 更新信息结构体
     * @return
     */
    public Result sendUpdateCertReqMessages(String raDN, String caDN, byte[] transId, int certValidity, TempInfo tempInfo, UpdateCertInfo updateCertInfo, String signSn, String userCertDN, String url) {
        Result result = new Result();
        byte[] recipNonce = this.sendRandomNumReq(transId, url);
        // 获取P10信息中的申请DN和公钥信息
        byte[] publicKey = null;
        ASN1ObjectIdentifier signAlgo = null;
        String signValue = null;
        if (updateCertInfo != null) {
            publicKey = updateCertInfo.getPublicKey();   // 密钥可更新 可不更新 不更新时 不传入公钥信息
            signAlgo = updateCertInfo.getSignAlgo();
            signValue = updateCertInfo.getSignValue();
        }

        long certReqId = new Date().getTime();

        // 封装证书请求对象 CertRequset
        CertRequest certRequest = this.genCertRequest(certValidity, userCertDN, publicKey, signAlgo, certReqId);

        // 封装证书请求消息对象 CertReqMessages
        CertReqMessages certReqMessages = this.genCertReqMessages(true, certRequest, signAlgo, signValue);
        if (certReqMessages == null) {
            result.setErrCode(111);
            result.setErrMsg("封装certReqMessages错误");
            return result;
        }

        // RA产生十六字节随机数
        byte[] senderNonce = new byte[16];
        SecureRandom secureRandom = new SecureRandom();
        secureRandom.nextBytes(senderNonce);

        // 封装header中的freeText属性
        FreeText freeText = new FreeText();
        freeText.setTempInfo(tempInfo);
        freeText.setSignSn(signSn);

        Map<String, Object> map = CommonVariable.getMap();
        BaseCMPInfo baseCMPInfo = (BaseCMPInfo) map.get(new String(Base64.encode(transId)));
        if (baseCMPInfo == null) {
            map.put(new String(Base64.encode(transId)), new BaseCMPInfo(senderNonce, recipNonce, transId, certReqId, 0));
        }

        // 封装PKIMessage消息格式体
        PKIMessage pkiMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_KEY_UPDATE_REQ, recipNonce, senderNonce, transId, certReqMessages, JSON.toJSONString(freeText));

        // 2. 发送证书申请请求
        byte[] caResp = this.sendCmpHttpPost(pkiMessage, url);
        if (caResp == null) {
            result.setErrCode(222);
            result.setErrMsg("没有收到CA的返回结果");
            return result;
        }

        // 检查返回消息的header和签名信息
        Result checkCRGResult = this.checkCmpHeaderAndSign(caResp, raDN, caDN, senderNonce, transId);
        if (!checkCRGResult.isSuccess()) {
            // 封装ErrorMsgContent返回给CA
            ErrorMsgContent errorMsgContent = this.genErrorMsgContent(PKIStatus.rejection, checkCRGResult.getErrCode(), checkCRGResult.getErrMsg());
            PKIMessage errorPKIMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_ERROR, recipNonce, senderNonce, transId, errorMsgContent, "");
            this.sendCmpHttpPost(errorPKIMessage, url);

            result.setErrCode(checkCRGResult.getErrCode());
            result.setErrMsg(checkCRGResult.getErrMsg());
            return result;
        }

        // 获取返回body中的证书信息
        Result resolveCmpRepResult = this.resolveVarietyRepMessage(caResp, transId, userCertDN);
        if (!resolveCmpRepResult.isSuccess()) {
            if (resolveCmpRepResult.getErrCode() == -1) {
                result.setErrCode(resolveCmpRepResult.getErrCode());
                result.setErrMsg(resolveCmpRepResult.getErrMsg());
                return result;
            } else {
                // 封装ErrorMsgContent返回给CA
                ErrorMsgContent errorMsgContent = this.genErrorMsgContent(PKIStatus.waiting, resolveCmpRepResult.getErrCode(), resolveCmpRepResult.getErrMsg());
                PKIMessage errorPKIMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_ERROR, recipNonce, senderNonce, transId, errorMsgContent, "");
                this.sendCmpHttpPost(errorPKIMessage, url);
                result.setErrCode(resolveCmpRepResult.getErrCode());
                result.setErrMsg(resolveCmpRepResult.getErrMsg());
                return result;
            }
        }
        result.setInfo(resolveCmpRepResult.getInfo());
        result.setSuccess(true);
        return result;
    }

    /**
     * 用户证书申请 - 发送证书撤销请求
     *
     * @param raDN   RA的唯一标识DN
     * @param caDN   CA的唯一标识DN
     * @param signSn 需要撤销的签名证书序列号
     * @return
     */
    public Result sendRecoveryCertReqMessages(String raDN, String caDN, byte[] transId, long sn, String signSn, String userCertDN, String url) {
        Result result = new Result();
        byte[] recipNonce = this.sendRandomNumReq(transId, url);

        // 生成 RevReqContent结构体
        RevReqContent revReqContent = this.genRevReqContent(caDN, userCertDN, sn);

        byte[] senderNonce = new byte[16];
        SecureRandom secureRandom = new SecureRandom();
        secureRandom.nextBytes(senderNonce);

        // 封装PKIMessage消息格式体
        PKIMessage pkiMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_REVOCATION_REQ, recipNonce, senderNonce, transId, revReqContent, "");

        // 发送证书申请请求
        if (pkiMessage == null) {
            result.setErrCode(111);
            result.setErrMsg("封装PKIMessage错误");
            return result;
        }

        byte[] caResp = this.sendCmpHttpPost(pkiMessage, url);


        // 3. 检查返回消息的header和签名信息
        Result checkCRGResult = this.checkCmpHeaderAndSign(caResp, raDN, caDN, senderNonce, transId);
        if (!checkCRGResult.isSuccess()) {
            // 封装ErrorMsgContent返回给CA
            ErrorMsgContent errorMsgContent = this.genErrorMsgContent(PKIStatus.keyUpdateWaiting, checkCRGResult.getErrCode(), checkCRGResult.getErrMsg());
            PKIMessage errorPKIMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_ERROR, recipNonce, senderNonce, transId, errorMsgContent, "");
            this.sendCmpHttpPost(errorPKIMessage, url);

            result.setErrCode(checkCRGResult.getErrCode());
            result.setErrMsg(checkCRGResult.getErrMsg());
            return result;
        }

        // 解析撤销响应结构体
        Result resolveCmpRepResult = this.resolveVarietyRepMessage(caResp, transId, userCertDN);
        if (!resolveCmpRepResult.isSuccess()) {
            // 封装ErrorMsgContent返回给CA
            ErrorMsgContent errorMsgContent = this.genErrorMsgContent(PKIStatus.rejection, resolveCmpRepResult.getErrCode(), resolveCmpRepResult.getErrMsg());
            PKIMessage errorPKIMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_ERROR, recipNonce, senderNonce, transId, errorMsgContent, "");
            this.sendCmpHttpPost(errorPKIMessage, url);

            result.setErrCode(resolveCmpRepResult.getErrCode());
            result.setErrMsg(resolveCmpRepResult.getErrMsg());
            return result;
        }

        result.setInfo(resolveCmpRepResult.getInfo());
        result.setSuccess(true);
        return result;
    }

    /**
     * 发送证书申请正确确认消息到CA
     *
     * @param raDN
     * @param caDN
     * @param recipNonce
     */
    public void sendCertConfirmContent(String raDN, String caDN, byte[] recipNonce, byte[] transId, long certReqId, String url) {
        // 封装PKIConfirmContent返回给CA  TODO hash值待定
        CertConfirmContent certConfirmContent = this.genCertConfirmContent("syg", certReqId);
        PKIMessage certConfirmPKIMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_CERT_CONFIRM, recipNonce, null, transId, certConfirmContent, "");
        this.sendCmpHttpPost(certConfirmPKIMessage, url);
    }

    /**
     * 发送错误确认消息到CA
     *
     * @param raDN
     * @param caDN
     * @param recipNonce
     * @param errorCode
     * @param errorMsg
     */
    public void sendErrorMsgContent(String raDN, String caDN, byte[] recipNonce, byte[] transId, int errorCode, String errorMsg, String url) {
        // 封装ErrorMsgContent返回给CA
        ErrorMsgContent errorMsgContent = this.genErrorMsgContent(PKIStatus.rejection, errorCode, errorMsg);
        PKIMessage errorPKIMessage = this.genPKIMessage(raDN, caDN, PKIBody.TYPE_ERROR, recipNonce, null, transId, errorMsgContent, "");
        this.sendCmpHttpPost(errorPKIMessage, url);
    }

    /*************************************************** 私有方法 *************************************************************/
    /**
     * 生成证书请求对象CertRequest (1)
     *
     * @param certValidity
     * @param userCertDN
     * @param publicKey
     * @param signAlg
     * @return
     */
    private CertRequest genCertRequest(int certValidity, String userCertDN, byte[] publicKey, ASN1ObjectIdentifier signAlg, long certReqId) {
        // 有效期
        OptionalValidity myOptionalValidity = null;
        if (certValidity != 0) {
            Date notBefore = new Date();
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(notBefore);
            calendar.add(Calendar.DATE,certValidity);
            Date notAfter = calendar.getTime();
            ASN1EncodableVector optionalValidityV = new ASN1EncodableVector();
            optionalValidityV.add(new DERTaggedObject(true, 0, new Time(notBefore)));
            optionalValidityV.add(new DERTaggedObject(true, 1, new Time(notAfter)));
            myOptionalValidity = OptionalValidity.getInstance(new DERSequence(optionalValidityV));
        }

        // 封装证书模板  可根据CA颁发证书模板来进行设置
        final CertTemplateBuilder myCertTemplate = new CertTemplateBuilder();
        myCertTemplate.setVersion(1);

        //  TODO 序列号 由CA侧进行补充
//        myCertTemplate.setSerialNumber(new ASN1Integer(1));
        //  TODO 证书签发者 这个由CA侧进行补充
//        myCertTemplate.setIssuer(issuerDN);

        myCertTemplate.setSigningAlg(new AlgorithmIdentifier(signAlg));
        myCertTemplate.setValidity(myOptionalValidity);
        myCertTemplate.setSubject(new X500Name(userCertDN));
        if (publicKey == null) {
            return null;
        }
        SubjectPublicKeyInfo keyInfo = SubjectPublicKeyInfo.getInstance(publicKey);
        myCertTemplate.setPublicKey(keyInfo);

        return new CertRequest(new ASN1Integer(certReqId), myCertTemplate.build(), null);
    }


    /**
     * 生成CertReqMsg (1)
     *
     * @param raVerifiedPopo
     * @param certRequest
     * @return
     * @throws Exception
     */
    private CertReqMessages genCertReqMessages(boolean raVerifiedPopo, CertRequest certRequest, ASN1ObjectIdentifier signAlg, String signValue) {
        ProofOfPossession myProofOfPossession = null;

        if (raVerifiedPopo && signValue != null) {
            // TODO RA调用密码机验签
            SubjectPublicKeyInfo publicKey = certRequest.getCertTemplate().getPublicKey();
            try {
                Signature signature = Signature.getInstance("SM3WithSM2");
                signature.initVerify((PublicKey) publicKey.parsePublicKey());
                boolean verify = signature.verify(signValue.getBytes());
                if (!verify) {
                    logger.info("RA验证签名POP失败");
                    return null;
                }
            } catch (Exception e) {
                logger.error("RA验证POP异常，原因{}", e);
                return null;
            }
        }
//        else {
//            // 用私钥对证书请求对象进行签名
//            ByteArrayOutputStream baos = new ByteArrayOutputStream();
//            DEROutputStream mout = new DEROutputStream(baos);
//            try {
//                mout.writeObject(certRequest);
//                mout.close();
//            } catch (IOException e) {
//                logger.error("生成CertReqMessages，读取certRequest信息失败，原因：{}", e);
//            }
//            byte[] popoProtectionBytes = baos.toByteArray();
//            Signature sig = null;
//            try {
//                sig = Signature.getInstance(PKCSObjectIdentifiers.sha1WithRSAEncryption.getId());
//                sig.initSign(this.keyPair.getPrivate());
//                sig.update(popoProtectionBytes);
//                DERBitString bs = new DERBitString(sig.sign());
//                POPOSigningKey myPOPOSigningKey = new POPOSigningKey(null, new AlgorithmIdentifier(PKCSObjectIdentifiers.sha1WithRSAEncryption), bs);
//                myProofOfPossession = new ProofOfPossession(myPOPOSigningKey);
//            } catch (Exception e) {
//                logger.error("生成CertReqMessages，使用私钥签名失败，原因：{}", e);
//            }
//        }

//        AttributeTypeAndValue av = new AttributeTypeAndValue(CRMFObjectIdentifiers.id_regCtrl_regToken, new DERUTF8String("foo123"));
//        AttributeTypeAndValue[] avs = {av};

        CertReqMsg myCertReqMsg = new CertReqMsg(certRequest, myProofOfPossession, null);
        CertReqMessages myCertReqMessages = new CertReqMessages(myCertReqMsg);
        return myCertReqMessages;
    }


    /**
     * 封装PKIMessage (1)
     *
     * @param raDN
     * @param caDN
     * @param tagNo
     * @param recipNonce
     * @param senderNonce
     * @param asn1Encodable certReqMessages ErrorMsgContent
     * @param freeText
     * @return
     * @throws Exception
     */
    private PKIMessage genPKIMessage(String raDN, String caDN, int tagNo, byte[] recipNonce, byte[] senderNonce, byte[] transId, ASN1Encodable asn1Encodable, String freeText) {
        PKIBody myPKIBody = new PKIBody(tagNo, asn1Encodable);        // initialization request
        GeneralName raName = new GeneralName(new X500Name(raDN));
        GeneralName caName = new GeneralName(new X500Name(caDN));
        final PKIHeaderBuilder myPKIHeader = new PKIHeaderBuilder(PKIHeader.CMP_1999, raName, caName);
        myPKIHeader.setMessageTime(new ASN1GeneralizedTime(new Date()));
        myPKIHeader.setSenderNonce(new DEROctetString(senderNonce));
        myPKIHeader.setRecipNonce(new DEROctetString(recipNonce));
        myPKIHeader.setTransactionID(new DEROctetString(transId));
        myPKIHeader.setProtectionAlg(new AlgorithmIdentifier(SM3withSM2));


        // 用于接收者可读的消息
        myPKIHeader.setFreeText(new PKIFreeText(freeText));

        // TODO RA后续调密码机进行签名
        Signature sig = null;
        PKIMessage pkiMessage = null;
        SecureRandom sr = new SecureRandom(String.valueOf(System.nanoTime()).getBytes());
        KeyPairGenerator kpg = null;
        try {
            kpg = KeyPairGenerator.getInstance("EC", new BouncyCastleProvider());
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        kpg.initialize(256, sr);
        KeyPair keyPair = kpg.generateKeyPair();
        PrivateKey privateKey = keyPair.getPrivate();
        SdkFileUtils.saveAs(new String(Base64.encode(privateKey.getEncoded())), "F:\\cmp_cert\\prikey.dat");
        PublicKey publicKey = keyPair.getPublic();
        SdkFileUtils.saveAs(new String(Base64.encode(publicKey.getEncoded())), "F:\\cmp_cert\\pubkey.dat");

        try {
            PKIHeader myHeader = myPKIHeader.build();
            sig = Signature.getInstance("SM3withSM2", new BouncyCastleProvider());
            sig.initSign(privateKey);
            sig.update(this.getProtectedBytes(myHeader, myPKIBody));
            pkiMessage = new PKIMessage(myHeader, myPKIBody, new DERBitString(sig.sign()));
//            pkiMessage = new PKIMessage(myHeader, myPKIBody,null);
        } catch (Exception e) {
            logger.error("封装PKIMessage异常，原因:{}", e);
        }
        return pkiMessage;
    }


    /**
     * 通过Http的方式发送PKIMessage (1)
     *
     * @param pkiMessage
     * @return
     */
    private byte[] sendCmpHttpPost(PKIMessage pkiMessage, String url) {
        byte[] message = new byte[0];
        try {
            message = pkiMessage.getEncoded();
        } catch (IOException e) {
            logger.error("PKIMessage的encode异常，原因{}", e);
        }
        byte[] result = CMPSendHttpUtils.sendPost(url, new String(Base64.encode(message)));

        byte[] pkiMessageByte = Base64.decode(result);
        return pkiMessageByte;
    }


    /**
     * 检查CMP响应  检查消息头(1)
     *
     * @param pkiMessage
     * @return
     */
    private Result checkCmpHeaderAndSign(byte[] pkiMessage, String raName, String caName, byte[] senderNonce, byte[] transId) {
        Result result = new Result();
        PKIMessage respObject = PKIMessage.getInstance(pkiMessage);
        if (respObject == null) {
            String errMsg = "No pkiMessage response message.";
            logger.info(errMsg);
            result.setErrMsg(errMsg);
            return result;
        }

        // 检测接收到的发送者是不是预期的CA发送的
        PKIHeader header = respObject.getHeader();
        if (header == null) {
            String errMsg = "No header in response message.";
            logger.info(errMsg);
            result.setErrMsg(errMsg);
            return result;
        }
        X500Name caDN = X500Name.getInstance(header.getSender().getName());
        if (caDN == null || !caDN.equals(new X500Name(caName))) {
            String errMsg = "received caDn is:" + caDN.toString() + " but expect:" + new X500Name(caName).toString();
            logger.info(errMsg);
            result.setErrMsg(errMsg);
            return result;
        }


        // 判断CA返回的nonce的长度
        if (header.getSenderNonce().getOctets().length != 16) {
            String errMsg = "Wrong length of received sender nonce (made up by server). Is " + header.getSenderNonce().getOctets().length + " byte but should be 16.";
            logger.info(errMsg);
            result.setErrMsg(errMsg);
            return result;
        }

        // 检测CA返回的nonce是否和senderNonce一致
        if (!Arrays.equals(header.getSenderNonce().getOctets(), senderNonce)) {
            String errMsg = "recipient nonce not the same as we sent away as the sender nonce. Sent: " + Arrays.toString(senderNonce) + " Received: " + Arrays.toString(header.getRecipNonce().getOctets());
            logger.info(errMsg);
            result.setErrMsg(errMsg);
            return result;
        }

        // 检测CA返回的transId是否和请求的一致
        if (!Arrays.equals(header.getTransactionID().getOctets(), transId)) {
            String errMsg = "transid is not the same as the one we sent";
            logger.info(errMsg);
            result.setErrMsg(errMsg);
            return result;
        }

        // TODO RA获取CertRepMessage中的签名值
//        PublicKey caPublicKey = null;
//        // 验证签名
//        final String id;
//        {
//            final AlgorithmIdentifier algId = header.getProtectionAlg();
//            if (algId == null || algId.getAlgorithm() == null || algId.getAlgorithm().getId() == null) {
//                String errMsg = "Not possible to get algorithm.";
//                logger.info(errMsg);
//                result.setErrMsg(errMsg);
//                return result;
//            }
//            id = algId.getAlgorithm().getId();
//
//            byte[] protBytes = this.getProtectedBytes(respObject);
//            final DERBitString bs = respObject.getProtection();
//            final Signature sig;
//            try {
//                sig = Signature.getInstance(id);
//                sig.initVerify(caPublicKey);
//                sig.update(protBytes);
//                if (!sig.verify(bs.getBytes())) {
//                    String errMsg = "CA signature not verifying";
//                    logger.info(errMsg);
//                    result.setErrMsg(errMsg);
//                    return result;
//                }
//            } catch (Exception e) {
//                String errMsg = "Not possible to verify signature.原因:" + e;
//                logger.info(errMsg);
//                result.setErrMsg(errMsg);
//                return result;
//            }
//        }
        result.setSuccess(true);
        return result;
    }


    /**
     * 解析各种CA响应的消息体
     *
     * @param pkiMessage
     * @param transId
     * @return
     * @throws IOException
     */
    private Result resolveVarietyRepMessage(byte[] pkiMessage, byte[] transId, String userCertDN) {
        Result result = new Result();
        PKIMessage respObject = PKIMessage.getInstance(pkiMessage);
        PKIBody body = respObject.getBody();


        // 处理23-ErrorMsgContent 结构体
        if (body.getType() == PKIBody.TYPE_ERROR) {
            Result errMsgResult = this.resolveErrorMsgContent(body);
            result.setErrMsg("CA返回ErrorCode:" + errMsgResult.getErrCode() + " errMsg:" + errMsgResult.getErrMsg());
            result.setErrCode(-1);
            return result;
        }
        // 处理1，3-CertRepMessage结构体
        else if (body.getType() == PKIBody.TYPE_INIT_REP || body.getType()== PKIBody.TYPE_CERT_REP) {
            Result certRepResult = this.resolveCertRepMessage(body, userCertDN,transId);
            if (!certRepResult.isSuccess()) {
                result.setErrMsg(certRepResult.getErrMsg());
                result.setErrCode(certRepResult.getErrCode());
                return result;
            }
            result.setSuccess(true);
            result.setInfo(certRepResult.getInfo());
        }
        // 处理12-RevRepContent结构体
        else if (body.getType() == PKIBody.TYPE_REVOCATION_REP) {
            Result revRepResult = this.resolveRevRepContent(body);
            if (!revRepResult.isSuccess()) {
                result.setErrMsg(revRepResult.getErrMsg());
                result.setErrCode(revRepResult.getErrCode());
                return result;
            }
            result.setSuccess(true);
            result.setInfo(revRepResult.getInfo());
        } else {
            String errMsg = "Cert body tag is:" + body.getType();
            logger.info(errMsg);
            result.setErrMsg(errMsg);
        }
        return result;
    }


    /**
     * 解析CertRepMessage结构体
     *
     * @param body
     * @return
     */
    private Result resolveCertRepMessage(PKIBody body, String userCertDN, byte[] transId) {
        Result result = new Result();
        // 获取certRepMessage结构体
        CertRepMessage certRepMessage = (CertRepMessage) body.getContent();
        if (certRepMessage == null) {
            String errMsg = "No CertRepMessage for certificate received.";
            logger.info(errMsg);
            result.setErrMsg(errMsg);
            return result;
        }

        // 获取证书链
        CMPCertificate[] caPubs = certRepMessage.getCaPubs();

        UserCertInfo userCertResp = new UserCertInfo();
        CertResponse[] certResponses = certRepMessage.getResponse();

        for (CertResponse certRespons : certResponses) {
            // 依次获取证书返回CertResponse
            final CertResponse resp = certRespons;
            if (resp == null) {
                String errMsg = "No CertResponse for certificate received.";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }

            // 与请求体中的requestId对应匹配
            Map<String, Object> map = CommonVariable.getMap();
            BaseCMPInfo baseCMPInfo = (BaseCMPInfo) map.get(new String(Base64.encode(transId)));
            if (baseCMPInfo == null) {
                String errMsg = "No ra send transId.";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }
            long requestId = baseCMPInfo.getRequestId();
            long receiveCertReqId = resp.getCertReqId().getValue().longValue();
            if (receiveCertReqId != requestId) {
                String errMsg = "Received CertReqId is " + receiveCertReqId + " but should be " + requestId;
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }

            // 获取返回对象中的PKIStatusInfo结构体
            final PKIStatusInfo info = resp.getStatus();
            if (info == null) {
                String errMsg = "No PKIStatusInfo for certificate received.";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }

            // 获取返回的PkiStatus内容
            int pkiStatus = info.getStatus().intValue();
            if (pkiStatus != 0) {
                String errMsg = "Received Status is " + pkiStatus + " but should be 0";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }

            // 获取CertifiedKeyPair结构体
            final CertifiedKeyPair certifiedKeyPair = resp.getCertifiedKeyPair();
            if (certifiedKeyPair == null) {
                String errMsg = "No CertifiedKeyPair for certificate received.";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }

            // 获取CertOrEncCert结构体
            final CertOrEncCert certOrEncCert = certifiedKeyPair.getCertOrEncCert();
            if (certOrEncCert == null) {
                String errMsg = "No CertOrEncCert for certificate received.";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }

            final CMPCertificate cmpCertificate = certOrEncCert.getCertificate();
            if (cmpCertificate == null) {
                String errMsg = "No X509CertificateStructure for certificate received.";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }
            // 获取返回中的证书结构体
            byte encoded[] = new byte[0];
            try {
                encoded = cmpCertificate.getEncoded();
                if (encoded == null || encoded.length <= 0) {
                    String errMsg = "No encoded certificate received.";
                    logger.info(errMsg);
                    result.setErrMsg(errMsg);
                    return result;
                }
            } catch (IOException e) {
                String errMsg = "CMPCertificate Encode Error.";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }

            X509Certificate cert = (X509Certificate) SdkCertUtils.convertUploadFileToCert((encoded));
            if (cert == null) {
                String errMsg = "Not possbile to create certificate.";
                logger.info(errMsg);
                result.setErrMsg(errMsg);
                return result;
            }

            // 检测生成的证书中用户DN，是否和申请时的DN一致
            X500Name certDN = X500Name.getInstance(cert.getSubjectX500Principal().getEncoded());
            if (certDN.hashCode() != new X500Name(userCertDN).hashCode()) {
                String errMsg = "Subject is '" + certDN.toString() + "' but should be '" + userCertDN;
                logger.info(errMsg);
                result.setErrMsg(errMsg);
//                return result;
            }
            // TODO 检测用户证书的颁发者DN，是否和申请时的DN一致
            String caDN = "CN = grgrgvreg,L = 金水区,L = 郑州市,C = CN";
            if (cert.getIssuerX500Principal().hashCode() != caDN.hashCode()) {
//                String errMsg = "Issuer is '" + cert.getIssuerX500Principal().getName() + "' but should be '" + this.cacert.getSubjectX500Principal();
                String errMsg = "Issuer is '" + cert.getIssuerX500Principal().getName() + "' but should be '" + caDN;
                logger.info(errMsg);
                result.setErrMsg(errMsg);
//                return result;
            }

            //  TODO RA验证是否是上级CA签发的
//            try {
//                cert.verify(this.cacert.getPublicKey());
//            } catch (Exception e) {
//                String errMsg = "Certificate not verifying. See exception" + e;
//                logger.info(errMsg);
//                result.setErrMsg(errMsg);
//                return result;
//            }


            // 如果是加密证书，获取返回结构体中的加密私钥
            boolean signCert = SdkCertUtils.isSignCert(cert);
            if (signCert) {
                String base64SignCert = SdkCertUtils.certToFullB64(cert);
                userCertResp.setSignCert(base64SignCert);
            } else {
                // TODO RA此处应该使用Ukey进行解密
                KeyPair protocolEncKey = null;
                EncryptedValue encValue = certifiedKeyPair.getPrivateKey();
//                AsymmetricKeyUnwrapper unwrapper = new JceAsymmetricKeyUnwrapper(encValue.getKeyAlg(), protocolEncKey.getPrivate());
//                byte[] secKeyBytes = new byte[0];
//                try {
//                    secKeyBytes = (byte[]) unwrapper.generateUnwrappedKey(encValue.getKeyAlg(), encValue.getEncSymmKey().getBytes()).getRepresentation();
//                } catch (OperatorException e) {
//                    String errMsg = "unwrapper.generateUnwrappedKey exception: " + e;
//                    logger.info(errMsg);
//                    result.setErrMsg(errMsg);
//                    return result;
//                }
//
//                // recover private key
//                PKCS8EncryptedPrivateKeyInfo respInfo = null;
//                try {
//                    respInfo = new PKCS8EncryptedPrivateKeyInfo(encValue.getEncValue().getBytes());
//                } catch (IOException e) {
//                    String errMsg = "EncValue.getEncValue().getBytes() exception: " + e;
//                    logger.info(errMsg);
//                    result.setErrMsg(errMsg);
//                    return result;
//                }
//                PrivateKeyInfo keyInfo = null;
//                try {
//                    keyInfo = respInfo.decryptPrivateKeyInfo(new JceInputDecryptorProviderBuilder().setProvider("BC").build(secKeyBytes));
//                } catch (PKCSException e) {
//                    String errMsg = "respInfo.decryptPrivateKeyInfo exception: " + e;
//                    logger.info(errMsg);
//                    result.setErrMsg(errMsg);
//                    return result;
//                }
//                String base64EncCert = SdkCertUtils.certToFullB64(cert);
//                userCertResp.setEncCert(base64EncCert);
//                try {
//                    if (keyInfo != null) {
//                        userCertResp.setEncPriKey(keyInfo.getEncoded());
//                    }
//                } catch (IOException e) {
//                    String errMsg = "PrivateKeyInfo getEncoded() 异常: " + e;
//                    logger.info(errMsg);
//                    result.setErrMsg(errMsg);
//                    return result;
//                }
            }
        }
        result.setSuccess(true);
        result.setInfo(userCertResp);
        return result;
    }


    /**
     * 解析错误消息结构体
     *
     * @param body
     * @return
     */
    private Result resolveErrorMsgContent(PKIBody body) {
        Result result = new Result();
        ErrorMsgContent errorMsgContent = (ErrorMsgContent) body.getContent();
        PKIStatusInfo pkiStatusInfo = errorMsgContent.getPKIStatusInfo();
        ASN1Integer errorCode = errorMsgContent.getErrorCode();
        PKIFreeText errorDetails = errorMsgContent.getErrorDetails();
        result.setErrCode(errorCode.getValue().intValue());
        result.setErrMsg(errorDetails.getStringAt(0).toString());
        return result;
    }


    /**
     * 解析撤销响应结构体
     *
     * @param body
     * @return
     */
    private Result resolveRevRepContent(PKIBody body) {
        Result result = new Result();
        RevRepContent revRepContent = (RevRepContent) body.getContent();
        PKIStatusInfo[] pkiStatusInfos = revRepContent.getStatus();
//        int errCode = PKIStatus.GRANTED;
//        if (errCode == pkiStatusInfos[0].getStatus().intValue()) {
        logger.info("撤销成功");
        result.setSuccess(true);
        result.setErrCode(pkiStatusInfos[0].getFailInfo().intValue());
        result.setErrMsg(pkiStatusInfos[0].getStatusString().toString());
        result.setInfo(pkiStatusInfos[0].getStatus().intValue());
//        }
        return result;
    }


    /**
     * 生成证书确认PKIMessage
     *
     * @param hash
     * @return
     */
    private CertConfirmContent genCertConfirmContent(String hash, long certReqId) {
        CertStatus certStatus = new CertStatus(hash.getBytes(), new BigInteger(String.valueOf(certReqId)));
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(certStatus);
        CertConfirmContent certConfirmContent = CertConfirmContent.getInstance(new DERSequence(v));
        return certConfirmContent;
    }


    /**
     * 封装 撤销 RevReqContent
     *
     * @param caDN
     * @param userCertDN
     * @param sn
     * @return
     */
    private RevReqContent genRevReqContent(String caDN, String userCertDN, long sn) {
        CertTemplateBuilder certTemplateBuilder = new CertTemplateBuilder();
        certTemplateBuilder.setIssuer(new X500Name(caDN));
        certTemplateBuilder.setSubject(new X500Name(userCertDN));
        certTemplateBuilder.setSerialNumber(new ASN1Integer(sn));

        // 此处可以封装多个需要撤销的证书信息 目前只封装一个
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(certTemplateBuilder.build());
        ASN1Sequence seq = new DERSequence(v);
        RevDetails revDetails = RevDetails.getInstance(seq);
        RevReqContent revReqContent = new RevReqContent(revDetails);
        return revReqContent;
    }


    /**
     * 生成错误确认消息
     *
     * @param pkiStatus    状态  PKIStatus.rejection
     * @param errorCode    错误码
     * @param errorDetails 错误细节描述
     * @return
     */
    private ErrorMsgContent genErrorMsgContent(PKIStatus pkiStatus, int errorCode, String errorDetails) {
        PKIFreeText pkiFreeText = new PKIFreeText(errorDetails);
        PKIStatusInfo pkiStatusInfo = new PKIStatusInfo(pkiStatus);
        ErrorMsgContent errorMessage = new ErrorMsgContent(pkiStatusInfo, new ASN1Integer(errorCode), pkiFreeText);
        return errorMessage;
    }


    /**
     * 生成错误确认消息
     *
     * @param pkiStatus    状态  PKIStatus.rejection
     * @param statusString 状态信息
     * @param failInfo     失败信息  PKIFailureInfo.badDataFormat
     * @param errorCode    错误码
     * @param errorDetails 错误细节描述
     * @return
     */
    private ErrorMsgContent genErrorMsgContent(int pkiStatus, String statusString, int failInfo, int errorCode, String errorDetails) {
        PKIFailureInfo pkiFailureInfo = new PKIFailureInfo(failInfo);
        PKIFreeText pkiFreeText = new PKIFreeText(statusString);
        PKIStatusInfo pkiStatusInfo = new PKIStatusInfo(PKIStatus.getInstance(pkiStatus), pkiFreeText, pkiFailureInfo);
        PKIFreeText errorDetailsPkiFreeText = new PKIFreeText(errorDetails);
        ErrorMsgContent errorMessage = new ErrorMsgContent(pkiStatusInfo, new ASN1Integer(errorCode), errorDetailsPkiFreeText);
        return errorMessage;
    }


    /**
     * 将PKIMessage结构体转为为header和body字节数组
     */
    private byte[] getProtectedBytes(PKIMessage msg) {
        return getProtectedBytes(msg.getHeader(), msg.getBody());
    }

    /**
     * 获取header和body组合的字节数组
     *
     * @param header
     * @param body
     * @return
     */
    private byte[] getProtectedBytes(PKIHeader header, PKIBody body) {
        byte[] res = null;
        ASN1EncodableVector v = new ASN1EncodableVector();
        v.add(header);
        v.add(body);
        ASN1Encodable protectedPart = new DERSequence(v);
        try {
            ByteArrayOutputStream bao = new ByteArrayOutputStream();
            DEROutputStream out = new DEROutputStream(bao);
            out.writeObject(protectedPart);
            res = bao.toByteArray();
        } catch (Exception ex) {
            logger.error(ex.getLocalizedMessage(), ex);
        }
        return res;
    }

    /**
     * 将DER编码的公钥封装成P10证书请求
     * @param pubDer
     * @param dn
     * @param caAlg
     * @return
     */
    public String createP10FromPubKeyDer(byte[] pubDer, String dn, String caAlg) {
        String p10 = "";
        try {
            ASN1Sequence seq = ASN1Sequence.getInstance(pubDer);
            SubjectPublicKeyInfo spi = new SubjectPublicKeyInfo(seq);
            CertificationRequestInfo cf1 = new CertificationRequestInfo(new X509Name(dn), spi, null);

            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            DEROutputStream derOutStream = new DEROutputStream(outStream);
            derOutStream.writeObject(cf1);
            byte[] tbsdata = outStream.toByteArray();
            DERBitString dbs = new DERBitString(tbsdata);
            CertificationRequest cr = new CertificationRequest(cf1, new AlgorithmIdentifier(new DERObjectIdentifier(
                    "1.2.840.113549.1.1.4")), dbs);
            ByteArrayOutputStream outStream1 = new ByteArrayOutputStream();
            DEROutputStream derOutStream1 = new DEROutputStream(outStream1);
            derOutStream1.writeObject(cr);
            byte[] p10data = outStream1.toByteArray();

            p10 = new String(Base64.encode(p10data)).replace("\n", "").replaceAll("\r", "");
        } catch (Exception e) {
            System.out.println("生成p10时错误。dn= " + dn);
            e.printStackTrace();
            p10 = "";
        }
        return p10;
    }
}
