package com.xdja.ca.service.impl;

import com.xdja.ca.cache.RedisClient;
import com.xdja.ca.constant.CaSdkRedisKey;
import com.xdja.ca.constant.SdkCommonVariable;
import com.xdja.ca.service.CaSdkRedisCacheManagerService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.MessageFormat;

public class CaSdkRedisCacheManagerServiceImpl implements CaSdkRedisCacheManagerService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private RedisClient redisClient;
    private int transIdExpireTime;
    /**
     * 是否启用Redis
     */
    private boolean enableRedis;

    /**
     * 不需要Redis支持
     */
    public CaSdkRedisCacheManagerServiceImpl() {
        this.enableRedis = false;
    }

    /**
     * 需要Redis支持
     * @param redisClient RedisClient
     * @param transIdExpireTime transId过期时间
     */
    public CaSdkRedisCacheManagerServiceImpl(RedisClient redisClient, int transIdExpireTime) {
        this.redisClient = redisClient;
        this.transIdExpireTime = transIdExpireTime;
        this.enableRedis = true;
    }

    @Override
    public Boolean cacheCaSdkCmpInfo(String transId, String baseCmpInfo) {
        if(this.enableRedis) {
            String key = MessageFormat.format(CaSdkRedisKey.CACHE_CMPINFO_CASDK, transId);
            boolean success = redisClient.lock(key, baseCmpInfo, transIdExpireTime);
            if(logger.isDebugEnabled()) {
                logger.debug("cacheCaSdkCmpInfo key:[{}], baseCmpInfo:[{}], transIdExpireTime:[{}] success:[{}]", key, baseCmpInfo, transIdExpireTime, success);
            }
            if (success) {
                return success;
            } else {
                throw new RuntimeException("锁定失败");
            }
        } else{
            if(logger.isDebugEnabled()) {
                logger.debug("cacheCaSdkCmpInfo transId:[{}], baseCmpInfo:[{}], transIdExpireTime:[{}] ", transId, baseCmpInfo, transIdExpireTime);
            }
            SdkCommonVariable.setCmpInfo(transId, baseCmpInfo);
            return true;
        }
    }

    @Override
    public String removeCaSdkCmpInfo(String transId) {
        if(this.enableRedis) {
            String key = MessageFormat.format(CaSdkRedisKey.CACHE_CMPINFO_CASDK, transId);
            String value = redisClient.get(key);
            redisClient.releaseLock(key);
            if(logger.isDebugEnabled()) {
                logger.debug("removeCaSdkCmpInfo key:[{}], baseCmpInfo:[{}]", key, value);
            }
            return value;
        } else {
            if(logger.isDebugEnabled()) {
                logger.debug("removeCaSdkCmpInfo transId:[{}]", transId);
            }
            return SdkCommonVariable.removeCmpInfo(transId);
        }
    }

    @Override
    public String getCaSdkCmpInfo(String transId) {
        if(this.enableRedis) {
            String key = MessageFormat.format(CaSdkRedisKey.CACHE_CMPINFO_CASDK, transId);
            String value = redisClient.get(key);
            if(logger.isDebugEnabled()) {
                logger.debug("getCaSdkCmpInfo key:[{}], baseCmpInfo:[{}]", key, value);
            }
            return value;
        } else {
            if(logger.isDebugEnabled()) {
                logger.debug("getCaSdkCmpInfo transId:[{}]", transId);
            }
            return SdkCommonVariable.getCmpInfo(transId);
        }
    }

    @Override
    public Long getIncCaCmpReqId() {
        Long returnLong = this.wrapIncNumber(CaSdkRedisKey.INCR_REQID_CASDK);
        logger.debug("getIncCaCmpReqId key:[{}], returnLong:[{}]" ,CaSdkRedisKey.INCR_REQID_CASDK, returnLong);
        return returnLong;
    }

    /**
     * 装配Inc数字
     * @param redisIncKey redis自增Key
     * @return long currentTimeMillis+'9'+6位递增数字
     */
    private long wrapIncNumber(String redisIncKey) {
        Long incrNumber = redisClient.incr(redisIncKey);
        incrNumber = incrNumber % 1000000 + 9000000;
        return System.currentTimeMillis() * 1000000 + incrNumber;
    }
}
