package com.xdja.ca.utils;

import com.xdja.ca.asn1.NISTObjectIdentifiers;
import com.xdja.ca.asn1.RsaObjectIdentifiers;
import com.xdja.ca.asn1.SM2ObjectIdentifiers;
import com.xdja.ca.constant.SdkConstants;
import com.xdja.pki.gmssl.core.utils.GMSSLBCSignUtils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.GMSSLECSignUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLRSASignUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2SignUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLCertUtils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;

import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.security.rsa.RSAPrivateKeyImpl;
import sun.security.util.DerValue;

import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.security.spec.PKCS8EncodedKeySpec;
import java.text.MessageFormat;

/**
 * @author wly
 */
public class SdkHsmUtils {

    protected static transient final Logger logger = LoggerFactory.getLogger(SdkHsmUtils.class.getClass());

    /**
     * 调密码机进行签名
     *
     * @param signAlgName
     * @param privateKeyIndex
     * @param privateKeyPassword
     * @param base64Data
     * @return
     * @throws Exception
     */
    public static String signByYunHsm(String signAlgName, int privateKeyIndex, String privateKeyPassword, String base64Data) throws Exception {
        String signedValue = null;
        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(privateKeyIndex, privateKeyPassword);
        if (SdkConstants.SIGN_ALG_NAME_SM3_WHIT_SM2.equalsIgnoreCase(signAlgName)) {
            signedValue = GMSSLSM2SignUtils.signByYunhsm(privateKeyIndex, privateKeyPassword, base64Data);
        } else if (SdkConstants.SIGN_ALG_NAME_SHA1_WHIT_RSA.equalsIgnoreCase(signAlgName) || SdkConstants.SIGN_ALG_NAME_SHA1_WHIT_RSA_2.equalsIgnoreCase(signAlgName)) {
            signedValue = GMSSLRSASignUtils.signByYunHsm(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName(), privateKeyIndex, privateKeyPassword, base64Data);
        } else if (SdkConstants.SIGN_ALG_NAME_SHA256_WHIT_RSA.equalsIgnoreCase(signAlgName)) {
            signedValue = GMSSLRSASignUtils.signByYunHsm(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName(), privateKeyIndex, privateKeyPassword, base64Data);
        } else if (SdkConstants.SIGN_ALG_NAME_SHA512_WHIT_RSA.equalsIgnoreCase(signAlgName)) {
            signedValue = GMSSLRSASignUtils.signByYunHsm(GMSSLSignatureAlgorithm.SHA512_WITH_RSA.getSigAlgName(), privateKeyIndex, privateKeyPassword, base64Data);
        } else if (SdkConstants.SIGN_ALG_NAME_SHA256_WHIT_ECDSA.equalsIgnoreCase(signAlgName)) {
            signedValue = Base64.toBase64String(GMSSLECSignUtils.signByYunHsm(privateKeyIndex, privateKeyPassword, Base64.decode(base64Data), GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName()));
        } else if(SdkConstants.SIGN_ALG_NAME_SHA384_WHIT_ECDSA.equalsIgnoreCase(signAlgName)){
            signedValue = Base64.toBase64String(GMSSLECSignUtils.sign(sdfPrivateKey, Base64.decode(base64Data), null, GMSSLSignatureAlgorithm.SHA384_WITH_ECDSA.getSigAlgName(), SdkConstants.NIST_384));
        } else if (SdkConstants.SIGN_ALG_NAME_SHA512_WHIT_ECDSA.equalsIgnoreCase(signAlgName)) {
            signedValue = Base64.toBase64String(GMSSLECSignUtils.sign(sdfPrivateKey, Base64.decode(base64Data), null, GMSSLSignatureAlgorithm.SHA512_WITH_ECDSA.getSigAlgName(), SdkConstants.NIST_521));
        } else {
            logger.error("{} 算法不支持", signAlgName);
            throw new Exception(MessageFormat.format("不支持的签名算法 {0}", signAlgName));
        }
        return signedValue;
    }

    /**
     * 通过BC软算法实现签名
     *
     * @param signAlgName
     * @param privateKey
     * @param base64Data
     * @return
     * @throws Exception
     */
    public static String signByBC(String signAlgName, byte[] privateKey, String base64Data) throws Exception {
        String signedValue = null;
        logger.info("========= SDK中使用BC进行软签名 ============");
        if (SdkConstants.SIGN_ALG_NAME_SM3_WHIT_SM2.equalsIgnoreCase(signAlgName)) {
            PrivateKeyInfo info = PrivateKeyInfo.getInstance(privateKey);
            KeyFactory keyFactory = KeyFactory.getInstance(info.getPrivateKeyAlgorithm().getAlgorithm().getId(), new BouncyCastleProvider());
            PrivateKey priKey = keyFactory.generatePrivate(new PKCS8EncodedKeySpec(info.getEncoded()));
            signedValue = GMSSLSM2SignUtils.signByBC(priKey,base64Data);
        } else if (SdkConstants.SIGN_ALG_NAME_SHA1_WHIT_RSA.equalsIgnoreCase(signAlgName) || SdkConstants.SIGN_ALG_NAME_SHA1_WHIT_RSA_2.equalsIgnoreCase(signAlgName)) {
            PrivateKey priKey = RSAPrivateKeyImpl.parseKey(new DerValue(privateKey));
            signedValue = GMSSLRSASignUtils.signByBC(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName(), priKey, base64Data);
        } else if (SdkConstants.SIGN_ALG_NAME_SHA256_WHIT_RSA.equalsIgnoreCase(signAlgName)) {
            PrivateKey priKey = RSAPrivateKeyImpl.parseKey(new DerValue(privateKey));
            signedValue = GMSSLRSASignUtils.signByBC(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName(), priKey, base64Data);
        } else if (SdkConstants.SIGN_ALG_NAME_SHA512_WHIT_RSA.equalsIgnoreCase(signAlgName)) {
            PrivateKey priKey = RSAPrivateKeyImpl.parseKey(new DerValue(privateKey));
            signedValue = GMSSLRSASignUtils.signByBC(GMSSLSignatureAlgorithm.SHA512_WITH_RSA.getSigAlgName(), priKey, base64Data);
        } else {
            PrivateKeyInfo info = PrivateKeyInfo.getInstance(privateKey);
            KeyFactory keyFactory = KeyFactory.getInstance(info.getPrivateKeyAlgorithm().getAlgorithm().getId(), new BouncyCastleProvider());
            PrivateKey priKey = keyFactory.generatePrivate(new PKCS8EncodedKeySpec(info.getEncoded()));
            if (SdkConstants.SIGN_ALG_NAME_SHA256_WHIT_ECDSA.equalsIgnoreCase(signAlgName)) {
                signedValue = Base64.toBase64String(GMSSLBCSignUtils.generateSignature(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName(), priKey, Base64.decode(base64Data)));
            } else if (SdkConstants.SIGN_ALG_NAME_SHA384_WHIT_ECDSA.equalsIgnoreCase(signAlgName)) {
                signedValue = Base64.toBase64String(GMSSLBCSignUtils.generateSignature(GMSSLSignatureAlgorithm.SHA384_WITH_ECDSA.getSigAlgName(), priKey, Base64.decode(base64Data)));
            } else if (SdkConstants.SIGN_ALG_NAME_SHA512_WHIT_ECDSA.equalsIgnoreCase(signAlgName)){
                signedValue = Base64.toBase64String(GMSSLBCSignUtils.generateSignature(GMSSLSignatureAlgorithm.SHA512_WITH_ECDSA.getSigAlgName(), priKey, Base64.decode(base64Data)));
            } else {
                logger.error("{} 算法不支持", signAlgName);
                throw new Exception(MessageFormat.format("不支持的签名算法 {0}", signAlgName));
            }
        }
        return signedValue;
    }
    /**
     * 验证证书通过密码机
     * @param cert      待验证证书
     * @param publicKey 上级CA证书公钥
     * @return boolean
     * @throws Exception 异常
     */
    public static boolean verifyCertByYunHsm(X509Certificate cert, PublicKey publicKey) throws Exception {
        String sigAlgOID = cert.getSigAlgOID();
        boolean verifyResult = false;
        if (SM2ObjectIdentifiers.sm2SignWithSm3.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLCertUtils.verifyCert(publicKey, cert);
        } else if (RsaObjectIdentifiers.sha1WithRSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLRSASignUtils.verifyByYunHsm(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (RsaObjectIdentifiers.sha256WithRSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLRSASignUtils.verifyByYunHsm(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (RsaObjectIdentifiers.sha512WithRSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLRSASignUtils.verifyByYunHsm(GMSSLSignatureAlgorithm.SHA512_WITH_RSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (NISTObjectIdentifiers.sha256WithECDSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLECSignUtils.verifyByYunHsm(publicKey, cert.getTBSCertificate(), cert.getSignature(), GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName());
        } else if (NISTObjectIdentifiers.sha384WithECDSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLECSignUtils.verify(publicKey, cert.getTBSCertificate(), cert.getSignature(),null, GMSSLSignatureAlgorithm.SHA384_WITH_ECDSA.getSigAlgName(), SdkConstants.NIST_384);
        } else if (NISTObjectIdentifiers.sha512WithECDSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLECSignUtils.verify(publicKey, cert.getTBSCertificate(), cert.getSignature(), null, GMSSLSignatureAlgorithm.SHA512_WITH_ECDSA.getSigAlgName(), SdkConstants.NIST_521);
        } else {
            logger.error("{} 算法OID不支持", sigAlgOID);
            throw new Exception(MessageFormat.format("不支持的OID算法 {0}", sigAlgOID));
        }
        return verifyResult;
    }
    /**
     * 验证证书通过BC
     * @param cert      待验证证书
     * @param publicKey 上级CA证书公钥
     * @return boolean
     * @throws Exception 异常
     */
    public static boolean verifyCertByBC(X509Certificate cert, PublicKey publicKey) throws Exception {
        String sigAlgOID = cert.getSigAlgOID();
        boolean verifyResult = false;
        if (SM2ObjectIdentifiers.sm2SignWithSm3.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLSM2SignUtils.verifyByBC(publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (RsaObjectIdentifiers.sha1WithRSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLRSASignUtils.verifyByBC(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (RsaObjectIdentifiers.sha256WithRSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLRSASignUtils.verifyByBC(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (RsaObjectIdentifiers.sha512WithRSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult = GMSSLRSASignUtils.verifyByBC(GMSSLSignatureAlgorithm.SHA512_WITH_RSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (NISTObjectIdentifiers.sha256WithECDSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult =  GMSSLBCSignUtils.verifySignature(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (NISTObjectIdentifiers.sha384WithECDSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult =  GMSSLBCSignUtils.verifySignature(GMSSLSignatureAlgorithm.SHA384_WITH_ECDSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else if (NISTObjectIdentifiers.sha512WithECDSA.getId().equalsIgnoreCase(sigAlgOID)) {
            verifyResult =  GMSSLBCSignUtils.verifySignature(GMSSLSignatureAlgorithm.SHA512_WITH_ECDSA.getSigAlgName(), publicKey, cert.getTBSCertificate(), cert.getSignature());
        } else {
            logger.error("{} 算法OID不支持", sigAlgOID);
            throw new Exception(MessageFormat.format("不支持的OID算法 {0}", sigAlgOID));
        }
        return verifyResult;
    }

    /**
     * BC证书链验签
     *
     * @param protectionAlgId 保护密钥算法oid
     * @param publicKey
     * @param sourceData
     * @param signData
     * @return
     * @throws Exception
     */
    public static boolean verifyCertByBC(String protectionAlgId, PublicKey publicKey, byte[] sourceData, byte[] signData) throws Exception {
        boolean verifyResult = false;
        if (SM2ObjectIdentifiers.sm2SignWithSm3.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLSM2SignUtils.verifyByBC(publicKey, sourceData, signData);
        } else if (RsaObjectIdentifiers.sha1WithRSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLRSASignUtils.verifyByBC(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName(), publicKey, sourceData, signData);
        } else if (RsaObjectIdentifiers.sha256WithRSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLRSASignUtils.verifyByBC(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName(), publicKey, sourceData, signData);
        } else if (RsaObjectIdentifiers.sha512WithRSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLRSASignUtils.verifyByBC(GMSSLSignatureAlgorithm.SHA512_WITH_RSA.getSigAlgName(), publicKey, sourceData, signData);
        } else if (NISTObjectIdentifiers.sha256WithECDSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult =  GMSSLBCSignUtils.verifySignature(GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName(), publicKey, sourceData, signData);
        } else if (NISTObjectIdentifiers.sha384WithECDSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult =  GMSSLBCSignUtils.verifySignature(GMSSLSignatureAlgorithm.SHA384_WITH_ECDSA.getSigAlgName(), publicKey, sourceData, signData);
        } else if (NISTObjectIdentifiers.sha512WithECDSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult =  GMSSLBCSignUtils.verifySignature(GMSSLSignatureAlgorithm.SHA512_WITH_ECDSA.getSigAlgName(), publicKey, sourceData, signData);
        } else {
            logger.error("{} 算法OID不支持", protectionAlgId);
            throw new Exception(MessageFormat.format("不支持的OID算法 {0}", protectionAlgId));
        }
        return verifyResult;
    }

    /**
     * 密码机证书链验签
     *
     * @param protectionAlgName
     * @param cert
     * @param caCert
     * @return
     * @throws Exception
     */
    public static boolean verifyCertByYunHsm(String protectionAlgName, X509Certificate cert, X509Certificate caCert) throws Exception {
        boolean verifyResult = false;
        if ("SM3withSm2".equalsIgnoreCase(protectionAlgName)) {
            verifyResult =  GMSSLCertUtils.verifyCert(caCert.getPublicKey(),cert);
        } else if ("SHA-1WithRSA".equalsIgnoreCase(protectionAlgName) || "SHA1WithRSA".equalsIgnoreCase(protectionAlgName)) {
            verifyResult = GMSSLRSASignUtils.verifyByYunHsm(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName(), caCert.getPublicKey(), cert.getTBSCertificate(), cert.getSignature());
        } else if ("SHA256WithRSA".equalsIgnoreCase(protectionAlgName)) {
            verifyResult = GMSSLRSASignUtils.verifyByYunHsm(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName(), caCert.getPublicKey(), cert.getTBSCertificate(), cert.getSignature());
        } else if("SHA256withECDSA".equalsIgnoreCase(protectionAlgName)){
            verifyResult = GMSSLECSignUtils.verifyByYunHsm(caCert.getPublicKey(), cert.getTBSCertificate(), cert.getTBSCertificate(), GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName());
        }

        return verifyResult;
    }

    /**
     * 密码机证书链验签
     *
     * @param protectionAlgId 保护密钥算法oid
     * @param publicKey
     * @param sourceData
     * @param signData
     * @return
     * @throws Exception
     */
    public static boolean verifyCertByYunHsm(String protectionAlgId, PublicKey publicKey, byte[] sourceData, byte[] signData) throws Exception {
        boolean verifyResult = false;
        if (SM2ObjectIdentifiers.sm2SignWithSm3.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLSM2SignUtils.verifyBySdf(SdfCryptoType.YUNHSM, publicKey, sourceData, signData);
        } else if (RsaObjectIdentifiers.sha1WithRSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLRSASignUtils.verifyByYunHsm(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName(), publicKey, sourceData, signData);
        } else if (RsaObjectIdentifiers.sha256WithRSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLRSASignUtils.verifyByYunHsm(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName(), publicKey, sourceData, signData);
        } else if (RsaObjectIdentifiers.sha512WithRSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLRSASignUtils.verifyByYunHsm(GMSSLSignatureAlgorithm.SHA512_WITH_RSA.getSigAlgName(), publicKey, sourceData, signData);
        } else if (NISTObjectIdentifiers.sha256WithECDSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLECSignUtils.verifyByYunHsm(publicKey, sourceData, signData, GMSSLSignatureAlgorithm.SHA256_WITH_ECDSA.getSigAlgName());
        } else if (NISTObjectIdentifiers.sha384WithECDSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLECSignUtils.verify(publicKey, sourceData, signData,null, GMSSLSignatureAlgorithm.SHA384_WITH_ECDSA.getSigAlgName(), SdkConstants.NIST_384);
        } else if (NISTObjectIdentifiers.sha512WithECDSA.getId().equalsIgnoreCase(protectionAlgId)) {
            verifyResult = GMSSLECSignUtils.verify(publicKey, sourceData, signData, null, GMSSLSignatureAlgorithm.SHA512_WITH_ECDSA.getSigAlgName(), SdkConstants.NIST_521);
        } else {
            logger.error("{} 算法OID不支持", protectionAlgId);
            throw new Exception(MessageFormat.format("不支持的OID算法 {0}", protectionAlgId));
        }
        return verifyResult;
    }
}
