package com.xdja.ca.asn1;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1Set;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.BERSequence;
import org.bouncycastle.asn1.cms.EncryptedContentInfo;

/**
 * <a href="http://tools.ietf.org/html/rfc5652#section-6.1">RFC 5652</a> EnvelopedData object.
 * <pre>
 * EnvelopedData ::= SEQUENCE {
 *     version CMSVersion,
 *     originatorInfo [0] IMPLICIT OriginatorInfo OPTIONAL,
 *     recipientInfos RecipientInfos,
 *     encryptedContentInfo EncryptedContentInfo,
 *     unprotectedAttrs [1] IMPLICIT UnprotectedAttributes OPTIONAL
 * }
 * </pre>
 */
public class SM2EnvelopedData
        extends ASN1Object
{
    private ASN1Integer              version;
    private ASN1Set                 recipientInfos;
    private EncryptedContentInfo encryptedContentInfo;

    public SM2EnvelopedData(
            ASN1Integer            version,
            ASN1Set                 recipientInfos,
            EncryptedContentInfo    encryptedContentInfo)
    {
        this.version = version;
        this.recipientInfos = recipientInfos;
        this.encryptedContentInfo = encryptedContentInfo;
    }

    /**
     * @deprecated use getInstance()
     */
    public SM2EnvelopedData(
            ASN1Sequence seq)
    {
        int     index = 0;

        version = (ASN1Integer)seq.getObjectAt(index++);

        Object tmp = seq.getObjectAt(index++);

        recipientInfos = ASN1Set.getInstance(tmp);

        encryptedContentInfo = EncryptedContentInfo.getInstance(seq.getObjectAt(index++));
    }

    /**
     * Return an EnvelopedData object from a tagged object.
     *
     * @param obj the tagged object holding the object we want.
     * @param explicit true if the object is meant to be explicitly
     *              tagged false otherwise.
     * @exception IllegalArgumentException if the object held by the
     *          tagged object cannot be converted.
     */
    public static SM2EnvelopedData getInstance(
            ASN1TaggedObject obj,
            boolean explicit)
    {
        return getInstance(ASN1Sequence.getInstance(obj, explicit));
    }

    /**
     * Return an EnvelopedData object from the given object.
     * <p>
     * Accepted inputs:
     * <ul>
     * <li> null &rarr; null
     * <li> {@link SM2EnvelopedData} object
     * <li> {@link ASN1Sequence#getInstance(java.lang.Object) ASN1Sequence} input formats with EnvelopedData structure inside
     * </ul>
     *
     * @param obj the object we want converted.
     * @exception IllegalArgumentException if the object cannot be converted.
     */
    public static SM2EnvelopedData getInstance(
            Object obj)
    {
        if (obj instanceof SM2EnvelopedData)
        {
            return (SM2EnvelopedData)obj;
        }

        if (obj != null)
        {
            return new SM2EnvelopedData(ASN1Sequence.getInstance(obj));
        }

        return null;
    }

    public ASN1Integer getVersion()
    {
        return version;
    }

    public ASN1Set getRecipientInfos()
    {
        return recipientInfos;
    }

    public EncryptedContentInfo getEncryptedContentInfo()
    {
        return encryptedContentInfo;
    }


    /**
     * Produce an object suitable for an ASN1OutputStream.
     */
    @Override
    public ASN1Primitive toASN1Primitive()
    {
        ASN1EncodableVector  v = new ASN1EncodableVector();

        v.add(version);


        v.add(recipientInfos);
        v.add(encryptedContentInfo);


        return new BERSequence(v);
    }
}
