package com.xdja.ca.utils;

import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.Security;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.pkcs.CertificationRequest;
import org.bouncycastle.asn1.pkcs.CertificationRequestInfo;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.jce.PKCS10CertificationRequest;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xdja.pki.ra.core.util.cert.DnUtil;
import com.xdja.pki.ra.core.util.cert.RFC4519StyleUpperCase;

/**
 * P10解析结构体
 *
 * @author syg
 */
public class SdkP10Utils {

    private static Logger logger = LoggerFactory.getLogger(Logger.class);

    static {
        Security.addProvider(new BouncyCastleProvider());
    }
    /**
     * 解析标准Pem格式的P10内容(去头尾)
     * @param pemP10
     * @return
     */
    private static String unFormatP10(String pemP10){
        String p10Value = null;
        String P10_HEAD = "-----BEGIN CERTIFICATE REQUEST-----";
        String P10_TAIL = "-----END CERTIFICATE REQUEST-----";
        pemP10 = pemP10.replace(P10_HEAD, "");
        pemP10 = pemP10.replace(P10_TAIL, "");
        pemP10 = pemP10.replace("\r", "");
        pemP10 = pemP10.replace("\n", "");
        p10Value = pemP10;
        return p10Value;
    }

    /**
     * 判断P10文件的正确性
     * @param p10
     * @return
     */
    public static boolean verifyP10Info(String p10) {
        String p10Vaule = SdkP10Utils.unFormatP10(p10);
        byte[] keytemp = null;
        try {
            keytemp = Base64.decode(p10Vaule);
            PKCS10CertificationRequest re = new PKCS10CertificationRequest(keytemp);
            return true;
        } catch (Exception e) {
            logger.error("判断P10文件正确性异常", e);
            return false;
        }
    }

    /**
     * 从P10中获取DN CA系统内生成P10正序显示
     * @param p10
     * @return
     */
    public static String p10ToCertDn(String p10) {
        String p10Vaule = SdkP10Utils.unFormatP10(p10);
        byte[] keytemp = null;
        try {
            keytemp = Base64.decode(p10Vaule);
            PKCS10CertificationRequest re = new PKCS10CertificationRequest(keytemp);
         //   return re.getCertificationRequestInfo().getSubject().toString();
            return getDnByPKCS10(re);
        } catch (Exception e) {
            logger.error("P10文件正确性异常", e);
            return null;
        }
    }

    /**
     * 2020.6.3 爱驰项目  by ssh
     * @param pkcs10
     * @return
     * @throws IOException
     */
    public static String getDnByPKCS10(PKCS10CertificationRequest pkcs10) throws IOException {
        CertificationRequest certificationRequest = CertificationRequest.getInstance(pkcs10.getEncoded());
        byte[] encoded = certificationRequest.getCertificationRequestInfo().getEncoded();
        ASN1Sequence seq = ASN1Sequence.getInstance(encoded);
        int seqStart = 0;
        if (!(seq.getObjectAt(0) instanceof ASN1TaggedObject)) {
            // field 0 is missing!
            seqStart = -1;
        }
        ASN1Encodable objectAt = seq.getObjectAt(seqStart + 2);
        X500Name x500Name = X500Name.getInstance(RFC4519StyleUpperCase.INSTANCE, objectAt.toASN1Primitive());
        return reverseDnSeq(x500Name);
    }

    /**
     * 反转下DN顺序
     *
     * @param x500Name
     * @return
     */
    public static String reverseDnSeq(X500Name x500Name) {
        RDN[] rdns = x500Name.getRDNs();
        if (0 == rdns.length) {
            return "";
        }
        //当DN项第一个为C，或者最后一项为CN的时进行DN顺序的反转
        if (rdns[0].getFirst().getType().equals(RFC4519StyleUpperCase.cn) || rdns[rdns.length - 1].getFirst().getType().equals(RFC4519StyleUpperCase.c)) {
            x500Name = new X500Name(RFC4519StyleUpperCase.INSTANCE, DnUtil.reverseRdns(rdns));
        }
        return x500Name.toString();
    }

    /**
     * 从P10中获取公钥
     *
     * @param p10
     * @return
     */
    public static PublicKey p10ToPublicKey(String p10)  {
        String p10Value = SdkP10Utils.unFormatP10(p10);
        byte[] bsCert = Base64.decode(p10Value);
        PKCS10CertificationRequest re = new PKCS10CertificationRequest(bsCert);
        re.getCertificationRequestInfo();
        PublicKey pk = null;
        try {
            pk = re.getPublicKey("BC");
        } catch (NoSuchAlgorithmException e) {
            logger.error("算法异常", e);
        } catch (NoSuchProviderException e) {
            logger.error("Provider异常", e);
        } catch (InvalidKeyException e) {
            logger.error("公钥信息异常", e);
        }
        return pk;

    }


    /**
     * 从P10中签名原文
     *
     * @param p10
     * @return
     */
    public static byte[] p10ToReqInfo(String p10){
        String p10Value = SdkP10Utils.unFormatP10(p10);
        byte[] bsCert = Base64.decode(p10Value);
        PKCS10CertificationRequest re = new PKCS10CertificationRequest(bsCert);
        CertificationRequestInfo certificationRequestInfo = re.getCertificationRequestInfo();
        byte[] reqInfo = new byte[0];
        try {
            reqInfo = certificationRequestInfo.getEncoded();
        } catch (IOException e) {
            logger.error("从P10获取签名异常", e);
        }
        return reqInfo;

    }
    /**
     * 从P10中获取签名值
     * @param p10
     * @return
     */
    public static String p10ToSignValue(String p10) {
        String p10Vaule = SdkP10Utils.unFormatP10(p10);
        byte[] keytemp = null;
        try {
            keytemp = Base64.decode(p10Vaule);
            PKCS10CertificationRequest re = new PKCS10CertificationRequest(keytemp);
            return re.getSignature().toString();
        } catch (Exception e) {
            logger.error("获取签名值异常", e);
            return null;
        }
    }

    /**
     * 从P10中获取签名算法
     * @param p10
     * @return
     */
    public static ASN1ObjectIdentifier p10ToSignAlgo(String p10) {
        String p10Vaule = SdkP10Utils.unFormatP10(p10);
        byte[] keytemp = null;
        try {
            keytemp = Base64.decode(p10Vaule);
            PKCS10CertificationRequest re = new PKCS10CertificationRequest(keytemp);
            return re.getSignatureAlgorithm().getAlgorithm();
        } catch (Exception e) {
            logger.error("获取签名算法异常", e);
            return null;
        }
    }
}
