package com.xdja.pki.gmssl.hsm.init;

import com.xdja.pki.gmssl.core.utils.GMSSLFileUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLHSMConstants;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.http.GMSSLHttpsClient;
import com.xdja.pki.gmssl.http.bean.GMSSLHttpRequest;
import com.xdja.pki.gmssl.http.bean.GMSSLHttpResponse;
import com.xdja.pki.gmssl.http.bean.GMSSLHttpsClientConfig;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreSpi;

/**
 * GMSSSL HSM 初始化类
 * 0、需要使用GMSSL HSM的系统，手动放置标识到指定位置
 * $CATALINA_HOME/conf/gmssl-hsm/ 空文件夹即可标识
 * 1、未初始化的系统，进行系统初始化，增加相关配置
 * 检查hsm-server.conf是否存在，存在的话，
 * 2、已初始化的系统，读取相关配置，同步密钥信息
 */
public class GMSSLHSMInit {

    private static Logger logger = LoggerFactory.getLogger(GMSSLHSMInit.class);

    /**
     * 是否支持 GMSSL HSM
     *
     * @return
     */
    public static boolean isHSMOpen() {
        boolean exists = GMSSLHSMConstants.CONFIG_PATH_FILE.exists();
        if (logger.isDebugEnabled()){
            logger.debug("hsm open {}", exists);
        }
        return exists;
    }

    /**
     * 初始化配置文件
     *
     * @param ip              GMSSL-HSM-SERVER IP地址
     * @param port            GMSSL-HSM-SERVER PORT端口
     * @param signPfxPassword 双向认证签名PFX密码
     * @param encPfxPassword  双向认证加密PFX密码
     * @param signPfxStream   双向认证签名证书及私钥
     * @param encPfxStream    双向认证加密证书及私钥
     * @param trustP7bStream  信任P7B
     * @throws Exception
     */
    public static boolean initConfigFile(String ip, String port,
                                         String signPfxPassword, String encPfxPassword,
                                         FileInputStream signPfxStream, FileInputStream encPfxStream,
                                         FileInputStream trustP7bStream) {
        //如果目录存在，则进行配置文件初始化
        if (isHSMOpen()) {
            try {
                //写入配置文件
                GMSSLHSMConfig gmsslhsmConfig = saveConfig(ip, port, signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream);
                if (logger.isDebugEnabled()){
                    logger.debug("save gmssl hsm config {}", gmsslhsmConfig);
                }
                //获取keystore
                getKeystore(gmsslhsmConfig);
                return true;
            } catch (Exception e) {
                logger.error("init hsm connection error!", e);
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * 同步密钥信息
     */
    public static void updateKeyFile() throws Exception {
        if (isHSMOpen()) {
            //获取配置文件
            GMSSLHSMConfig gmsslhsmConfig = GMSSLHSMConfig.parseConfig(GMSSLHSMConstants.CONFIG_FILE_PATH);
            //获取keystore
            getKeystore(gmsslhsmConfig);
        }
    }

    /**
     * 测试连接
     *
     * @param ip              GMSSL-HSM-SERVER IP地址
     * @param port            GMSSL-HSM-SERVER PORT端口
     * @param signPfxPassword 双向认证签名PFX密码
     * @param encPfxPassword  双向认证加密PFX密码
     * @param signPfxStream   双向认证签名证书及私钥
     * @param encPfxStream    双向认证加密证书及私钥
     * @param trustP7bStream  信任P7B
     * @return 连接状态
     */
    public static boolean testHSMConnect(String ip, String port,
                                         String signPfxPassword, String encPfxPassword,
                                         FileInputStream signPfxStream, FileInputStream encPfxStream,
                                         FileInputStream trustP7bStream) {
        if (isHSMOpen()) {
            //获取配置文件
            GMSSLHSMConfig gmsslhsmConfig = null;
            try {
                gmsslhsmConfig = generateConfig(ip, port, signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream);
                KeyStore keyStore = getKeystoreFromServer(gmsslhsmConfig);
                boolean isConnect = keyStore != null;
                if (logger.isDebugEnabled()){
                    logger.debug("test connect config={}, isConnect={}", gmsslhsmConfig, isConnect);
                }
                return isConnect;
            } catch (Exception e) {
                logger.error("test hsm connection error!", e);
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * 测试连接
     * @return 连接状态
     * @throws Exception
     */
    public static boolean testHSMConnect() throws Exception {
        if (isHSMOpen()) {
            //获取配置文件
            GMSSLHSMConfig gmsslhsmConfig = GMSSLHSMConfig.parseConfig(GMSSLHSMConstants.CONFIG_FILE_PATH);
            KeyStore keyStore = getKeystoreFromServer(gmsslhsmConfig);
            boolean isConnect = keyStore != null;
            if (logger.isDebugEnabled()){
                logger.debug("test connect config={}, isConnect={}", gmsslhsmConfig, isConnect);
            }
            return isConnect;
        } else {
            return false;
        }
    }

    /**
     * 根据参数配置文件
     *
     * @param ip              GMSSL-HSM-SERVER IP地址
     * @param port            GMSSL-HSM-SERVER PORT端口
     * @param signPfxPassword 双向认证签名PFX密码
     * @param encPfxPassword  双向认证加密PFX密码
     * @param signPfxStream   双向认证签名证书及私钥
     * @param encPfxStream    双向认证加密证书及私钥
     * @param trustP7bStream  信任P7B
     * @return 配置文件对象
     * @throws IOException
     */
    private static GMSSLHSMConfig generateConfig(String ip, String port,
                                                 String signPfxPassword, String encPfxPassword,
                                                 FileInputStream signPfxStream, FileInputStream encPfxStream,
                                                 FileInputStream trustP7bStream) throws IOException {
        GMSSLHSMConfig config = new GMSSLHSMConfig();
        config.setIp(ip);
        config.setPort(port);

        //sign
        ByteArrayOutputStream signByteOut = new ByteArrayOutputStream();
        IOUtils.copy(signPfxStream, signByteOut);
        GMSSLFileUtils.writeFile(GMSSLHSMConstants.SIGN_FILE_PATH, signByteOut.toByteArray());
        config.setSignPath(GMSSLHSMConstants.SIGN_FILE_PATH);
        config.setSignType(GMSSLHSMConstants.PKCA12_TYPE);
        config.setSignProvider(GMSSLHSMConstants.PROVIDER);
        config.setSignPassword(signPfxPassword);

        //enc
        ByteArrayOutputStream encByteOut = new ByteArrayOutputStream();
        IOUtils.copy(encPfxStream, encByteOut);
        GMSSLFileUtils.writeFile(GMSSLHSMConstants.ENC_FILE_PATH, encByteOut.toByteArray());
        config.setEncPath(GMSSLHSMConstants.ENC_FILE_PATH);
        config.setEncType(GMSSLHSMConstants.PKCA12_TYPE);
        config.setEncProvider(GMSSLHSMConstants.PROVIDER);
        config.setEncPassword(encPfxPassword);

        //trust
        ByteArrayOutputStream trustByteOut = new ByteArrayOutputStream();
        IOUtils.copy(trustP7bStream, trustByteOut);
        GMSSLFileUtils.writeFile(GMSSLHSMConstants.TRUST_FILE_PATH, trustByteOut.toByteArray());
        config.setTrustPath(GMSSLHSMConstants.TRUST_FILE_PATH);
        config.setTrustType(GMSSLHSMConstants.PKCA7_TYPE);
        config.setTrustProvider(GMSSLHSMConstants.PROVIDER);
        config.setTrustPassword(GMSSLHSMConstants.TRUST_KEYSTORE_PASSWORD_DEFAULT);

        config.setKeyStorePassword(GMSSLHSMConstants.KEYSTORE_PASSWORD);
        return config;
    }

    /**
     * 生成并保存配置文件
     * @param ip              GMSSL-HSM-SERVER IP地址
     * @param port            GMSSL-HSM-SERVER PORT端口
     * @param signPfxPassword 双向认证签名PFX密码
     * @param encPfxPassword  双向认证加密PFX密码
     * @param signPfxStream   双向认证签名证书及私钥
     * @param encPfxStream    双向认证加密证书及私钥
     * @param trustP7bStream  信任P7B
     * @return 配置文件对象
     * @throws IOException
     */
    private static GMSSLHSMConfig saveConfig(String ip, String port,
                                             String signPfxPassword, String encPfxPassword,
                                             FileInputStream signPfxStream, FileInputStream encPfxStream,
                                             FileInputStream trustP7bStream) throws IOException {
        GMSSLHSMConfig config = generateConfig(ip, port, signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream);
        config.saveConfig(GMSSLHSMConstants.CONFIG_FILE_PATH);
        if (logger.isDebugEnabled()){
            logger.debug("save gmssl hsm config {}", config);
        }
        return config;
    }

    /**
     * 尝试从 server 和本地获取 keystore
     * 设置keystore
     * @param config
     * @return
     * @throws Exception
     */
    private static void getKeystore(GMSSLHSMConfig config) throws Exception {
        //获取keystore
        KeyStore keystore = getKeystoreFromServer(config);
        if (logger.isDebugEnabled()){
            logger.debug("get gmssl hsm keystore from server {}", keystore);
        }
        if (keystore == null) {
            keystore = getKeystoreFromFile();
            if (logger.isDebugEnabled()){
                logger.debug("get gmssl hsm keystore from config {}", keystore);
            }
            if (keystore == null) {
                logger.error("Fail to get keyStore from HSM-SERVER and Can`t find local file. \n" +
                        "Now will be exit.\n" +
                        "You can You can manually place files under {} and restart application", GMSSLHSMConstants.HSM_KEYSTORE_PATH);
                System.exit(-1);
                return;
            }
        }
        //设置keystore
        GMSSLPkiCryptoInit.setHsmKeyStore(keystore);
        if (logger.isDebugEnabled()){
            logger.debug("set gmssl hsm keystore success");
        }
    }

    /**
     * 从HSM-SERVER 获取keystore
     *
     * @param config
     * @throws Exception
     */
    private static KeyStore getKeystoreFromServer(GMSSLHSMConfig config) throws Exception {
        GMSSLHttpsClientConfig clientConfig = new GMSSLHttpsClientConfig();
        KeyStore clientKeyStore = config.getKeyStore();
        KeyStore trustKeyStore = config.getTrustKeyStore();
        clientConfig.setTrustStore(trustKeyStore);
        clientConfig.setTrustStorePassword(config.getTrustPassword());
        clientConfig.setTrustStoreType(GMSSLHSMConstants.KEYSTORE_TYPE);
        clientConfig.setSslProtocol(GMSSLHSMConstants.SSL_PROTOCOL);
        clientConfig.setClientKeyStoreType(GMSSLHSMConstants.KEYSTORE_TYPE);
        clientConfig.setSslEnabled(true);
        clientConfig.setClientKeyStore(clientKeyStore);
        clientConfig.setClientKeyStorePassword(GMSSLHSMConstants.KEYSTORE_PASSWORD);
        GMSSLHttpsClient gmsslHttpsClient = new GMSSLHttpsClient(clientConfig);
        GMSSLHttpRequest request = new GMSSLHttpRequest();
        String url = GMSSLHSMConstants.HSM_SERVER_URI
                .replace(GMSSLHSMConstants.HSM_SERVER_IP_HOLDER, config.getIp())
                .replace(GMSSLHSMConstants.HSM_SERVER_PORT_HOLDER, config.getPort());
        request.setUrl(url);
        GMSSLHttpResponse response = gmsslHttpsClient.get(request);
        byte[] body = response.getBody();
        if (logger.isDebugEnabled()){
            logger.debug("get gmssl hsm keystore from server response={}", response);
        }
        if (response.getStatusCode() == 200 && body != null) {
            KeyStore keyStore = GMSSLKeyStoreUtils.readKeyStoreFromBytes(GMSSLHSMConstants.KEYSTORE_PASSWORD.toCharArray(), GMSSLHSMConstants.KEYSTORE_TYPE, body);
            GMSSLKeyStoreUtils.saveGMSSLKeyStoreFullName(keyStore, GMSSLHSMConstants.KEYSTORE_PASSWORD, GMSSLHSMConstants.HSM_KEYSTORE_PATH);
            return keyStore;
        } else {
            logger.error("get hsm server keyStore error, now use local keyStore! statusCode={}, body={}",
                    response.getStatusCode(), body == null ? "null" : new String(body));
            return null;
        }
    }

    private static KeyStore getKeystoreFromFile() throws Exception {
        if (GMSSLHSMConstants.HSM_KEYSTORE_FILE.exists()) {
            return GMSSLKeyStoreUtils.readKeyStoreFromPath(GMSSLHSMConstants.HSM_KEYSTORE_PATH, GMSSLHSMConstants.KEYSTORE_PASSWORD.toCharArray());
        } else {
            return null;
        }
    }
}
