package com.xdja.pki.gmssl.hsm.server;

import com.xdja.pki.gmssl.core.utils.GMSSLCertPathUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLFileUtils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.hsm.server.constant.Constants;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLCertUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLExtensionUtils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;

import java.math.BigInteger;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author: houzhe
 * @date: 2021-11-01
 * @description: 启动类
 **/
@SpringBootApplication
public class HsmServerApplication {

    private static final Logger logger = LoggerFactory.getLogger(HsmServerApplication.class);

    public static void main(String[] args) {
        if (null != args && args.length != 0 && "init".equalsIgnoreCase(args[0])) {
            try {
                init();
            } catch (Exception e) {
                logger.error("初始化失败", e);
            }
            logger.info("初始化成功");
            return;
        }

        SpringApplication.run(HsmServerApplication.class, args);
    }

    private static void init() throws Exception {
        AtomicLong serialNumber = new AtomicLong(System.currentTimeMillis());

        //生成 CA密钥、CA根证书
        Date now = new Date();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.add(Calendar.DATE, -1);
        Date noBefore = calendar.getTime();
        calendar.add(Calendar.YEAR, 30);
        Date noAfter = calendar.getTime();

        String rootDn = "cn=root,o=gmssl-hsm,o=xdja,c=cn";
        KeyPair rootKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> rootExtensions = new ArrayList<>();
        rootExtensions.add(GMSSLExtensionUtils.genRootCertKeyUsageExtension());

        X509Certificate rootCert = GMSSLCertUtils.generateCertByBC(
                rootDn, rootDn,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootKeyPair.getPrivate(),
                rootKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                rootExtensions
        );

        KeyStore trustKeyStore = GMSSLKeyStoreUtils.generateGMSSLTrustStoreWithBKS(rootCert);
        GMSSLKeyStoreUtils.saveGMSSLKeyStore(trustKeyStore, Constants.PWD, Constants.PATH, Constants.ROOT_KEY_STORE_NAME);
        List<X509Certificate> certificates = new ArrayList<>();
        certificates.add(rootCert);
        byte[] bytes = GMSSLCertPathUtils.generateP7b(certificates);
        GMSSLFileUtils.writeFile(Constants.PATH, Constants.ROOT_KEY_STORE_NAME + ".p7b", bytes);

        KeyStore keyStore = KeyStore.getInstance(GMSSLCertPathUtils.KEYSTORE_TYPE_BKS, BouncyCastleProvider.PROVIDER_NAME);
        keyStore.load(null, null);
        keyStore.setKeyEntry("trust", rootKeyPair.getPrivate(), Constants.PWD.toCharArray(), new X509Certificate[]{rootCert});
        GMSSLKeyStoreUtils.saveGMSSLBKSKeyStore(keyStore, Constants.PWD, Constants.PATH, Constants.ROOT_KEY_STORE_NAME);

        KeyStore serverKeyStore = generateKeyStore(noBefore, noAfter, serialNumber, rootCert, rootDn, rootKeyPair.getPrivate());
        GMSSLKeyStoreUtils.saveGMSSLKeyStore(serverKeyStore, Constants.PWD, Constants.PATH, Constants.SERVER_KEY_STORE_NAME);

        KeyStore clientKeyStore = generateKeyStore(noBefore, noAfter, serialNumber, rootCert, rootDn, rootKeyPair.getPrivate());
        GMSSLKeyStoreUtils.saveGMSSLKeyStore(serverKeyStore, Constants.PWD, Constants.PATH, Constants.CLIENT_KEY_STORE_NAME);
        GMSSLKeyStoreUtils.saveGMSSLPfx(clientKeyStore, Constants.PWD, Constants.PATH);
    }

    private static KeyStore generateKeyStore(Date noBefore, Date noAfter, AtomicLong serialNumber,
                                             X509Certificate rootCert, String rootDn, PrivateKey rootPrivateKey) throws Exception {
        //生成 server 密钥、证书
        String signDn = "cn=sign,o=gmssl-hsm,o=xdja,c=cn";
        KeyPair signKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> signExtensions = new ArrayList<>();
        signExtensions.add(GMSSLExtensionUtils.genSignatureCertKeyUsageExtension());
        signExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(signKeyPair.getPublic()));
        signExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rootCert));
        X509Certificate signCert = GMSSLCertUtils.generateCertByBC(
                rootDn, signDn,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootPrivateKey,
                signKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                signExtensions
        );

        String encDn = "cn=enc,o=gmssl-hsm,o=xdja,c=cn";
        KeyPair encKeyPair = GMSSLSM2KeyUtils.generateSM2KeyPairByBC();
        List<Extension> encExtensions = new ArrayList<>();
        encExtensions.add(GMSSLExtensionUtils.genEncryptCertKeyUsageExtension());
        encExtensions.add(GMSSLExtensionUtils.genSubjectKeyIdentifierExtension(encKeyPair.getPublic()));
        encExtensions.add(GMSSLExtensionUtils.genAuthorityKeyIdentifierExtension(rootCert));
        X509Certificate encCert = GMSSLCertUtils.generateCertByBC(
                rootDn, encDn,
                BigInteger.valueOf(serialNumber.getAndIncrement()),
                noBefore, noAfter,
                rootPrivateKey,
                encKeyPair.getPublic(),
                GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName(),
                encExtensions
        );

        return GMSSLKeyStoreUtils.generateGMSSLKeyStoreWithBKS(
                Constants.PWD, rootCert,
                "sign", signKeyPair.getPrivate(), signCert,
                "enc", encKeyPair.getPrivate(), encCert
        );

    }
}
