package com.xdja.pki.gmssl;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jsse.provider.XDJAJsseProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.*;
import java.security.*;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

public class GMSSLContext {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
        if (Security.getProvider(XDJAJsseProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new XDJAJsseProvider());
        }
    }

    private final static String KEY_MANAGER_FACTORY_NAME = "PKIX";
    private final static Logger logger = LoggerFactory.getLogger(GMSSLContext.class.getName());

    public static class GMSSLException extends Exception {
        public GMSSLException(String message) {
            super(message);
        }

        public GMSSLException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    private SSLContext sslContext;

    public static GMSSLContext getClientInstance(String protocol) throws GMSSLException {
        TrustManager trustManager = new X509TrustManager() {
            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return new X509Certificate[0];
            }

            @Override
            public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                if (chain == null || chain.length < 1 || authType == null || authType.length() < 1) {
                    throw new IllegalArgumentException();
                }

                String subject = chain[0].getSubjectX500Principal().getName();
                logger.info("Auto-trusted server certificate chain for: " + subject);
            }

            @Override
            public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            }
        };

        return GMSSLContext.getClientInstance(new TrustManager[]{trustManager}, protocol);
    }

    public static GMSSLContext getClientInstance(char[] password, KeyStore trustStore, String protocol) throws GMSSLException {
        return new GMSSLContext(null, password, trustStore, protocol);
    }

    public static GMSSLContext getClientInstance(KeyStore keyStore, char[] password, KeyStore trustStore, String protocol) throws GMSSLException {
        return new GMSSLContext(keyStore, password, trustStore, protocol);
    }

    public static GMSSLContext getClientInstance(TrustManager[] trustManagers, String protocol) throws GMSSLException {
        return new GMSSLContext(null, trustManagers, protocol);
    }

    public static GMSSLContext getServerInstance(KeyStore serverStore, char[] password, KeyStore trustStore, String protocol) throws GMSSLException {
        return new GMSSLContext(serverStore, password, trustStore, protocol);
    }

    public static GMSSLContext getServerInstance(KeyManager[] keyManagers, TrustManager[] trustManagers, String protocol) throws GMSSLException {
        return new GMSSLContext(keyManagers, trustManagers, protocol);
    }

    private static KeyManager[] getKeyManagers(KeyStore keyStore, char[] password) throws GMSSLException {
        try {
            KeyManager[] keyManagers = null;
            if (keyStore != null && password.length > 0) {
                KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KEY_MANAGER_FACTORY_NAME,
                        XDJAJsseProvider.PROVIDER_NAME);
                keyManagerFactory.init(keyStore, password);
                keyManagers = keyManagerFactory.getKeyManagers();
            }

            return keyManagers;
        } catch (NoSuchProviderException | NoSuchAlgorithmException | UnrecoverableKeyException | KeyStoreException e) {
            throw new GMSSLException("GMSSLContext get key managers exception", e);
        }
    }

    private static TrustManager[] getTrustManagers(KeyStore trustStore) throws GMSSLException {
        try {
            TrustManager[] trustManagers = null;
            if (trustStore != null) {
                TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(KEY_MANAGER_FACTORY_NAME,
                        XDJAJsseProvider.PROVIDER_NAME);
                trustManagerFactory.init(trustStore);
                trustManagers = trustManagerFactory.getTrustManagers();
            }

            return trustManagers;
        } catch (NoSuchProviderException | NoSuchAlgorithmException | KeyStoreException e) {
            throw new GMSSLException("GMSSLContext get trust managers exception", e);
        }
    }

    private GMSSLContext(KeyStore keyStore, char[] password, KeyStore trustStore, String protocol) throws GMSSLException {
        this(getKeyManagers(keyStore, password), getTrustManagers(trustStore), protocol);
    }

    private GMSSLContext(KeyManager[] keyManagers, TrustManager[] trustManagers, String protocol) throws GMSSLException {
        try {
            this.sslContext = SSLContext.getInstance(protocol, XDJAJsseProvider.PROVIDER_NAME);
            this.sslContext.init(keyManagers, trustManagers, null);
        } catch (KeyManagementException | NoSuchProviderException | NoSuchAlgorithmException e) {
            throw new GMSSLException("GMSSLContext new instance exception", e);
        }
    }

    public SSLSocketFactory getSocketFactory() {
        return this.sslContext.getSocketFactory();
    }

    public SSLServerSocketFactory getServerSocketFactory() {
        return this.sslContext.getServerSocketFactory();
    }

    public SSLContext getSslContext() {
        return sslContext;
    }
}
