package com.xdja.pki.gmssl.https;

import com.sun.net.httpserver.*;
import com.xdja.pki.gmssl.GMSSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLParameters;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.KeyStore;
import java.util.concurrent.Executors;

public class GMHttpsServer {
    private static Logger logger = LoggerFactory.getLogger(GMHttpsServer.class.getName());

    public static class Context {
        private String path;
        private GMHttpsMethod method;
        private Handler handler;

        public Context(String path, GMHttpsMethod method, Handler handler) {
            this.path = path;
            this.method = method;
            this.handler = handler;
        }
    }

    public interface Handler {
        GMHttpsResponse onRequest(GMHttpsRequest request);

        void onRequestError(String message);
    }

    public static void startHttpsServer(KeyStore serverStore, char[] password,
                                        KeyStore trustStore, boolean needClientAuth,
                                        Context[] contexts,
                                        String host, int port, String protocol) throws GMHttpsException {
        GMSSLContext gmsslContext = null;
        try {
            // 获取 国密SSL context
            gmsslContext = GMSSLContext.getServerInstance(serverStore, password, trustStore, protocol);
        } catch (GMSSLContext.GMSSLException e) {
            throw new GMHttpsException("get server instance exception", e);
        }
        HttpsServer httpsServer = null;
        try {
            httpsServer = HttpsServer.create(new InetSocketAddress(host, port), 0);
        } catch (IOException e) {
            throw new GMHttpsException("create https server exception", e);
        }

        // 为 https server 设置 国密SSL context
        httpsServer.setHttpsConfigurator(new HttpsConfigurator(gmsslContext.getSslContext()) {
            @Override
            public void configure(HttpsParameters httpsParameters) {
                SSLContext sslContext = getSSLContext();
                SSLParameters sslParameters = sslContext.getDefaultSSLParameters();
                sslParameters.setNeedClientAuth(needClientAuth);
                httpsParameters.setSSLParameters(sslParameters);
            }
        });

        for (Context context : contexts) {
            httpsServer.createContext(context.path, new HttpHandler() {
                @Override
                public void handle(HttpExchange httpExchange) throws IOException {
                    HttpsExchange httpsExchange = (HttpsExchange) httpExchange;
                    httpsExchange.getSSLSession();

                    if (!httpsExchange.getRequestMethod().equals(context.method.toString())) {
                        context.handler.onRequestError(httpsExchange.getRequestURI() + " method want: " + context.method + ", but request is: " + httpsExchange.getRequestMethod());
                        httpsExchange.sendResponseHeaders(405, 0);
                        return;
                    }

                    byte[] requestBody = GMHttpsUtils.readBytes(httpsExchange.getRequestBody());
                    GMHttpsRequest request = new GMHttpsRequest(null, httpsExchange.getRequestURI().toString(), httpsExchange.getRequestMethod(), requestBody);
                    GMHttpsResponse response = context.handler.onRequest(request);
                    httpExchange.getResponseHeaders().set("Content-Type", "text/html");
                    httpExchange.sendResponseHeaders(response.getCode(), response.getBody().length);
                    GMHttpsUtils.writeBytes(httpsExchange.getResponseBody(), response.getBody());
                }
            });
        }

        httpsServer.setExecutor(Executors.newCachedThreadPool());
        logger.info("Starting server on port " + port + "...");
        httpsServer.start();
        logger.info("Server started successfully!");
    }

}
