package com.xdja.pki.gmssl.http.api;

import com.xdja.pki.gmssl.GMSSLContext;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.http.bean.*;
import com.xdja.pki.gmssl.http.exception.GMSSLHttpErrorCode;
import com.xdja.pki.gmssl.http.exception.GMSSLHttpException;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jsse.provider.XDJAJsseProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import java.io.*;
import java.net.*;
import java.security.Security;
import java.util.Map;

public abstract class BaseHttpClient {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
        if (Security.getProvider(XDJAJsseProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new XDJAJsseProvider());
        }
    }

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    protected GMSSLHttpClientConfig httpClientConfig;

    public BaseHttpClient() {
    }

    public BaseHttpClient(GMSSLHttpClientConfig httpClientConfig) {
        this.httpClientConfig = httpClientConfig;
    }

    /**
     * 通过 httpRequest 创建 链接
     *
     * @param request 请求结构体
     * @return HttpsURLConnection 连接对象
     * @throws GMSSLHttpException http 异常结构体
     */
    public GMSSLHttpResponse connect(GMSSLHttpMethod httpMethod, GMSSLHttpRequest request) throws GMSSLHttpException {
        // 构造 http url
        String uri = generateURL(request);

        // 打开链接
        HttpURLConnection connection = null;
        URL conUrl = null;
        try {
            conUrl = new URL(uri);
        } catch (MalformedURLException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.GENERATE_URL_ERROR);
        }
        try {
            connection = (HttpURLConnection) conUrl.openConnection();
        } catch (IOException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.OPENCONNECTION_ERROR);
        }

        // 添加 http headers
        Map<String, String> headers = request.getHeaders();
        if (headers != null && headers.size() != 0) {
            for (String key : headers.keySet()) {
                connection.setRequestProperty(key, headers.get(key));
                logger.info("http request set header {}={}", key, headers.get(key));
            }
        }

        if (httpClientConfig != null) {
            // 设置链接超时时间
            int connectTimeout = httpClientConfig.getConnectTimeout();
            if (connectTimeout != 0) {
                connection.setConnectTimeout(connectTimeout);
                logger.info("http request set connect timeout {}", connectTimeout);
            }

            // 设置 读超时时间
            int readTimeout = httpClientConfig.getReadTimeout();
            if (readTimeout != 0) {
                connection.setReadTimeout(readTimeout);
                logger.info("http request set read timeout {}", readTimeout);
            }

            // 是否是 https 请求
            setSSLSocketFactory(connection);
        }

        // 设置接收输入流 即 接收 responsez
        connection.setDoInput(true);

        // 设置 http method
        try {
            String method = httpMethod.getValue();
            logger.info("http request method is {}", method);
            connection.setRequestMethod(method);
        } catch (ProtocolException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.SET_REQUEST_METHOD_ERROR);
        }

        // 设置打开输出流 即 允许写入 request body
        connection.setDoOutput(true);

        //  开始链接
        try {
            connection.connect();
        } catch (IOException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.CONNECTION_ERROR);
        }

        // 写入 request body
        writeRequestBody(connection, request);

        // 构造 http response
        GMSSLHttpResponse response = new GMSSLHttpResponse();
        response.setHeaders(connection.getHeaderFields());
        int statusCode;
        try {
            statusCode = connection.getResponseCode();
//            logger.info("http response status code {}", statusCode);
            response.setStatusCode(statusCode);
        } catch (IOException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.CONNECTION_GET_RESPONSE_STATUS_CODE_ERROR);
        }
        try {
            String responseMessage = connection.getResponseMessage();
            response.setStatusMessage(responseMessage);
//            logger.info("http response status message {}", responseMessage);
        } catch (IOException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.CONNECTION_GET_RESPONSE_STATUS_CODE_ERROR);
        }

        readResponseBody(connection, statusCode, response);

        return response;
    }

    private String generateURL(GMSSLHttpRequest request) throws GMSSLHttpException {
        StringBuilder sb = new StringBuilder(request.getUrl());
        URL url = null;
        try {
            url = new URL(sb.toString());
        } catch (MalformedURLException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.URL_ERROR);
        }
        // 添加 http url params
        Map<String, String> params = request.getParams();
        if (params != null && params.size() != 0) {
            if (url.getQuery() == null) {
                sb.append("?");
            }
            for (String key : params.keySet()) {
                sb.append(key).append("=").append(params.get(key)).append("&");
            }
            sb.deleteCharAt(sb.length() - 1);
        }
        String uri = sb.toString();
        logger.info("http request url is {}", uri);
        return uri;
    }

    private void setSSLSocketFactory(HttpURLConnection connection) throws GMSSLHttpException {
        if (httpClientConfig instanceof GMSSLHttpsClientConfig) {
            GMSSLHttpsClientConfig httpsClientConfig = (GMSSLHttpsClientConfig) httpClientConfig;
            String sslProtocol = httpsClientConfig.getSslProtocol();
            if (httpsClientConfig.isSslEnabled() && sslProtocol != null) {
                logger.info("http request set ssl protocol {}", sslProtocol);
                try {
                    SSLSocketFactory socketFactory = null;
                    if (httpsClientConfig.getClientKeyStore() != null && httpsClientConfig.getTrustStore() != null) {
                        socketFactory = GMSSLContext.getClientInstance(
                                httpsClientConfig.getClientKeyStore(),
                                httpsClientConfig.getClientKeyStorePassword().toCharArray(),
                                httpsClientConfig.getTrustStore(),
                                sslProtocol
                        ).getSocketFactory();
                    } else if (httpsClientConfig.getTrustStore() != null) {
                        socketFactory = GMSSLContext.getClientInstance(
                                httpsClientConfig.getTrustStorePassword().toCharArray(),
                                httpsClientConfig.getTrustStore(),
                                sslProtocol
                        ).getSocketFactory();
                    } else {
                        socketFactory = GMSSLContext.getClientInstance(sslProtocol).getSocketFactory();
                    }
                    ((HttpsURLConnection) connection).setSSLSocketFactory(socketFactory);
                } catch (GMSSLContext.GMSSLException | IOException e) {
                    throw new GMSSLHttpException(e, GMSSLHttpErrorCode.GENERATE_SSL_SOCKET_FACTORY_ERROR);
                }
            }
        }
    }

    private void writeRequestBody(HttpURLConnection connection, GMSSLHttpRequest request) throws GMSSLHttpException {
        byte[] body = request.getBody();
        if (body != null && body.length > 0) {
//            logger.info("http request body String {}", new String(body));
//            logger.info("http request body base64Encode {}", GMSSLByteArrayUtils.base64Encode(body));
//            logger.info("http request body hexEncode {}", GMSSLByteArrayUtils.hexEncode(body));
            OutputStream outputStream;
            try {
                outputStream = connection.getOutputStream();
            } catch (IOException e) {
                throw new GMSSLHttpException(e, GMSSLHttpErrorCode.CONNECTION_GET_OUTPUT_STREAM_ERROR);
            }
            DataOutputStream dos = new DataOutputStream(outputStream);
            try {
                dos.write(body);
            } catch (IOException e) {
                throw new GMSSLHttpException("write error", e, GMSSLHttpErrorCode.CONNECTION_WRITE_BODY_ERROR);
            } finally {
                try {
                    dos.flush();
                } catch (IOException e) {
                    throw new GMSSLHttpException("flush error", e, GMSSLHttpErrorCode.CONNECTION_WRITE_BODY_ERROR);
                } finally {
                    try {
                        dos.close();
                        outputStream.close();
                    } catch (IOException e) {
                        throw new GMSSLHttpException("close error", e, GMSSLHttpErrorCode.CONNECTION_WRITE_BODY_ERROR);
                    }
                }
            }
        }
    }

    private void readResponseBody(HttpURLConnection connection, int statusCode, GMSSLHttpResponse response) throws GMSSLHttpException {
        int contentLength = connection.getContentLength();
//        if (contentLength == 0) {
//            logger.info("connection content length is 0");
//            return;
//        }
        // http response body
        InputStream inputStream;
        try {
            if (200 <= statusCode && statusCode <= 299) {
                inputStream = connection.getInputStream();
            } else {
                inputStream = connection.getErrorStream();
            }
        } catch (IOException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.CONNECTION_GET_INPUT_STREAM_ERROR);
        }
//        try {
//            Object content = connection.getContent();
//            logger.info("connection content length is {}, content is {}", contentLength, content);
//        } catch (IOException e) {
//            logger.error("get connection content error", e);
//        }
        if (inputStream == null) {
            logger.error("connection content length is {}, but input stream is null", contentLength);
            return;
        }

        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        int length = 0;
        try {
            while ((length = inputStream.read(buffer)) != -1) {
                bos.write(buffer, 0, length);
            }
        } catch (IOException e) {
            throw new GMSSLHttpException(e, GMSSLHttpErrorCode.CONNECTION_GET_RESPONSE_BODY_ERROR);
        } finally {
            try {
                bos.close();
            } catch (IOException e) {
                throw new GMSSLHttpException("close bos error", e, GMSSLHttpErrorCode.CONNECTION_GET_RESPONSE_BODY_ERROR);
            } finally {
                try {
                    if (inputStream != null)
                        inputStream.close();
                } catch (IOException e) {
                    throw new GMSSLHttpException("close input stream error", e, GMSSLHttpErrorCode.CONNECTION_GET_RESPONSE_BODY_ERROR);
                }
            }
        }
        byte[] body = bos.toByteArray();
//        logger.info("http response body String {}", new String(body));
//        logger.info("http response body base64Encode {}", GMSSLByteArrayUtils.base64Encode(body));
//        logger.info("http response body hexEncode {}", GMSSLByteArrayUtils.hexEncode(body));
        response.setBody(body);
    }
}
