package com.xdja.pki.gmssl.http;

import com.xdja.pki.gmssl.http.bean.*;
import com.xdja.pki.gmssl.http.exception.GMSSLHttpErrorCode;
import com.xdja.pki.gmssl.http.exception.GMSSLHttpException;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.security.KeyStore;
import java.security.cert.X509Certificate;

public class GMSSLHttpsClient extends GMSSLHttpClient {
    private Logger logger = LoggerFactory.getLogger(GMSSLHttpsClient.class);

    public GMSSLHttpsClient() {
        super();
        GMSSLHttpsClientConfig httpsClientConfig = new GMSSLHttpsClientConfig();
        httpsClientConfig.setSslEnabled(true);
        httpsClientConfig.setSslProtocol(GMSSLProtocol.GMSSLv11.getValue());
        this.httpClientConfig = httpsClientConfig;
    }

    public GMSSLHttpsClient(GMSSLHttpClientConfig httpClientConfig) {
        super(httpClientConfig);
    }

    public GMSSLHttpsClient(X509Certificate caCert) throws GMSSLHttpException {
        this(new X509Certificate[]{caCert});
    }

    public GMSSLHttpsClient(X509Certificate[] caCerts) throws GMSSLHttpException {
        super();
        GMSSLHttpsClientConfig httpsClientConfig = new GMSSLHttpsClientConfig();
        httpsClientConfig.setSslEnabled(true);
        KeyStore keyStore;
        logger.info("get caCerts signAlg is " + caCerts[0].getSigAlgName());
        try {
            if (caCerts[0].getSigAlgName().contains("RSA") || caCerts[0].getSigAlgName().contains("ECDSA")) {
                keyStore = KeyStore.getInstance("JKS");
                httpsClientConfig.setSslProtocol(GMSSLProtocol.TLSV12.getValue());
            } else {
                keyStore = KeyStore.getInstance("BKS", BouncyCastleProvider.PROVIDER_NAME);
                httpsClientConfig.setSslProtocol(GMSSLProtocol.GMSSLv11.getValue());
            }
            keyStore.load(null, "password".toCharArray());
            for (int i = 0; i < caCerts.length; i++) {
                keyStore.setCertificateEntry("trust-" + i, caCerts[i]);
            }
            httpsClientConfig.setTrustStore(keyStore);
            httpsClientConfig.setTrustStorePassword("password");
        } catch (Exception e) {
            e.printStackTrace();
            throw new GMSSLHttpException("client keystore", GMSSLHttpErrorCode.GENERATE_TRUST_KEYSTORE__ERROR);
        }
        this.httpClientConfig = httpsClientConfig;
    }

    public GMSSLHttpsClient(X509Certificate[] caCerts, boolean isByYunHsm) throws GMSSLHttpException {
        super();
        GMSSLHttpsClientConfig httpsClientConfig = new GMSSLHttpsClientConfig();
        httpsClientConfig.setSslEnabled(true);
        KeyStore keyStore;
        logger.info("get caCerts signAlg is " + caCerts[0].getSigAlgName());
        try {
            if (caCerts[0].getSigAlgName().contains("RSA")) {
                keyStore = KeyStore.getInstance("JKS");
                httpsClientConfig.setSslProtocol(GMSSLProtocol.TLSV12.getValue());
            } else {
                if (isByYunHsm) {
                    keyStore = KeyStore.getInstance("BKS", BouncyCastleProvider.PROVIDER_NAME);
                    httpsClientConfig.setSslProtocol(GMSSLProtocol.GMSSLSDFYUNHSMv11.getValue());
                } else {
                    keyStore = KeyStore.getInstance("BKS", BouncyCastleProvider.PROVIDER_NAME);
                    httpsClientConfig.setSslProtocol(GMSSLProtocol.GMSSLv11.getValue());
                }
            }
            keyStore.load(null, "password".toCharArray());
            for (int i = 0; i < caCerts.length; i++) {
                keyStore.setCertificateEntry("trust-" + i, caCerts[i]);
            }
            httpsClientConfig.setTrustStore(keyStore);
            httpsClientConfig.setTrustStorePassword("password");
        } catch (Exception e) {
            e.printStackTrace();
            throw new GMSSLHttpException("client keystore", GMSSLHttpErrorCode.GENERATE_TRUST_KEYSTORE__ERROR);
        }
        this.httpClientConfig = httpsClientConfig;
    }

    private void checkConfig() throws GMSSLHttpException {
        if (!(this.httpClientConfig instanceof GMSSLHttpsClientConfig)) {
            throw new GMSSLHttpException("config", GMSSLHttpErrorCode.HTTPS_CONFIG_ERROR);
        }
        GMSSLHttpsClientConfig httpsClientConfig = (GMSSLHttpsClientConfig) this.httpClientConfig;
        if (!(httpsClientConfig.isSslEnabled() && httpsClientConfig.getSslProtocol() != null)) {
            throw new GMSSLHttpException("ssl protocol", GMSSLHttpErrorCode.HTTPS_CONFIG_ERROR);
        }
    }

    @Override
    public GMSSLHttpResponse get(GMSSLHttpRequest request) throws GMSSLHttpException {
        checkConfig();
        return super.get(request);
    }

    @Override
    public GMSSLHttpResponse post(GMSSLHttpRequest request) throws GMSSLHttpException {
        checkConfig();
        return super.post(request);
    }

    @Override
    public GMSSLHttpResponse put(GMSSLHttpRequest request) throws GMSSLHttpException {
        checkConfig();
        return super.put(request);
    }

    @Override
    public GMSSLHttpResponse delete(GMSSLHttpRequest request) throws GMSSLHttpException {
        checkConfig();
        return super.delete(request);
    }
}
