package org.bouncycastle.jsse.provider;

import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.util.PrivateKeyFactory;
import org.bouncycastle.jsse.BCSNIMatcher;
import org.bouncycastle.jsse.BCSNIServerName;
import org.bouncycastle.tls.*;
import org.bouncycastle.tls.crypto.TlsCrypto;
import org.bouncycastle.tls.crypto.TlsCryptoParameters;
import org.bouncycastle.tls.crypto.TlsDHConfig;
import org.bouncycastle.tls.crypto.impl.AbstractTlsCrypto;
import org.bouncycastle.tls.crypto.impl.bc.*;
import org.bouncycastle.tls.crypto.impl.jcajce.JcaDefaultTlsCredentialedSigner;
import org.bouncycastle.tls.crypto.impl.jcajce.JcaTlsCrypto;
import org.bouncycastle.tls.crypto.impl.jcajce.JceDefaultTlsCredentialedAgreement;
import org.bouncycastle.tls.crypto.impl.jcajce.JceDefaultTlsCredentialedDecryptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLException;
import javax.net.ssl.X509KeyManager;
import javax.net.ssl.X509TrustManager;
import java.io.IOException;
import java.net.Socket;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.*;

//GMSSL SUPPORT: : 2018/7/18 remove public
public class ProvTlsServer extends DefaultTlsServer implements ProvTlsPeer {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final int provEphemeralDHKeySize = PropertyUtils.getIntegerSystemProperty("jdk.tls.ephemeralDHKeySize", 2048, 1024, 8192);

    protected final ProvTlsManager manager;
    protected final ProvSSLParameters sslParameters;

    protected ProvSSLSessionImpl sslSession = null;
    protected BCSNIServerName matchedSNIServerName = null;
    protected Set<String> keyManagerMissCache = null;
    protected TlsCredentials credentials = null;
    protected boolean handshakeComplete = false;

    ProvTlsServer(ProvTlsManager manager, ProvSSLParameters sslParameters) throws SSLException {
        super(manager.getContextData().getCrypto());

        this.manager = manager;
        this.sslParameters = sslParameters;

        if (!manager.getEnableSessionCreation()) {
            throw new SSLException("Session resumption not implemented yet and session creation is disabled");
        }
    }

    //获取最大可协商曲线二进制数字
    @Override
    protected int getMaximumNegotiableCurveBits() {
        // NOTE: BC supports all the current set of point formats so we don't check them here
        //BC支持所有当前的点格式集合所以我们不会在这里检查它们

        return SupportedGroups.getServerMaximumNegotiableCurveBits(manager.getContext().isFips(), clientSupportedGroups);
    }

    //获得最大可转让的有限字段位
    @Override
    protected int getMaximumNegotiableFiniteFieldBits() {
        int maxBits = SupportedGroups.getServerMaximumNegotiableFiniteFieldBits(manager.getContext().isFips(), clientSupportedGroups);

        return maxBits >= provEphemeralDHKeySize ? maxBits : 0;
    }

    @Override
    protected ProtocolVersion getMaximumVersion() {
        return manager.getContext().getMaximumVersion(sslParameters.getProtocols());
    }

    @Override
    protected boolean selectCipherSuite(int cipherSuite) throws IOException {
        //选择证书管理器
        if (!selectCredentials(cipherSuite)) {
            return false;
        }

        //验证协商的加密套件
        manager.getContext().validateNegotiatedCipherSuite(cipherSuite);

        return super.selectCipherSuite(cipherSuite);
    }

    @Override
    protected int selectCurve(int minimumCurveBits) {
        if (clientSupportedGroups == null) {
            return selectDefaultCurve(minimumCurveBits);
        }

        boolean isFips = manager.getContext().isFips();

        return SupportedGroups.getServerSelectedCurve(isFips, minimumCurveBits, clientSupportedGroups);
    }

    @Override
    protected int selectDefaultCurve(int minimumCurveBits) {
        return SupportedGroups.getServerDefaultCurve(manager.getContext().isFips(), minimumCurveBits);
    }

    @Override
    protected TlsDHConfig selectDefaultDHConfig(int minimumFiniteFieldBits) {
        return SupportedGroups.getServerDefaultDHConfig(manager.getContext().isFips(), minimumFiniteFieldBits);
    }

    @Override
    protected TlsDHConfig selectDHConfig(int minimumFiniteFieldBits) {
        minimumFiniteFieldBits = Math.max(minimumFiniteFieldBits, provEphemeralDHKeySize);

        if (clientSupportedGroups == null) {
            return selectDefaultDHConfig(minimumFiniteFieldBits);
        }

        boolean isFips = manager.getContext().isFips();

        int namedGroup = SupportedGroups.getServerSelectedFiniteField(isFips, minimumFiniteFieldBits,
                clientSupportedGroups);

        return TlsDHUtils.createNamedDHConfig(namedGroup);
    }

    @Override
    public synchronized boolean isHandshakeComplete() {
        return handshakeComplete;
    }

    @Override
    public TlsCredentials getCredentials()
            throws IOException {
        return credentials;
    }

    @Override
    public int[] getCipherSuites() {
        return TlsUtils.getSupportedCipherSuites(manager.getContextData().getCrypto(),
                manager.getContext().convertCipherSuites(sslParameters.getCipherSuites()));
    }

    @Override
    protected short[] getCompressionMethods() {
        return manager.getContext().isFips()
                ? new short[]{CompressionMethod._null}
                : super.getCompressionMethods();
    }

//  public TlsKeyExchange getKeyExchange() throws IOException
//  {
//      // TODO[jsse] Check that all key exchanges used in JSSE supportedCipherSuites are handled
//      return super.getKeyExchange();
//  }

    @Override
    public CertificateRequest getCertificateRequest() throws IOException {
        boolean shouldRequest = sslParameters.getNeedClientAuth() || sslParameters.getWantClientAuth();
        if (!shouldRequest) {
            return null;
        }


        short[] certificateTypes = new short[]{ClientCertificateType.rsa_sign,
                ClientCertificateType.dss_sign, ClientCertificateType.ecdsa_sign, ClientCertificateType.ibc_params};

        Vector serverSigAlgs = null;
        if (TlsUtils.isSignatureAlgorithmsExtensionAllowed(serverVersion)) {
            serverSigAlgs = JsseUtils.getSupportedSignatureAlgorithms(getCrypto());
        }

        Vector certificateAuthorities = new Vector();
        X509TrustManager tm = manager.getContextData().getTrustManager();
        if (tm != null) {
            for (X509Certificate caCert : tm.getAcceptedIssuers()) {
                certificateAuthorities.addElement(X500Name.getInstance(caCert.getSubjectX500Principal().getEncoded()));
            }
        }

        return new CertificateRequest(certificateTypes, serverSigAlgs, certificateAuthorities);
    }

    @Override
    public int getSelectedCipherSuite() throws IOException {
        keyManagerMissCache = new HashSet<String>();

        int selectedCipherSuite = super.getSelectedCipherSuite();

        logger.debug("Server selected CipherSuite: " + manager.getContext().getCipherSuiteString(selectedCipherSuite));

        keyManagerMissCache = null;

        return selectedCipherSuite;
    }

    @Override
    public Hashtable getServerExtensions() throws IOException {
        super.getServerExtensions();

        /*
         * TODO[jsse] RFC 6066 When resuming a session, the server MUST NOT include a server_name
         * extension in the server hello.
         */
        if (matchedSNIServerName != null) {
            checkServerExtensions().put(TlsExtensionsUtils.EXT_server_name, TlsExtensionsUtils.createEmptyExtensionData());
        }

        return serverExtensions;
    }

    @Override
    public TlsSession getSessionToResume(byte[] sessionID) {
        ProvSSLSessionContext sessionContext = manager.getContextData().getServerSessionContext();
        this.sslSession = sessionContext.getSessionImpl(sessionID);

        if (sslSession != null) {
            TlsSession sessionToResume = sslSession.getTlsSession();
            if (sessionToResume != null) {
                return sessionToResume;
            }
        }

        if (!manager.getEnableSessionCreation()) {
            throw new IllegalStateException("No resumable sessions and session creation is disabled");
        }

        return null;
    }

    @Override
    public void notifyAlertRaised(short alertLevel, short alertDescription, String message, Throwable cause) {

        String msg = JsseUtils.getAlertLogMessage("Server raised", alertLevel, alertDescription);
        if (message != null) {
            msg = msg + ": " + message;
        }
        if (alertLevel == AlertLevel.warning || alertDescription == AlertDescription.internal_error) {
            logger.warn(msg);
        } else if (alertLevel == AlertLevel.fatal) {
            logger.error(msg);
        } else {
            logger.info(msg);
        }
    }

    @Override
    public void notifyAlertReceived(short alertLevel, short alertDescription) {
        super.notifyAlertReceived(alertLevel, alertDescription);

        String msg = JsseUtils.getAlertLogMessage("Server received", alertLevel, alertDescription);
        if (alertLevel == AlertLevel.warning) {
            logger.warn(msg);
        } else {
            logger.info(msg);
        }
    }

    @Override
    public ProtocolVersion getServerVersion() throws IOException {
        /*
         * TODO[jsse] It may be best to just require the "protocols" list to be a contiguous set
         * (especially in light of TLS_FALLBACK_SCSV), then just determine the minimum/maximum,
         * and keep the super class implementation of this.
         */
        String[] protocols = sslParameters.getProtocols();
        logger.debug("sslParameters getProtocols: {}", Arrays.toString(protocols));
        if (protocols != null && protocols.length > 0) {
            ProtocolVersion version = clientVersion;
            if (version != null) {
                version = version.getPreviousVersion();
                String versionString = manager.getContext().getProtocolString(version);
                logger.debug("versionString: {} protocols: {}", versionString, protocols);
                if (versionString != null && JsseUtils.contains(protocols, versionString)) {
                    logger.debug("server selected Protocol: {}", version);
                    return serverVersion = version;
                }
            }
        }
        throw new TlsFatalAlert(AlertDescription.protocol_version);
    }

    @Override
    public void notifyClientCertificate(Certificate clientCertificate) throws IOException {
        // NOTE: This method isn't called unless we returned non-null from getCertificateRequest() earlier
        assert sslParameters.getNeedClientAuth() || sslParameters.getWantClientAuth();

        boolean noClientCert = clientCertificate == null || clientCertificate.isEmpty();
        if (noClientCert) {
            if (sslParameters.getNeedClientAuth()) {
                logger.info("client cert is null!");
                throw new TlsFatalAlert(AlertDescription.handshake_failure);
            }
        } else {
            X509Certificate[] chain = JsseUtils.getX509CertificateChain(manager.getContextData().getCrypto(), clientCertificate);
            short clientCertificateType = clientCertificate.getCertificateAt(0).getClientCertificateType();
            String authType = JsseUtils.getAuthTypeClient(clientCertificateType);

            if (!manager.isClientTrusted(chain, authType)) {
                /*
                 * TODO[jsse] The low-level TLS API currently doesn't provide a way to indicate that
                 * we wish to proceed with an untrusted client certificate, so we always fail here.
                 */
                logger.info(
                        "client cert not trust! subjectDN={} sn={} issueDN={}",
                        chain[0].getSubjectDN().toString(),
                        chain[0].getSerialNumber(),
                        chain[0].getIssuerDN().toString()
                );
                throw new TlsFatalAlert(AlertDescription.bad_certificate);
            }
        }
    }

    @Override
    public synchronized void notifyHandshakeComplete() throws IOException {
        this.handshakeComplete = true;

        TlsSession handshakeSession = context.getSession();

        if (sslSession == null || sslSession.getTlsSession() != handshakeSession) {
            sslSession = manager.getContextData().getServerSessionContext().reportSession(handshakeSession, null, -1);
        }

        manager.notifyHandshakeComplete(new ProvSSLConnection(context, sslSession));
    }

    @Override
    public void notifySecureRenegotiation(boolean secureRenegotiation) throws IOException {
        if (!secureRenegotiation) {
            boolean allowLegacyHelloMessages = PropertyUtils.getBooleanSystemProperty("sun.security.ssl.allowLegacyHelloMessages", true);
            if (!allowLegacyHelloMessages) {
                /*
                 * RFC 5746 3.4/3.6. In this case, some clients/servers may want to terminate the handshake instead
                 * of continuing; see Section 4.1/4.3 for discussion.
                 */
                throw new TlsFatalAlert(AlertDescription.handshake_failure);
            }
        }
    }

    @Override
    public void processClientExtensions(Hashtable clientExtensions) throws IOException {
        super.processClientExtensions(clientExtensions);

        if (clientExtensions != null) {
            /*
             * TODO[jsse] RFC 6066 A server that implements this extension MUST NOT accept the
             * request to resume the session if the server_name extension contains a different name.
             */
            Collection<BCSNIMatcher> sniMatchers = sslParameters.getSNIMatchers();
            if (sniMatchers != null && !sniMatchers.isEmpty()) {
                ServerNameList serverNameList = TlsExtensionsUtils.getServerNameExtension(clientExtensions);
                if (serverNameList != null) {
                    matchedSNIServerName = JsseUtils.findMatchingSNIServerName(serverNameList, sniMatchers);
                    if (matchedSNIServerName == null) {
                        throw new TlsFatalAlert(AlertDescription.unrecognized_name);
                    }
                }
            }
        }
    }

    // GMSSL SUPPORT: 2018/7/12 gmssl 双证
    protected boolean selectCredentials(int cipherSuite) throws IOException {
        this.credentials = null;

        int keyExchangeAlgorithm = TlsUtils.getKeyExchangeAlgorithm(cipherSuite);
        switch (keyExchangeAlgorithm) {
            case KeyExchangeAlgorithm.DH_anon:
            case KeyExchangeAlgorithm.ECDH_anon:
                return true;

            //以下几种算法放到下边判断
            case KeyExchangeAlgorithm.DH_DSS:
            case KeyExchangeAlgorithm.DH_RSA:
            case KeyExchangeAlgorithm.DHE_DSS:
            case KeyExchangeAlgorithm.DHE_RSA:
            case KeyExchangeAlgorithm.ECDH_ECDSA:
            case KeyExchangeAlgorithm.ECDH_RSA:
            case KeyExchangeAlgorithm.ECDHE_ECDSA:
            case KeyExchangeAlgorithm.ECDHE_RSA:
                //GMSSL SUPPORT: add sm2
            case KeyExchangeAlgorithm.RSA:
                return generateCredentials(keyExchangeAlgorithm);

            case KeyExchangeAlgorithm.ECC_SM2:
                return generateCredentialsGMSSL(keyExchangeAlgorithm);
            default:
                return false;
        }


    }

    //GMSSL SUPPORT:  for ecc sm2 select credentials
    private boolean generateCredentialsGMSSL(int keyExchangeAlgorithm) throws IOException {
        if (!(getCrypto() instanceof BcTlsCrypto) && !(getCrypto() instanceof BcTlsCryptoSdf)) {
            throw new TlsFatalAlert(AlertDescription.internal_error);
        }

        short signatureAlgorithm = TlsUtils.getSignatureAlgorithm(keyExchangeAlgorithm);
        SignatureAndHashAlgorithm sigAlg = TlsUtils.chooseSignatureAndHashAlgorithm(context,
                supportedSignatureAlgorithms, signatureAlgorithm);

        X509KeyManager keyManager = manager.getContextData().getKeyManager();
        if (keyManager == null) {
            logger.info("generate credentials gmssl x509 keyManager is null!");
            return false;
        }

        X509TrustManager trustManager = manager.getContextData().getTrustManager();
        if (trustManager == null) {
            logger.info("generate credentials gmssl x509 trustManager is null!");
            return false;
        }

        String keyType = JsseUtils.getAuthTypeServer(keyExchangeAlgorithm);
        if (keyManagerMissCache.contains(keyType)) {
            logger.info("generate credentials keyType: " + keyType + " keyManagerMissCache: " + keyManagerMissCache);
            return false;
        }

        this.credentials = GMSSLUtils.generateCredentials(keyType, (AbstractTlsCrypto) getCrypto(), keyManager, trustManager, sigAlg, context, logger);

        return credentials != null;
    }

    private boolean generateCredentials(int keyExchangeAlgorithm) throws IOException {
        X509KeyManager km = manager.getContextData().getKeyManager();
        if (km == null) {
            return false;
        }

        String keyType = JsseUtils.getAuthTypeServer(keyExchangeAlgorithm);
        if (keyManagerMissCache.contains(keyType)) {
            return false;
        }

        // TODO[jsse] Is there some extension where the client can specify these (SNI maybe)?
        //是否有一些扩展，客户端可以指定这些（SNI可能）？
        Principal[] issuers = null;
        // TODO[jsse] How is this used?
        //这是如何使用?
        Socket socket = null;

        //GMSSL SUPPORT: 获取别名，这里是一个keyManager对于双证要想下如何处理
        String alias = km.chooseServerAlias(keyType, issuers, socket);
        if (alias == null) {
            keyManagerMissCache.add(keyType);
            return false;
        }

        //GMSSL SUPPORT: 获取 crypto
        TlsCrypto crypto = getCrypto();

        if (!(crypto instanceof JcaTlsCrypto) && !(crypto instanceof BcTlsCrypto)) {
            // TODO[jsse] Need to have TlsCrypto construct the credentials from the certs/key
            //需要有TlsCrypto从cer/key中构造凭证
            throw new UnsupportedOperationException();
        }

        //GMSSL SUPPORT: 获取 private key
        PrivateKey privateKey = km.getPrivateKey(alias);
        //GMSSL SUPPORT: 获取 证书
        Certificate certificate = JsseUtils.getCertificateMessage(crypto, km.getCertificateChain(alias));

        //key 是否可用、certificate 是否为空 检查
        if (privateKey == null
                || !JsseUtils.isUsableKeyForServer(keyExchangeAlgorithm, privateKey)
                || certificate.isEmpty()) {
            keyManagerMissCache.add(keyType);
            return false;
        }

        /*
         * TODO[jsse] Before proceeding with EC credentials, should we check (TLS 1.2+) that the
         * used curve is supported by the client according to the elliptic_curves/named_groups
         * extension?
         */

        switch (keyExchangeAlgorithm) {
            case KeyExchangeAlgorithm.DH_DSS:
            case KeyExchangeAlgorithm.DH_RSA:
            case KeyExchangeAlgorithm.ECDH_ECDSA:
            case KeyExchangeAlgorithm.ECDH_RSA: {
                // TODO[jsse] Need to have TlsCrypto construct the credentials from the certs/key
                //需要使用TlsCrypto从cer/key中构造凭证
                if (crypto instanceof JcaTlsCrypto) {
                    this.credentials = new JceDefaultTlsCredentialedAgreement((JcaTlsCrypto) crypto, certificate, privateKey);
                } else {
                    //GMSSL SUPPORT: : 2018/8/2 use bc tls crypto
                    AsymmetricKeyParameter parameter = PrivateKeyFactory.createKey(PrivateKeyInfo.getInstance(privateKey.getEncoded()));
                    this.credentials = new BcDefaultTlsCredentialedAgreement((BcTlsCrypto) crypto, certificate, parameter);
                }
                return true;
            }

            case KeyExchangeAlgorithm.DHE_DSS:
            case KeyExchangeAlgorithm.DHE_RSA:
            case KeyExchangeAlgorithm.ECDHE_ECDSA:
            case KeyExchangeAlgorithm.ECDHE_RSA: {
                short signatureAlgorithm = TlsUtils.getSignatureAlgorithm(keyExchangeAlgorithm);
                SignatureAndHashAlgorithm sigAlg = TlsUtils.chooseSignatureAndHashAlgorithm(context,
                        supportedSignatureAlgorithms, signatureAlgorithm);

                // TODO[jsse] Need to have TlsCrypto construct the credentials from the certs/key
                //需要有TlsCrypto从cer/key中构造凭证
                //GMSSL SUPPORT: certificate 这时和 JcaDefaultTlsCredentialedSigner 关联，可能牵涉到相关类的改动
                if (crypto instanceof JcaTlsCrypto) {
                    this.credentials = new JcaDefaultTlsCredentialedSigner(new TlsCryptoParameters(context), (JcaTlsCrypto) crypto,
                            privateKey, certificate, sigAlg);
                } else {
                    //GMSSL SUPPORT: : 2018/8/2 use bc tls crypto
                    AsymmetricKeyParameter parameter = PrivateKeyFactory.createKey(PrivateKeyInfo.getInstance(privateKey.getEncoded()));
                    this.credentials = new BcDefaultTlsCredentialedSigner(new TlsCryptoParameters(context), (BcTlsCrypto) crypto,
                            parameter, certificate, sigAlg);
                }
                return true;
            }

            case KeyExchangeAlgorithm.RSA: {
                // TODO[jsse] Need to have TlsCrypto construct the credentials from the certs/key
                if (crypto instanceof JcaTlsCrypto) {
                    this.credentials = new JceDefaultTlsCredentialedDecryptor((JcaTlsCrypto) crypto, certificate, privateKey);
                } else {
                    //GMSSL SUPPORT: : 2018/8/2 use bc tls crypto
                    AsymmetricKeyParameter parameter = PrivateKeyFactory.createKey(PrivateKeyInfo.getInstance(privateKey.getEncoded()));
                    this.credentials = new BcDefaultTlsCredentialedDecryptor((BcTlsCrypto) crypto, certificate, parameter);
                }
                return true;
            }

            default:
                return false;
        }
    }
}
