package org.bouncycastle.tls;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import org.bouncycastle.tls.crypto.*;
import org.bouncycastle.tls.crypto.impl.bc.BcDefaultTlsCredentialedECCSM2;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Vector;

/**
 * GMSSL ECC SM2 key exchange (see GMT 0024-2014).
 */
//GMSSL SUPPORT: : 2018/7/17 ECC Key Exchange
public class TlsECCSM2KeyExchange extends AbstractTlsKeyExchange {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    //客户端证书
    protected TlsCertificate peerCertificate;

    public BcDefaultTlsCredentialedECCSM2 serverCredentials = null;

    //加密证书
    protected TlsCertificate encryptionCertificate;
    protected TlsVerifier verifier = null;

    //客户端生成  预主密钥
    protected TlsSecret preMasterSecret;

    public TlsECCSM2KeyExchange(int keyExchange, Vector supportedSignatureAlgorithms,
                                TlsECConfigVerifier ecConfigVerifier, short[] clientECPointFormats, short[] serverECPointFormats) {
        this(keyExchange, supportedSignatureAlgorithms, ecConfigVerifier, null, clientECPointFormats,
                serverECPointFormats);
    }

    public TlsECCSM2KeyExchange(int keyExchange, Vector supportedSignatureAlgorithms, TlsECConfig ecConfig,
                                short[] serverECPointFormats) {
        this(keyExchange, supportedSignatureAlgorithms, null, ecConfig, null, serverECPointFormats);
    }

    private TlsECCSM2KeyExchange(int keyExchange, Vector supportedSignatureAlgorithms,
                                 TlsECConfigVerifier ecConfigVerifier, TlsECConfig ecConfig, short[] clientECPointFormats,
                                 short[] serverECPointFormats) {
        super(keyExchange, supportedSignatureAlgorithms);
    }

    //跳过 服务端 证书认证 TlsClientProtocol TlsServerProtocol
    @Override
    public void skipServerCredentials() throws IOException {
        //不允许跳过 服务端 证书认证
        throw new TlsFatalAlert(AlertDescription.internal_error);
    }

    //进行 服务端 证书认证 TlsServerProtocol
    @Override
    public void processServerCredentials(TlsCredentials serverCredentials) throws IOException {
        if (!(serverCredentials instanceof BcDefaultTlsCredentialedECCSM2)) {
            throw new TlsFatalAlert(AlertDescription.internal_error);
        }

        this.serverCredentials = (BcDefaultTlsCredentialedECCSM2) serverCredentials;

        this.encryptionCertificate = this.serverCredentials.getEncryptionCertificate();
    }
    //TlsServerProtocol 调用的是上边的 processServerCredentials

    //TlsClientProtocol 进行 服务端 证书 TlsClientProtocol
    @Override
    public void processServerCertificate(Certificate serverCertificate) throws IOException {
        if (serverCertificate.isEmpty()) {
            throw new TlsFatalAlert(AlertDescription.bad_certificate);
        }

        checkServerCertSigAlg(serverCertificate);

        //GMSSL SUPPORT:  服务端传送的 证书列表 签名证书在前、加密证书在后
        this.encryptionCertificate = GMSSLUtils.getEncryptionCertificate(serverCertificate);
        this.verifier = GMSSLUtils.getSignatureCertificate(serverCertificate).createVerifier(TlsUtils.getSignatureAlgorithm(keyExchange));
    }

    //需要 服务端 密钥交换
    @Override
    public boolean requiresServerKeyExchange() {
        return true;
    }

    //生成 服务端 密钥交换 TlsServerProtocol
    @Override
    public byte[] generateServerKeyExchange() throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();

        //生成服务端 key exchange 消息 并签名
        byte[] signature = GMSSLUtils.generateECCSM2ServerKeyExchangeSignature(context, serverCredentials, encryptionCertificate);
        TlsUtils.writeOpaque16(signature, out);

        byte[] signed = out.toByteArray();
        GMSSLByteArrayUtils.printHexBinary(logger, "generateServerKeyExchange signed", signed);

        return signed;
    }

    //进行 服务端 密钥交换 TlsClientProtocol
    @Override
    public void processServerKeyExchange(InputStream input) throws IOException {
        byte[] signature = TlsUtils.readOpaque16(input);

        GMSSLByteArrayUtils.printHexBinary(logger, "processServerKeyExchange signature", signature);

        DigitallySigned signedParams = new DigitallySigned(new SignatureAndHashAlgorithm(HashAlgorithm.sm3, SignatureAlgorithm.sm2), signature);

        GMSSLUtils.verifyECCSM2ServerKeyExchangeSignature(context, verifier, signedParams, encryptionCertificate);
    }

    //获取 客户端 证书 类型 TlsClientProtocol TlsServerProtocol
    @Override
    public short[] getClientCertificateTypes() {
        //GMSSL SUPPORT: : 2018/7/27 add gmssl support client certificate types
        return new short[]{ClientCertificateType.rsa_sign, ClientCertificateType.ecdsa_sign, ClientCertificateType.ibc_params};
    }

    //进行 客户端 证书 TlsServerProtocol
    @Override
    public void processClientCredentials(TlsCredentials clientCredentials) throws IOException {
        //GMSSL SUPPORT: : 2018/7/31 undo 客户端证书验证
        if (!(clientCredentials instanceof TlsCredentialedSigner)) {
            throw new TlsFatalAlert(AlertDescription.internal_error);
        }
    }

    //生成 客户端 密钥交换 TlsClientProtocol
    @Override
    public void generateClientKeyExchange(OutputStream output) throws IOException {
        //生成 预主密钥，并用服务器加密公钥加密 预主密钥，写入output
        this.preMasterSecret = TlsUtils.generateEncryptedPreMasterSecret(context, encryptionCertificate, output);
    }

    //进行 客户端 证书 TlsServerProtocol
    @Override
    public void processClientCertificate(Certificate clientCertificate) throws IOException {
        if (clientCertificate.isEmpty()) {
            throw new TlsFatalAlert(AlertDescription.bad_certificate);
        }

        // GMSSL SUPPORT: 2018/10/31 客户端证书验证
        // TODO: 2018/10/31  .useInRole(ConnectionEnd.client, keyExchange);
        this.peerCertificate = GMSSLUtils.getSignatureCertificate(clientCertificate);
    }

    //进行 客户端 密钥交换 TlsServerProtocol
    @Override
    public void processClientKeyExchange(InputStream input) throws IOException {
        byte[] encryptedPreMasterSecret = TlsUtils.readOpaque16(input);
        GMSSLByteArrayUtils.printHexBinary(logger, "server receive encrypted pre master secret: ", encryptedPreMasterSecret);
        this.preMasterSecret = serverCredentials.decrypt(new TlsCryptoParameters(context), encryptedPreMasterSecret);
    }

    //生成 预制 主 密钥 TlsClientProtocol TlsServerProtocol
    @Override
    public TlsSecret generatePreMasterSecret() throws IOException {
        TlsSecret tmp = this.preMasterSecret;
        this.preMasterSecret = null;
        return tmp;
    }
}
