package org.bouncycastle.jsse.provider;

import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.util.PrivateKeyFactory;
import org.bouncycastle.jsse.BCSNIServerName;
import org.bouncycastle.tls.*;
import org.bouncycastle.tls.crypto.TlsCrypto;
import org.bouncycastle.tls.crypto.TlsCryptoParameters;
import org.bouncycastle.tls.crypto.impl.AbstractTlsCrypto;
import org.bouncycastle.tls.crypto.impl.bc.BcDefaultTlsCredentialedAgreement;
import org.bouncycastle.tls.crypto.impl.bc.BcDefaultTlsCredentialedSigner;
import org.bouncycastle.tls.crypto.impl.bc.BcTlsCrypto;
import org.bouncycastle.tls.crypto.impl.bc.BcTlsCryptoSdf;
import org.bouncycastle.tls.crypto.impl.jcajce.JcaDefaultTlsCredentialedSigner;
import org.bouncycastle.tls.crypto.impl.jcajce.JcaTlsCrypto;
import org.bouncycastle.tls.crypto.impl.jcajce.JceDefaultTlsCredentialedAgreement;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.IPAddress;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.X509KeyManager;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.x500.X500Principal;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.Socket;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Set;
import java.util.Vector;

// GMSSL SUPPORT: : 2018/7/18 remove public
public class ProvTlsClient extends DefaultTlsClient implements ProvTlsPeer {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final boolean provEnableSNIExtension = PropertyUtils.getBooleanSystemProperty("jsse.enableSNIExtension", true);

    protected final ProvTlsManager manager;
    protected final ProvSSLParameters sslParameters;

    protected ProvSSLSessionImpl sslSession = null;
    protected boolean handshakeComplete = false;

    ProvTlsClient(ProvTlsManager manager, ProvSSLParameters sslParameters) {
        super(manager.getContextData().getCrypto(), new DefaultTlsKeyExchangeFactory(), new ProvDHConfigVerifier());

        this.manager = manager;
        this.sslParameters = sslParameters;
    }

    @Override
    protected CertificateStatusRequest getCertificateStatusRequest() {
        return null;
    }

    @Override
    protected Vector getSupportedGroups(boolean offeringDH, boolean offeringEC) {
        return SupportedGroups.getClientSupportedGroups(manager.getContext().isFips(), offeringDH, offeringEC);
    }

    @Override
    protected Vector getSNIServerNames() {
        if (provEnableSNIExtension) {
            List<BCSNIServerName> sniServerNames = sslParameters.getServerNames();
            if (sniServerNames == null) {
                String peerHost = manager.getPeerHost();
                if (peerHost != null && peerHost.indexOf('.') > 0 && !IPAddress.isValid(peerHost)) {
                    Vector serverNames = new Vector(1);
                    serverNames.addElement(new ServerName(NameType.host_name, peerHost));
                    return serverNames;
                }
            } else {
                Vector serverNames = new Vector(sniServerNames.size());
                for (BCSNIServerName sniServerName : sniServerNames) {
                    /*
                     * TODO[jsse] Add support for constructing ServerName using
                     * BCSNIServerName.getEncoded() directly, then remove the 'host_name' limitation
                     * (although it's currently the only defined type).
                     */
                    if (sniServerName.getType() == NameType.host_name) {
                        try {
                            serverNames.addElement(new ServerName((short) sniServerName.getType(), new String(sniServerName.getEncoded(), "ASCII")));
                        } catch (UnsupportedEncodingException e) {
                            logger.warn("Unable to include SNI server name", e);
                        }
                    }
                }

                // NOTE: We follow SunJSSE behaviour and disable SNI if there are no server names to send
                if (!serverNames.isEmpty()) {
                    return serverNames;
                }
            }
        }
        return null;
    }

    @Override
    protected Vector getSupportedSignatureAlgorithms() {
        return JsseUtils.getSupportedSignatureAlgorithms(getCrypto());
    }

    @Override
    public synchronized boolean isHandshakeComplete() {
        return handshakeComplete;
    }

    @Override
    public TlsAuthentication getAuthentication() throws IOException {
        return new TlsAuthentication() {
            @Override
            public TlsCredentials getClientCredentials(CertificateRequest certificateRequest) throws IOException {
                // TODO[jsse] What criteria determines whether we are willing to send client authentication?

                int keyExchangeAlgorithm = TlsUtils.getKeyExchangeAlgorithm(selectedCipherSuite);
                switch (keyExchangeAlgorithm) {
                    case KeyExchangeAlgorithm.DH_DSS:
                    case KeyExchangeAlgorithm.DH_RSA:
                    case KeyExchangeAlgorithm.ECDH_ECDSA:
                    case KeyExchangeAlgorithm.ECDH_RSA:
                        // TODO[jsse] Add support for the static key exchanges
                        return null;

                    case KeyExchangeAlgorithm.DHE_DSS:
                    case KeyExchangeAlgorithm.DHE_RSA:
                    case KeyExchangeAlgorithm.ECDHE_ECDSA:
                    case KeyExchangeAlgorithm.ECDHE_RSA:
                    case KeyExchangeAlgorithm.RSA:
                        return generateCredentials(keyExchangeAlgorithm, certificateRequest);
                    case KeyExchangeAlgorithm.ECC_SM2:
                        //GMSSL SUPPORT: add sm2
                        return generateCredentialsECCSM2(keyExchangeAlgorithm, certificateRequest);
                    default:
                        /* Note: internal error here; selected a key exchange we don't implement! */
                        throw new TlsFatalAlert(AlertDescription.internal_error);
                }
            }

            //通过 server certificate 消息，获取服务端证书
            @Override
            public void notifyServerCertificate(TlsServerCertificate serverCertificate) throws IOException {
                boolean noServerCert = serverCertificate == null || serverCertificate.getCertificate() == null
                        || serverCertificate.getCertificate().isEmpty();
                if (noServerCert) {
                    throw new TlsFatalAlert(AlertDescription.handshake_failure);
                } else {
                    X509Certificate[] chain = JsseUtils.getX509CertificateChain(manager.getContextData().getCrypto(), serverCertificate.getCertificate());
                    String authType = JsseUtils.getAuthTypeServer(TlsUtils.getKeyExchangeAlgorithm(selectedCipherSuite));

                    if (!manager.isServerTrusted(chain, authType)) {
                        logger.error("server cert is not trust, alert bad_certificate");
                        throw new TlsFatalAlert(AlertDescription.bad_certificate);
                    }
                }
            }
        };
    }

    @Override
    public int[] getCipherSuites() {
        return TlsUtils.getSupportedCipherSuites(manager.getContextData().getCrypto(),
                manager.getContext().convertCipherSuites(sslParameters.getCipherSuites()));
    }

    @Override
    public short[] getCompressionMethods() {
        return manager.getContext().isFips()
                ? new short[]{CompressionMethod._null}
                : super.getCompressionMethods();
    }

//    public TlsKeyExchange getKeyExchange() throws IOException
//    {
//        // TODO[jsse] Check that all key exchanges used in JSSE supportedCipherSuites are handled
//        return super.getKeyExchange();
//    }

    @Override
    public ProtocolVersion getMinimumVersion() {
        return manager.getContext().getMinimumVersion(sslParameters.getProtocols());
    }

    @Override
    public ProtocolVersion getClientVersion() {
        return manager.getContext().getMaximumVersion(sslParameters.getProtocols());
    }

    @Override
    public TlsSession getSessionToResume() {
        ProvSSLSessionContext sessionContext = manager.getContextData().getClientSessionContext();
        this.sslSession = sessionContext.getSessionImpl(manager.getPeerHost(), manager.getPeerPort());

        if (sslSession != null) {
            TlsSession sessionToResume = sslSession.getTlsSession();
            if (sessionToResume != null) {
                return sessionToResume;
            }
        }

        if (!manager.getEnableSessionCreation()) {
            throw new IllegalStateException("No resumable sessions and session creation is disabled");
        }

        return null;
    }

    @Override
    public void notifyAlertRaised(short alertLevel, short alertDescription, String message, Throwable cause) {
        super.notifyAlertRaised(alertLevel, alertDescription, message, cause);

        String msg = JsseUtils.getAlertLogMessage("Client raised", alertLevel, alertDescription);
        if (message != null) {
            msg = msg + ": " + message;
        }
        if (alertLevel == AlertLevel.warning || alertDescription == AlertDescription.internal_error){
            logger.warn(msg);
        } else if (alertLevel == AlertLevel.fatal){
            logger.error(msg);
        } else {
            logger.info(msg);
        }
    }

    @Override
    public void notifyAlertReceived(short alertLevel, short alertDescription) {
        super.notifyAlertReceived(alertLevel, alertDescription);

        String msg = JsseUtils.getAlertLogMessage("Client received", alertLevel, alertDescription);

        if (alertLevel == AlertLevel.warning){
            logger.warn(msg);
        } else {
            logger.info(msg);
        }
    }

    @Override
    public synchronized void notifyHandshakeComplete() throws IOException {
        this.handshakeComplete = true;

        TlsSession handshakeSession = context.getSession();

        if (sslSession == null || sslSession.getTlsSession() != handshakeSession) {
            sslSession = manager.getContextData().getClientSessionContext().reportSession(handshakeSession,
                    manager.getPeerHost(), manager.getPeerPort());
        }

        manager.notifyHandshakeComplete(new ProvSSLConnection(context, sslSession));
    }

    @Override
    public void notifySecureRenegotiation(boolean secureRenegotiation) throws IOException {
        if (!secureRenegotiation) {
            boolean allowLegacyHelloMessages = PropertyUtils.getBooleanSystemProperty("sun.security.ssl.allowLegacyHelloMessages", true);
            if (!allowLegacyHelloMessages) {
                /*
                 * RFC 5746 3.4/3.6. In this case, some clients/servers may want to terminate the handshake instead
                 * of continuing; see Section 4.1/4.3 for discussion.
                 */
                throw new TlsFatalAlert(AlertDescription.handshake_failure);
            }
        }
    }

    @Override
    public void notifySelectedCipherSuite(int selectedCipherSuite) {
        manager.getContext().validateNegotiatedCipherSuite(selectedCipherSuite);

        super.notifySelectedCipherSuite(selectedCipherSuite);

        logger.debug("Client notified of selected CipherSuite: " + manager.getContext().getCipherSuiteString(selectedCipherSuite));
    }

    @Override
    public void notifyServerVersion(ProtocolVersion serverVersion) throws IOException {
        String selected = manager.getContext().getProtocolString(serverVersion);
        if (selected != null) {
            for (String protocol : sslParameters.getProtocols()) {
                if (selected.equals(protocol)) {
                    logger.debug("Client notified of selected Protocol: " + selected);
                    return;
                }
            }
        }
        throw new TlsFatalAlert(AlertDescription.protocol_version);
    }

    @Override
    public void notifySessionID(byte[] sessionID) {
        super.notifySessionID(sessionID);

        if (sessionID == null || sessionID.length == 0) {
            logger.info("Server did not specify a session ID");
        } else if (sslSession != null && Arrays.areEqual(sessionID, sslSession.getId())) {
            logger.debug("Server resumed session: " + Hex.toHexString(sessionID));
        } else if (!manager.getEnableSessionCreation()) {
            throw new IllegalStateException("Server did not resume session and session creation is disabled");
        } else {
            logger.debug("Server specified new Session: " + Hex.toHexString(sessionID));
        }
    }

    // TODO : 2018/8/6 must be support for client certificate
    //GMSSL SUPPORT:  for ecc sm2 select credentials
    private TlsCredentials generateCredentialsECCSM2(int keyExchangeAlgorithm, CertificateRequest certificateRequest) throws IOException {
        if (!(getCrypto() instanceof BcTlsCrypto) && !(getCrypto() instanceof BcTlsCryptoSdf)) {
            throw new TlsFatalAlert(AlertDescription.internal_error);
        }

        short signatureAlgorithm = TlsUtils.getSignatureAlgorithm(keyExchangeAlgorithm);
        SignatureAndHashAlgorithm sigAlg = TlsUtils.chooseSignatureAndHashAlgorithm(context,
                supportedSignatureAlgorithms, signatureAlgorithm);

        X509KeyManager keyManager = manager.getContextData().getKeyManager();
        if (keyManager == null) {
            logger.info("generate credentials gmssl x509 keyManager is null!");
            return null;
        }

        X509TrustManager trustManager = manager.getContextData().getTrustManager();
        if (trustManager == null) {
            logger.info("generate credentials gmssl x509 trustManager is null!");
            return null;
        }

        String keyType = JsseUtils.getAuthTypeServer(keyExchangeAlgorithm);

        return GMSSLUtils.generateCredentials(keyType, (AbstractTlsCrypto) getCrypto(), keyManager, trustManager, sigAlg, context, logger);
    }

    private TlsCredentials generateCredentials(int keyExchangeAlgorithm, CertificateRequest certificateRequest) throws IOException {
        X509KeyManager km = manager.getContextData().getKeyManager();
        if (km == null) {
            return null;
        }

        short[] certTypes = certificateRequest.getCertificateTypes();
        if (certTypes == null || certTypes.length == 0) {
            // TODO[jsse] Or does this mean ANY type - or something else?
            return null;
        }

        String[] keyTypes = new String[certTypes.length];
        for (int i = 0; i < certTypes.length; ++i) {
            // TODO[jsse] Need to also take notice of certificateRequest.getSupportedSignatureAlgorithms(), if present
            keyTypes[i] = JsseUtils.getAuthTypeClient(certTypes[i]);
        }

        Principal[] issuers = null;
        Vector<X500Name> cas = (Vector<X500Name>) certificateRequest.getCertificateAuthorities();
        if (cas != null && cas.size() > 0) {
            X500Name[] names = cas.toArray(new X500Name[cas.size()]);
            Set<X500Principal> principals = JsseUtils.toX500Principals(names);
            issuers = principals.toArray(new Principal[principals.size()]);
        }

        // TODO[jsse] How is this used?
        Socket socket = null;

        String alias = km.chooseClientAlias(keyTypes, issuers, socket);
        if (alias == null) {
            return null;
        }

        TlsCrypto crypto = getCrypto();
        if (!(crypto instanceof JcaTlsCrypto) && !(crypto instanceof BcTlsCrypto)) {
            // TODO[jsse] Need to have TlsCrypto construct the credentials from the certs/key
            throw new UnsupportedOperationException();
        }

        PrivateKey privateKey = km.getPrivateKey(alias);
        Certificate certificate = JsseUtils.getCertificateMessage(crypto, km.getCertificateChain(alias));

        if (privateKey == null || certificate.isEmpty()) {
            // TODO[jsse] Log the probable misconfigured keystore
            return null;
        }

        /*
         * TODO[jsse] Before proceeding with EC credentials, should we check (TLS 1.2+) that
         * the used curve was actually declared in the client's elliptic_curves/named_groups
         * extension?
         */

        switch (keyExchangeAlgorithm) {
            case KeyExchangeAlgorithm.DH_DSS:
            case KeyExchangeAlgorithm.DH_RSA:
            case KeyExchangeAlgorithm.ECDH_ECDSA:
            case KeyExchangeAlgorithm.ECDH_RSA: {
                // TODO[jsse] Need to have TlsCrypto construct the credentials from the certs/key
                if (crypto instanceof JcaTlsCrypto) {
                    return new JceDefaultTlsCredentialedAgreement((JcaTlsCrypto) crypto, certificate, privateKey);
                } else {
                    // GMSSL SUPPORT: : 2018/8/2 use bc tls crypto
                    AsymmetricKeyParameter parameter = PrivateKeyFactory.createKey(PrivateKeyInfo.getInstance(privateKey.getEncoded()));
                    return new BcDefaultTlsCredentialedAgreement((BcTlsCrypto) crypto, certificate, parameter);
                }
            }

            case KeyExchangeAlgorithm.DHE_DSS:
            case KeyExchangeAlgorithm.DHE_RSA:
            case KeyExchangeAlgorithm.ECDHE_ECDSA:
            case KeyExchangeAlgorithm.ECDHE_RSA:

            case KeyExchangeAlgorithm.RSA: {
                short certificateType = certificate.getCertificateAt(0).getClientCertificateType();
                short signatureAlgorithm = TlsUtils.getSignatureAlgorithmClient(certificateType);
                SignatureAndHashAlgorithm sigAlg = TlsUtils.chooseSignatureAndHashAlgorithm(context,
                        supportedSignatureAlgorithms, signatureAlgorithm);

                // TODO[jsse] Need to have TlsCrypto construct the credentials from the certs/key
                if (crypto instanceof JcaTlsCrypto) {
                    return new JcaDefaultTlsCredentialedSigner(new TlsCryptoParameters(context), (JcaTlsCrypto) crypto,
                            privateKey, certificate, sigAlg);
                } else {
                    // GMSSL SUPPORT: : 2018/8/2 use bc tls crypto
                    AsymmetricKeyParameter parameter = PrivateKeyFactory.createKey(PrivateKeyInfo.getInstance(privateKey.getEncoded()));
                    return new BcDefaultTlsCredentialedSigner(new TlsCryptoParameters(context), (BcTlsCrypto) crypto,
                            parameter, certificate, sigAlg);
                }
            }

            default:
                /* Note: internal error here; selected a key exchange we don't implement! */
                throw new TlsFatalAlert(AlertDescription.internal_error);
        }
    }

}
